package net.kronoz.odyssey.block;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import net.minecraft.class_2248;
import net.minecraft.class_2350;
import net.minecraft.class_259;
import net.minecraft.class_265;
import net.minecraft.class_2960;
import net.minecraft.class_3532;

public final class CollisionShapeHelper {
    private static final Pattern UNR = Pattern.compile("col+ision_\\d+");
    private static final Pattern DIR = Pattern.compile("col+ision_[neswud]\\d+");

    private static final Map<String, class_265> UNR_CACHE = new ConcurrentHashMap<>();
    private static final Map<String, Map<class_2350, class_265>> DIR_CACHE = new ConcurrentHashMap<>();

    private CollisionShapeHelper() {}

    /* =========================
       PUBLIC API (BACK-COMPAT)
       ========================= */

    public static class_265 loadUnrotatedCollisionFromModelJson(String namespace, String path) {
        String key = namespace + ":" + path;
        class_265 c = UNR_CACHE.get(key);
        if (c != null) return c;

        List<class_265> shapes = new ArrayList<>();
        JsonArray elements = readElements(namespace, path);
        if (elements != null) {
            for (JsonElement element : elements) {
                JsonObject obj = element.getAsJsonObject();
                if (!obj.has("name")) continue;
                String name = obj.get("name").getAsString().toLowerCase(Locale.ROOT);
                if (!UNR.matcher(name).matches()) continue;
                class_265 s = readCuboid(obj);
                if (s != null) shapes.add(s);
            }
        }
        class_265 out = shapes.isEmpty() ? class_259.method_1073() : combine(shapes);
        UNR_CACHE.put(key, out.method_1097());
        return out;
    }

    public static class_265 loadUnrotatedCollisionPart(String namespace, String path, String partName) {
        JsonArray elements = readElements(namespace, path);
        if (elements == null) return class_259.method_1073();
        String want = partName.toLowerCase(Locale.ROOT);
        List<class_265> parts = new ArrayList<>();
        for (JsonElement el : elements) {
            JsonObject obj = el.getAsJsonObject();
            if (!obj.has("name")) continue;
            String name = obj.get("name").getAsString().toLowerCase(Locale.ROOT);
            if (!name.equals(want)) continue;
            class_265 s = readCuboid(obj);
            if (s != null) parts.add(s);
        }
        return parts.isEmpty() ? class_259.method_1073() : combine(parts).method_1097();
    }

    /* =========================
       NEW: DIRECTIONAL API
       ========================= */


    public static Map<class_2350, class_265> loadDirectionalCollisionFromModelJson(String namespace, String path) {
        String key = namespace + ":" + path;
        Map<class_2350, class_265> cached = DIR_CACHE.get(key);
        if (cached != null) return cached;

        JsonArray elements = readElements(namespace, path);
        Map<class_2350, List<class_265>> buckets = new EnumMap<>(class_2350.class);
        for (class_2350 d : class_2350.values()) buckets.put(d, new ArrayList<>());

        boolean foundDirectional = false;
        if (elements != null) {
            for (JsonElement element : elements) {
                JsonObject obj = element.getAsJsonObject();
                if (!obj.has("name")) continue;
                String raw = obj.get("name").getAsString().toLowerCase(Locale.ROOT);
                if (!DIR.matcher(raw).matches()) continue;

                char c = raw.charAt(raw.indexOf('_') + 1);
                class_2350 dir = switch (c) {
                    case 'n' -> class_2350.field_11043;
                    case 'e' -> class_2350.field_11034;
                    case 's' -> class_2350.field_11035;
                    case 'w' -> class_2350.field_11039;
                    case 'u' -> class_2350.field_11036;
                    case 'd' -> class_2350.field_11033;
                    default -> null;
                };
                if (dir == null) continue;

                class_265 s = readCuboid(obj);
                if (s != null) {
                    buckets.get(dir).add(s);
                    foundDirectional = true;
                }
            }
        }

        Map<class_2350, class_265> result = new EnumMap<>(class_2350.class);
        if (foundDirectional) {
            for (class_2350 d : class_2350.values()) {
                List<class_265> list = buckets.get(d);
                result.put(d, list.isEmpty() ? class_259.method_1073() : combine(list).method_1097());
            }
        } else {
            class_265 north = loadUnrotatedCollisionFromModelJson(namespace, path);
            result.put(class_2350.field_11043, north);
            result.put(class_2350.field_11034, rotateY(north, 1));
            result.put(class_2350.field_11035, rotateY(north, 2));
            result.put(class_2350.field_11039, rotateY(north, 3));
            result.put(class_2350.field_11036, rotateX(north, 1));
            result.put(class_2350.field_11033, rotateX(north, 3));
        }

        EnumMap<class_2350, class_265> simplified = new EnumMap<>(class_2350.class);
        for (var e : result.entrySet()) simplified.put(e.getKey(), e.getValue().method_1097());
        DIR_CACHE.put(key, simplified);
        return simplified;
    }

    public static class_265 getDirectionalShape(String namespace, String path, class_2350 facing) {
        return loadDirectionalCollisionFromModelJson(namespace, path).getOrDefault(facing, class_259.method_1073());
    }

    /* =========================
       INTERNAL UTILS
       ========================= */

    private static class_265 combine(List<class_265> list) {
        if (list.isEmpty()) return class_259.method_1073();
        class_265 out = list.get(0);
        for (int i = 1; i < list.size(); i++) out = class_259.method_1084(out, list.get(i));
        return out;
    }

    private static JsonArray readElements(String namespace, String path) {
        class_2960 id = class_2960.method_60655(namespace, "models/block/" + path + ".json");
        String resPath = "assets/" + id.method_12836() + "/" + id.method_12832();
        InputStream stream = tryOpen(resPath);
        if (stream == null) return null;
        try (InputStreamReader reader = new InputStreamReader(stream, StandardCharsets.UTF_8)) {
            JsonObject json = JsonParser.parseReader(reader).getAsJsonObject();
            if (!json.has("elements")) return null;
            return json.getAsJsonArray("elements");
        } catch (Exception e) {
            return null;
        }
    }

    private static InputStream tryOpen(String resourcePath) {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        InputStream in = cl != null ? cl.getResourceAsStream(resourcePath) : null;
        if (in != null) return in;
        return CollisionShapeHelper.class.getClassLoader().getResourceAsStream(resourcePath);
    }

    private static class_265 readCuboid(JsonObject element) {
        if (!element.has("from") || !element.has("to")) return null;
        JsonArray from = element.getAsJsonArray("from");
        JsonArray to = element.getAsJsonArray("to");
        if (from.size() != 3 || to.size() != 3) return null;
        double x1 = clamp16(from.get(0).getAsDouble());
        double y1 = clamp16(from.get(1).getAsDouble());
        double z1 = clamp16(from.get(2).getAsDouble());
        double x2 = clamp16(to.get(0).getAsDouble());
        double y2 = clamp16(to.get(1).getAsDouble());
        double z2 = clamp16(to.get(2).getAsDouble());
        return class_2248.method_9541(x1, y1, z1, x2, y2, z2);
    }

    private static double clamp16(double v) {
        return class_3532.method_15350(v, 0.0, 16.0);
    }

    private static class_265 rotateY(class_265 shape, int quarterTurns) {
        class_265[] buffer = new class_265[]{shape, class_259.method_1073()};
        for (int i = 0; i < quarterTurns; i++) {
            buffer[1] = class_259.method_1073();
            buffer[0].method_1089((minX, minY, minZ, maxX, maxY, maxZ) -> {
                double nMinX = 1.0 - maxZ;
                double nMinZ = minX;
                double nMaxX = 1.0 - minZ;
                double nMaxZ = maxX;
                buffer[1] = class_259.method_1084(buffer[1], class_259.method_1081(nMinX, minY, nMinZ, nMaxX, maxY, nMaxZ));
            });
            buffer[0] = buffer[1];
        }
        return buffer[0];
    }

    private static class_265 rotateX(class_265 shape, int quarterTurns) {
        class_265[] buffer = new class_265[]{shape, class_259.method_1073()};
        for (int i = 0; i < quarterTurns; i++) {
            buffer[1] = class_259.method_1073();
            buffer[0].method_1089((minX, minY, minZ, maxX, maxY, maxZ) -> {
                double nMinY = 1.0 - maxZ;
                double nMinZ = minY;
                double nMaxY = 1.0 - minZ;
                double nMaxZ = maxY;
                buffer[1] = class_259.method_1084(buffer[1], class_259.method_1081(minX, nMinY, nMinZ, maxX, nMaxY, nMaxZ));
            });
            buffer[0] = buffer[1];
        }
        return buffer[0];
    }
}
