package net.kronoz.odyssey.systems.physics.jetpack;

import org.joml.Matrix4f;
import org.joml.Vector3f;

import java.util.Random;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_265;
import net.minecraft.class_2680;
import net.minecraft.class_3532;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4608;

public final class OdysseySmokeField {

    private static final double MAX_WORLD_RADIUS = 80.0;
    private static final double DRAG = 0.90;
    private static final double LATERAL_VISC = 0.94;
    private static final double BUOYANCY_BASE = 0.020;
    private static final double BUOYANCY_TEMP = 0.040;
    private static final double GRAVITY_RESIDUAL = 0.004;
    private static final double MIN_UP_SPEED = 0.008;

    private static final double CURL_EPS = 0.22;
    private static final double CURL_FORCE = 0.45;
    private static final double WIND_MAG = 0.010;

    private static final double SAMPLE_STEP = 0.10;
    private static final double WALL_PUSH = 0.65;

    private static final double LIFE_MIN = 2.5;
    private static final double LIFE_MAX = 5.0;
    private static final double SIZE_MIN = 0.02;
    private static final double SIZE_MAX = 0.05;

    private static final double CONE_ANGLE_RAD = Math.toRadians(16.0);

    private static final double AXIS_SPIRAL = 0.55;
    private static final double AXIS_PULL = 0.35;
    private static final double EARLY_SPRING = 16.0;
    private static final double EARLY_SPRING_TIME = 0.12;
    private static final double MAX_SPEED = 0.35;

    private double emitRate = 30.0;
    private double emitAccumulator = 0.0;
    private int aliveCount = 0;
    private int nextDeadIdx = 0;

    private final Node[] nodes;
    private final Random rng = new Random(0x9E3779B97F4A7C15L);

    private float endR=0.729f, endG=0.729f, endB=0.729f;
    private float midR=1.00f, midG=0.60f, midB=0.0f;
    private float startR=0.322f, startG=0.0f, startB=1.0f;

    private class_243 thrustAxis = new class_243(0, 1, 0);
    private boolean emitting = true;

    public OdysseySmokeField(int capacity) {
        nodes = new Node[capacity];
        for (int i=0;i<capacity;i++) nodes[i] = Node.dead();
    }

    public OdysseySmokeField(int capacity, int r, int g, int b) {
        this(capacity);
        this.endR = class_3532.method_15363(r/255f,0,1);
        this.endG = class_3532.method_15363(g/255f,0,1);
        this.endB = class_3532.method_15363(b/255f,0,1);
    }

    public int aliveCount() { return aliveCount; }
    public OdysseySmokeField setEmitting(boolean e) { this.emitting = e; return this; }
    public OdysseySmokeField setAxis(class_243 axisWS) {
        if (axisWS != null && axisWS.method_1027() > 1e-9) this.thrustAxis = axisWS.method_1029();
        return this;
    }
    public OdysseySmokeField setJetAxis(class_243 axisWS) { return setAxis(axisWS); }
    public double getAxisSpiral() { return AXIS_SPIRAL; }
    public double getAxisPull()   { return AXIS_PULL; }
    public boolean isEmpty() { return aliveCount == 0; }

    private static final class Node {
        boolean alive;
        class_243 p, v;
        double temp, age, life, baseSize;
        float yaw, pitch, roll, yawVel, pitchVel, rollVel;
        static Node dead(){
            Node n = new Node();
            n.alive=false; n.p=class_243.field_1353; n.v=class_243.field_1353;
            n.temp=0; n.age=0; n.life=0; n.baseSize=0;
            return n;
        }
    }

    public void update(double dt, class_243 originWS, class_1937 world, long worldTime) {
        if (dt <= 0.0) return;

        if (emitting && aliveCount < nodes.length) {
            emitAccumulator += emitRate * dt;
            while (emitAccumulator >= 1.0 && aliveCount < nodes.length) {
                int start = nextDeadIdx;
                do {
                    Node n = nodes[nextDeadIdx];
                    if (!n.alive) {
                        spawnInto(n, originWS);
                        aliveCount++;
                        emitAccumulator -= 1.0;
                        nextDeadIdx = (nextDeadIdx + 1) % nodes.length;
                        break;
                    }
                    nextDeadIdx = (nextDeadIdx + 1) % nodes.length;
                } while (nextDeadIdx != start);
                if (start == nextDeadIdx) break;
            }
        }

        stepAll(dt, originWS, world, worldTime);
    }

    private void spawnInto(Node n, class_243 origin) {
        class_243 dir = sampleConeDir(thrustAxis, CONE_ANGLE_RAD);
        n.p = origin;
        n.v = dir.method_1021(0.18 + rng.nextDouble()*0.08).method_1019(jitter3(0.01));
        n.life = LIFE_MIN + rng.nextDouble()*(LIFE_MAX - LIFE_MIN);
        n.baseSize = SIZE_MIN + rng.nextDouble()*(SIZE_MAX - SIZE_MIN);
        n.temp = 1.0;
        n.age = 0.0;
        n.yaw   = (float)(rng.nextDouble() * Math.PI * 2.0);
        n.pitch = (float)((rng.nextDouble()-0.5) * Math.toRadians(40));
        n.roll  = (float)((rng.nextDouble()-0.5) * Math.toRadians(40));
        n.yawVel   = (float)((rng.nextDouble()-0.5) * 1.5);
        n.pitchVel = (float)((rng.nextDouble()-0.5) * 1.2);
        n.rollVel  = (float)((rng.nextDouble()-0.5) * 1.2);
        n.alive = true;
    }

    private void stepAll(double dt, class_243 origin, class_1937 world, long worldTime) {
        for (int i = 0; i < nodes.length; i++) {
            Node n = nodes[i];
            if (!n.alive) continue;

            n.age += dt;

            if (n.age >= n.life || n.p.method_1025(origin) > (MAX_WORLD_RADIUS*MAX_WORLD_RADIUS)) {
                n.alive = false; aliveCount--; continue;
            }
            if (isInsideSolid(n.p, world)) { n.alive = false; aliveCount--; continue; }

            double t = (worldTime * 0.05) + i * 7.31;

            class_243 curl = curl3(n.p.field_1352*0.55 + t*0.08, n.p.field_1351*0.50 + t*0.08, n.p.field_1350*0.55 + t*0.08);
            class_243 wind = windAt(n.p, worldTime, WIND_MAG);
            double buoy = BUOYANCY_BASE + Math.max(0.0, n.temp) * BUOYANCY_TEMP;

            class_243 a = new class_243(
                    curl.field_1352*CURL_FORCE + wind.field_1352,
                    curl.field_1351*CURL_FORCE + buoy - GRAVITY_RESIDUAL + wind.field_1351*0.25,
                    curl.field_1350*CURL_FORCE + wind.field_1350
            );

            a = a.method_1019(axisField(n, origin, dt));

            n.v = n.v.method_1019(a.method_1021(dt));

            double spd = n.v.method_1033();
            if (spd > MAX_SPEED) n.v = n.v.method_1021(MAX_SPEED / Math.max(spd, 1e-6));

            n.v = new class_243(n.v.field_1352*LATERAL_VISC, n.v.field_1351*DRAG, n.v.field_1350*LATERAL_VISC);
            if (n.v.field_1351 < MIN_UP_SPEED) n.v = new class_243(n.v.field_1352, MIN_UP_SPEED, n.v.field_1350);

            class_243 next = n.p.method_1019(n.v.method_1021(dt));
            class_243 after = collideWorld(n.p, next, world);
            if (isInsideSolid(after, world)) { n.alive = false; aliveCount--; continue; }
            n.p = after;

            n.temp *= 0.996;
            n.yaw   += n.yawVel   * dt;
            n.pitch += n.pitchVel * dt;
            n.roll  += n.rollVel  * dt;
        }
    }

    private class_243 axisField(Node n, class_243 origin, double dt) {
        class_243 a = thrustAxis;
        class_243 rel = n.p.method_1020(origin);
        double along = rel.method_1026(a);
        class_243 proj = a.method_1021(along);
        class_243 radial = rel.method_1020(proj);

        double k = class_3532.method_15350(radial.method_1033(), 0.0, 1.0);
        class_243 pull = radial.method_1027() > 1e-9 ? radial.method_1029().method_1021(-AXIS_PULL * k) : class_243.field_1353;

        class_243 basisU = Math.abs(a.field_1352) < 0.9 ? new class_243(1,0,0) : new class_243(0,1,0);
        class_243 u = a.method_1036(basisU).method_1029();
        class_243 v = a.method_1036(u).method_1029();
        double spin = AXIS_SPIRAL * (0.5 + 0.5 * Math.sin(along * 10.0));
        class_243 swirl = u.method_1021(-spin).method_1019(v.method_1021(spin));

        if (n.age < EARLY_SPRING_TIME) {
            double w = 1.0 - (n.age / EARLY_SPRING_TIME);
            class_243 kick = a.method_1021(EARLY_SPRING * w);
            return pull.method_1019(swirl).method_1019(kick);
        }
        return pull.method_1019(swirl);
    }

    private boolean isInsideSolid(class_243 p, class_1937 world) {
        class_2338 bp = class_2338.method_49638(p);
        class_2680 st = world.method_8320(bp);
        if (st.method_26215()) return false;
        class_265 sh = st.method_26220(world, bp);
        if (sh.method_1110()) return false;
        var aabb = sh.method_1107();
        double lx = p.field_1352 - bp.method_10263(), ly = p.field_1351 - bp.method_10264(), lz = p.field_1350 - bp.method_10260();
        return aabb != null && aabb.method_1008(lx, ly, lz);
    }

    private class_243 collideWorld(class_243 cur, class_243 next, class_1937 world) {
        class_243 delta = next.method_1020(cur);
        int steps = Math.max(1, (int)Math.ceil(delta.method_1033() / SAMPLE_STEP));
        class_243 p = cur;

        for (int i = 1; i <= steps; i++) {
            double t = (double)i / (double)steps;
            class_243 probe = cur.method_35590(next, t);

            class_2338 bp = class_2338.method_49638(probe);
            class_2680 st = world.method_8320(bp);
            if (!st.method_26215()) {
                class_265 sh = st.method_26220(world, bp);
                if (!sh.method_1110()) {
                    class_243 back = cur.method_35590(next, Math.max(0, t - 0.10));
                    class_2350 face = nearestFace(probe, bp);
                    class_243 nrm = new class_243(face.method_23955().x(), face.method_23955().y(), face.method_23955().z());
                    class_243 slide = projectOntoPlane(next.method_1020(back), nrm).method_1021(0.72);
                    p = back.method_1019(slide).method_1019(nrm.method_1021(WALL_PUSH * 0.012));
                    return p;
                }
            }
            p = probe;
        }
        return p;
    }

    private static class_243 projectOntoPlane(class_243 v, class_243 n) {
        double dot = v.field_1352*n.field_1352 + v.field_1351*n.field_1351 + v.field_1350*n.field_1350;
        return new class_243(v.field_1352 - dot*n.field_1352, v.field_1351 - dot*n.field_1351, v.field_1350 - dot*n.field_1350);
    }

    private static class_2350 nearestFace(class_243 p, class_2338 bp) {
        double cx = bp.method_10263() + 0.5, cy = bp.method_10264() + 0.5, cz = bp.method_10260() + 0.5;
        double dx = p.field_1352 - cx, dy = p.field_1351 - cy, dz = p.field_1350 - cz;
        double ax = Math.abs(dx), ay = Math.abs(dy), az = Math.abs(dz);
        if (ay >= ax && ay >= az) return dy > 0 ? class_2350.field_11036 : class_2350.field_11033;
        if (ax >= az) return dx > 0 ? class_2350.field_11034 : class_2350.field_11039;
        return dz > 0 ? class_2350.field_11035 : class_2350.field_11043;
    }

    private class_243 windAt(class_243 p, long time, double mag) {
        double tt = time * 0.02;
        double nx = fbm(p.field_1352*0.08 + tt*0.05, p.field_1351*0.05, p.field_1350*0.08) - 0.5;
        double nz = fbm(p.field_1352*0.08, p.field_1351*0.05 + tt*0.05, p.field_1350*0.08) - 0.5;
        return new class_243(nx * mag, 0.0, nz * mag);
    }

    private double fbm(double x, double y, double z) {
        double s = 0.0, a = 0.5;
        for (int i = 0; i < 3; i++) {
            s += noise3(x, y, z) * a;
            x *= 2.0; y *= 2.0; z *= 2.0;
            a *= 0.5;
        }
        return s;
    }

    private class_243 curl3(double x, double y, double z) {
        double ex = CURL_EPS, ey = CURL_EPS, ez = CURL_EPS;
        double Ny_z1 = noise3(x, y, z + ez), Ny_z0 = noise3(x, y, z - ez);
        double Nz_y1 = noise3(x, y + ey, z), Nz_y0 = noise3(x, y - ey, z);
        double Nz_x1 = noise3(x + ex, y, z), Nz_x0 = noise3(x - ex, y, z);
        double Nx_z1 = noise3(x, y, z + ez), Nx_z0 = noise3(x, y, z - ez);
        double Nx_y1 = noise3(x, y + ey, z), Nx_y0 = noise3(x, y - ey, z);
        double Ny_x1 = noise3(x + ex, y, z), Ny_x0 = noise3(x - ex, y, z);
        double cx = (Nz_y1 - Nz_y0) - (Ny_z1 - Ny_z0);
        double cy = (Nx_z1 - Nx_z0) - (Nz_x1 - Nz_x0);
        double cz = (Ny_x1 - Ny_x0) - (Nx_y1 - Nx_y0);
        double len = Math.sqrt(cx*cx + cy*cy + cz*cz) + 1e-8;
        return new class_243(cx/len, cy/len, cz/len);
    }

    private double noise3(double x, double y, double z) {
        int xi = fastFloor(x), yi = fastFloor(y), zi = fastFloor(z);
        double xf = x - xi, yf = y - yi, zf = z - zi;
        double u = smooth(xf), v = smooth(yf), w = smooth(zf);
        double n000 = hash(xi,   yi,   zi);
        double n100 = hash(xi+1, yi,   zi);
        double n010 = hash(xi,   yi+1, zi);
        double n110 = hash(xi+1, yi+1, zi);
        double n001 = hash(xi,   yi,   zi+1);
        double n101 = hash(xi+1, yi,   zi+1);
        double n011 = hash(xi,   yi+1, zi+1);
        double n111 = hash(xi+1, yi+1, zi+1);
        double x00 = lerp(u, n000, n100);
        double x10 = lerp(u, n010, n110);
        double x01 = lerp(u, n001, n101);
        double x11 = lerp(u, n011, n111);
        double y0 = lerp(v, x00, x10);
        double y1 = lerp(v, x01, x11);
        return lerp(w, y0, y1) * 2.0 - 1.0;
    }

    private static int fastFloor(double x) { int i = (int)x; return x < i ? i-1 : i; }
    private static double smooth(double t){ return t*t*(3 - 2*t); }
    private static double lerp(double t, double a, double b){ return a + t*(b-a); }
    private static double hash(int x, int y, int z) {
        long h = (long)x * 374761393L + (long)y * 668265263L + (long)z * 700001L;
        h = (h ^ (h >> 13)) * 1274126177L;
        h ^= (h >> 16);
        return (h & 0xFFFFFFFFL) / 4294967296.0;
    }

    private class_243 sampleConeDir(class_243 axis, double angRad) {
        class_243 a = axis.method_1029();
        class_243 tmp = Math.abs(a.field_1352) < 0.9 ? new class_243(1,0,0) : new class_243(0,1,0);
        class_243 u = a.method_1036(tmp).method_1029();
        class_243 v = a.method_1036(u).method_1029();

        double u1 = rng.nextDouble();
        double u2 = rng.nextDouble();
        double theta = 2.0 * Math.PI * u1;
        double cosAng = Math.cos(angRad);
        double z = cosAng + (1 - cosAng) * u2;
        double s = Math.sqrt(Math.max(0.0, 1 - z*z));
        double x = s * Math.cos(theta);
        double y = s * Math.sin(theta);

        return u.method_1021(x).method_1019(v.method_1021(y)).method_1019(a.method_1021(z)).method_1029();
    }

    private class_243 jitter3(double s) {
        return new class_243((Math.random()*2-1)*s, (Math.random()*2-1)*s, (Math.random()*2-1)*s);
    }

    public void render(class_4587 ms, class_4588 vc, class_243 cameraBase, float tickDelta) {
        Matrix4f m4 = ms.method_23760().method_23761();
        final int FULL_BRIGHT = 0x00F000F0;
        final int OVERLAY = class_4608.field_21444;

        for (Node n : nodes) {
            if (!n.alive) continue;

            double lifeT = n.age / n.life;

            float cR, cG, cB;
            if (lifeT < 0.45) {
                double t = lifeT / 0.45;
                cR = (float)lerp(t, midR, startR);
                cG = (float)lerp(t, midG, startG);
                cB = (float)lerp(t, midB, startB);
            } else {
                double t = (lifeT - 0.45) / 0.55;
                cR = (float)lerp(t, startR, endR);
                cG = (float)lerp(t, startG, endG);
                cB = (float)lerp(t, startB, endB);
            }

            float rise = (float)class_3532.method_15350(lifeT * 1.2, 0.0, 1.0);
            float fall = (float)Math.pow(1.0 - class_3532.method_15350(lifeT, 0.0, 1.0), 1.15);
            float alpha = class_3532.method_15363(0.08f + 0.85f*rise*fall, 0f, 0.9f);

            float edge = (float)(n.baseSize * (0.9 + lifeT*2.0));
            float hs = edge * 0.5f;

            float cx = (float)(n.p.field_1352 - cameraBase.field_1352);
            float cy = (float)(n.p.field_1351 - cameraBase.field_1351);
            float cz = (float)(n.p.field_1350 - cameraBase.field_1350);

            float cyw = (float)Math.cos(n.yaw), syw = (float)Math.sin(n.yaw);
            float cpi = (float)Math.cos(n.pitch), spi = (float)Math.sin(n.pitch);
            float cro = (float)Math.cos(n.roll),  sro = (float)Math.sin(n.roll);

            Vector3f X = new Vector3f(1,0,0);
            Vector3f Y = new Vector3f(0,1,0);
            Vector3f Z = new Vector3f(0,0,1);

            X = new Vector3f(cyw*X.x + 0*X.y + -syw*X.z, X.y, syw*X.x + 0*X.y + cyw*X.z);
            Y = new Vector3f(cyw*Y.x + 0*Y.y + -syw*Y.z, Y.y, syw*Y.x + 0*Y.y + cyw*Y.z);
            Z = new Vector3f(cyw*Z.x + 0*Z.y + -syw*Z.z, Z.y, syw*Z.x + 0*Z.y + cyw*Z.z);

            X = new Vector3f(X.x,  cpi*X.y - spi*X.z,  spi*X.y + cpi*X.z);
            Y = new Vector3f(Y.x,  cpi*Y.y - spi*Y.z,  spi*Y.y + cpi*Y.z);
            Z = new Vector3f(Z.x,  cpi*Z.y - spi*Z.z,  spi*Z.y + cpi*Z.z);

            X = new Vector3f( cro*X.x - sro*X.y, sro*X.x + cro*X.y, X.z);
            Y = new Vector3f( cro*Y.x - sro*Y.y, sro*Y.x + cro*Y.y, Y.z);
            Z = new Vector3f( cro*Z.x - sro*Z.y, sro*Z.x + cro*Z.y, Z.z);

            Vector3f RX = new Vector3f(X).mul(hs);
            Vector3f RY = new Vector3f(Y).mul(hs);
            Vector3f RZ = new Vector3f(Z).mul(hs);

            Vector3f C = new Vector3f(cx,cy,cz);
            Vector3f p000 = new Vector3f(C).sub(RX).sub(RY).sub(RZ);
            Vector3f p100 = new Vector3f(C).add(RX).sub(RY).sub(RZ);
            Vector3f p110 = new Vector3f(C).add(RX).add(RY).sub(RZ);
            Vector3f p010 = new Vector3f(C).sub(RX).add(RY).sub(RZ);

            Vector3f p001 = new Vector3f(C).sub(RX).sub(RY).add(RZ);
            Vector3f p101 = new Vector3f(C).add(RX).sub(RY).add(RZ);
            Vector3f p111 = new Vector3f(C).add(RX).add(RY).add(RZ);
            Vector3f p011 = new Vector3f(C).sub(RX).add(RY).add(RZ);

            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p100, p101, p111, p110, X);
            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p000, p010, p011, p001, new Vector3f(X).negate());
            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p010, p110, p111, p011, Y);
            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p000, p001, p101, p100, new Vector3f(Y).negate());
            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p101, p001, p011, p111, Z);
            putQuad(m4, ms, vc, cR,cG,cB,alpha, 0,1,1,0, p100, p110, p010, p000, new Vector3f(Z).negate());
        }
    }

    private void putQuad(Matrix4f m4, class_4587 ms, class_4588 vc,
                         float r, float g, float b, float a,
                         float u0, float v1, float u1, float v0,
                         Vector3f P0, Vector3f P1, Vector3f P2, Vector3f P3, Vector3f N) {
        final int packedLight = 0x00F000F0;
        final int packedOverlay = class_4608.field_21444;
        int nx = (int)Math.signum(N.x()); int ny = (int)Math.signum(N.y()); int nz = (int)Math.signum(N.z());
        vc.method_22918(m4, P0.x, P0.y, P0.z).method_22915(r,g,b,a).method_22913(u0, v1).method_22922(packedOverlay).method_60803(packedLight).method_60831(ms.method_23760(), nx, ny, nz);
        vc.method_22918(m4, P1.x, P1.y, P1.z).method_22915(r,g,b,a).method_22913(u1, v1).method_22922(packedOverlay).method_60803(packedLight).method_60831(ms.method_23760(), nx, ny, nz);
        vc.method_22918(m4, P2.x, P2.y, P2.z).method_22915(r,g,b,a).method_22913(u1, v0).method_22922(packedOverlay).method_60803(packedLight).method_60831(ms.method_23760(), nx, ny, nz);
        vc.method_22918(m4, P3.x, P3.y, P3.z).method_22915(r,g,b,a).method_22913(u0, v0).method_22922(packedOverlay).method_60803(packedLight).method_60831(ms.method_23760(), nx, ny, nz);
    }
}
