package net.kronoz.odyssey.world;

import com.mojang.logging.LogUtils;
import net.minecraft.class_18;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2382;
import net.minecraft.class_2415;
import net.minecraft.class_2470;
import net.minecraft.class_2487;
import net.minecraft.class_26;
import net.minecraft.class_2741;
import net.minecraft.class_2806;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3485;
import net.minecraft.class_3491;
import net.minecraft.class_3492;
import net.minecraft.class_3499;
import net.minecraft.class_3545;
import net.minecraft.class_3828;
import net.minecraft.class_4538;
import net.minecraft.class_5321;
import net.minecraft.class_5819;
import net.minecraft.class_7225;
import net.minecraft.class_7924;
import net.minecraft.server.MinecraftServer;
import org.slf4j.Logger;

import java.util.*;

public final class FixedStructurePlacer {
    private static final Logger LOGGER = LogUtils.getLogger();

    private static final class_2960[] BASES = new class_2960[] {
            class_2960.method_60655("odyssey", "facility")
    };

    private static final class_2338 ORIGIN = new class_2338(0, 12, 0);
    private static final class_5321<class_1937> VOID_DIM =
            class_5321.method_29179(class_7924.field_41223, class_2960.method_60655("odyssey", "void"));

    private static final ArrayDeque<Runnable> JOBS = new ArrayDeque<>();
    private static final int BLOCKS_PER_TICK = 40000;

    private enum Phase { IDLE, CARVING, PLACING, DONE }
    private static Phase PHASE = Phase.IDLE;
    private static int CARVES_LEFT = 0;
    private static int PLACES_LEFT = 0;

    private static List<class_3545<class_3499, class_2338>> TILES = List.of();

    private FixedStructurePlacer() {}

    public static void onWorldLoaded(class_3218 world) {
        if (!world.method_27983().equals(VOID_DIM)) return;

        class_26 psm = world.method_17983();
        StructuresPlacedState state = psm.method_17924(StructuresPlacedState.TYPE, StructuresPlacedState.KEY);
        if (state.alreadyPlaced) {
            LOGGER.info("[Odyssey] already placed");
            return;
        }

        enqueue(world);
    }

    public static void tick(MinecraftServer server) {
        int steps = 64;
        while (steps-- > 0) {
            Runnable r = JOBS.poll();
            if (r == null) break;
            r.run();
        }
    }

    private static void enqueue(class_3218 world) {
        if (PHASE != Phase.IDLE) return;

        class_3485 stm = world.method_14183();
        List<class_3545<class_3499, class_2338>> tiles = findGrid3D(stm, BASES, ORIGIN);
        if (tiles.isEmpty()) {
            LOGGER.error("[Odyssey] No facility grid tiles found (expected facilityX_Y_Z.nbt or facility_X_Y_Z.nbt)");
            return;
        }
        assertUniformTileSize(tiles);
        forceChunks(world, tiles);

        TILES = tiles;

        CARVES_LEFT = tiles.size();
        PHASE = Phase.CARVING;

        for (var p : tiles) {
            JOBS.add(new CarveTask(world, p.method_15441(), p.method_15442().method_15160(), 1));
        }

        LOGGER.info("[Odyssey] queued {} carve jobs", tiles.size());
    }

    private static void enqueuePlacements(class_3218 world) {
        if (PHASE != Phase.CARVING) return;
        PHASE = Phase.PLACING;
        PLACES_LEFT = TILES.size();

        for (var p : TILES) {
            JOBS.add(() -> {
                place(world, p.method_15442(), p.method_15441());
                if (--PLACES_LEFT == 0) {
                    PHASE = Phase.DONE;
                    markPlaced(world);
                    LOGGER.info("[Odyssey] placement phase done");
                }
            });
        }
        LOGGER.info("[Odyssey] queued {} placement jobs", TILES.size());
    }

    private static List<class_3545<class_3499, class_2338>> findGrid3D(class_3485 stm, class_2960[] bases, class_2338 origin) {
        for (class_2960 base : bases) {
            var grid = collectGrid3D(stm, base, origin, false); // facility0_0_0
            if (!grid.isEmpty()) {
                LOGGER.info("[Odyssey] Using 3D grid w/o underscore for {}", base);
                return grid;
            }
            grid = collectGrid3D(stm, base, origin, true); // facility_0_0_0
            if (!grid.isEmpty()) {
                LOGGER.info("[Odyssey] Using 3D grid with underscore for {}", base);
                return grid;
            }
        }
        return List.of();
    }

    private static class_3499 getTemplate(class_3485 stm, class_2960 id) {
        class_3499 t = stm.method_15094(id).orElse(null);
        if (t != null) return t;
        class_2960 s1 = class_2960.method_60655(id.method_12836(), "structure/" + id.method_12832());
        t = stm.method_15094(s1).orElse(null);
        if (t != null) return t;
        class_2960 s2 = class_2960.method_60655(id.method_12836(), "structures/" + id.method_12832());
        return stm.method_15094(s2).orElse(null);
    }

    private static List<class_3545<class_3499, class_2338>> collectGrid3D(class_3485 stm, class_2960 base, class_2338 origin, boolean underscore) {
        List<class_3545<class_3499, class_2338>> out = new ArrayList<>();
        class_2382 tile = null;
        final int MAX = 128;
        for (int gx = 0; gx < MAX; gx++) {
            boolean anyX = false;
            for (int gy = 0; gy < MAX; gy++) {
                boolean anyY = false;
                for (int gz = 0; gz < MAX; gz++) {
                    String name = underscore
                            ? base.method_12832() + "_" + gx + "_" + gy + "_" + gz
                            : base.method_12832() + gx + "_" + gy + "_" + gz;
                    class_2960 id = class_2960.method_60655(base.method_12836(), name);
                    class_3499 t = getTemplate(stm, id);
                    if (t == null) { if (gz == 0) break; else continue; }
                    if (tile == null) tile = t.method_15160();
                    class_2338 pos = origin.method_10069(tile.method_10263()*gx, tile.method_10264()*gy, tile.method_10260()*gz);
                    out.add(new class_3545<>(t, pos));
                    anyY = true;
                }
                if (anyY) anyX = true; else break;
            }
            if (!anyX) break;
        }
        return out;
    }

    private static void assertUniformTileSize(List<class_3545<class_3499, class_2338>> tiles) {
        if (tiles.isEmpty()) return;
        class_2382 s0 = tiles.get(0).method_15442().method_15160();
        for (var p : tiles) {
            class_2382 s = p.method_15442().method_15160();
            if (!s.equals(s0)) {
                throw new IllegalStateException("[Odyssey] Tile size mismatch: expected " + s0 + " but found " + s +
                        " at " + p.method_15441());
            }
        }
    }

    private static void place(class_3218 world, class_3499 template, class_2338 origin) {
        class_3492 data = new class_3492()
                .method_15123(class_2470.field_11467)
                .method_15125(class_2415.field_11302)
                .method_15133(false)
                .method_16184(WaterlogSanitizerProcessor.INSTANCE)
                .method_16184(RemoveStructureBlocksProcessor.INSTANCE); // <- remove STRUCTURE_BLOCKs

        class_5819 rng = world.method_8409();
        boolean ok = template.method_15172(world, origin, origin, data, rng, 2);
        LOGGER.info("[Odyssey] placed {} at {} -> {}", template, origin, ok);
    }

    private static void markPlaced(class_3218 world) {
        class_26 psm = world.method_17983();
        StructuresPlacedState state = psm.method_17924(StructuresPlacedState.TYPE, StructuresPlacedState.KEY);
        state.alreadyPlaced = true;
        state.method_80();
        psm.method_125();
        LOGGER.info("[Odyssey] marked as placed");
    }

    private static void forceChunks(class_3218 world, List<class_3545<class_3499, class_2338>> tiles) {
        for (var p : tiles) {
            class_2382 s = p.method_15442().method_15160();
            class_2338 o = p.method_15441();
            int minX = o.method_10263() - 1, minZ = o.method_10260() - 1, maxX = o.method_10263() + s.method_10263(), maxZ = o.method_10260() + s.method_10260();
            for (int cx = (minX >> 4); cx <= (maxX >> 4); cx++) {
                for (int cz = (minZ >> 4); cz <= (maxZ >> 4); cz++) {
                    world.method_14178().method_12121(cx, cz, class_2806.field_12803, true);
                }
            }
        }
    }

    private static final class CarveTask implements Runnable {
        private final class_3218 world;
        private final int minX, minY, minZ, maxX, maxY, maxZ;
        private int x, y, z;
        private boolean started;

        CarveTask(class_3218 world, class_2338 origin, class_2382 size, int margin) {
            this.world = world;
            this.minX = origin.method_10263() - margin;
            this.minY = origin.method_10264() - margin;
            this.minZ = origin.method_10260() - margin;
            this.maxX = origin.method_10263() + size.method_10263() + margin - 1;
            this.maxY = origin.method_10264() + size.method_10264() + margin - 1;
            this.maxZ = origin.method_10260() + size.method_10260() + margin - 1;
            this.x = minX; this.y = minY; this.z = minZ;
        }

        @Override public void run() {
            if (!started) { started = true; }
            int left = BLOCKS_PER_TICK;
            while (left-- > 0 && y <= maxY) {
                class_2338 p = new class_2338(x, y, z);
                if (!world.method_22347(p)) world.method_8652(p, class_2246.field_10124.method_9564(), 2);
                x++; if (x > maxX) { x = minX; z++; if (z > maxZ) { z = minZ; y++; } }
            }
            if (y <= maxY) {
                JOBS.add(this);
            } else {
                if (--CARVES_LEFT == 0) {
                    enqueuePlacements(world);
                }
            }
        }
    }

    public static final class StructuresPlacedState extends class_18 {
        public static final String KEY = "odyssey_fixed_structures";
        public static final class_18.class_8645<StructuresPlacedState> TYPE =
                new class_18.class_8645<>(StructuresPlacedState::new, StructuresPlacedState::fromNbt, null);
        public boolean alreadyPlaced = false;
        public StructuresPlacedState() {}
        public static StructuresPlacedState fromNbt(class_2487 nbt, class_7225.class_7874 lookup) {
            StructuresPlacedState s = new StructuresPlacedState();
            s.alreadyPlaced = nbt.method_10577("alreadyPlaced");
            return s;
        }
        @Override public class_2487 method_75(class_2487 nbt, class_7225.class_7874 lookup) {
            nbt.method_10556("alreadyPlaced", alreadyPlaced);
            return nbt;
        }
    }

    private static final class WaterlogSanitizerProcessor extends class_3491 {
        static final WaterlogSanitizerProcessor INSTANCE = new WaterlogSanitizerProcessor();
        private WaterlogSanitizerProcessor() {}
        @Override
        public class_3499.class_3501 method_15110(
                class_4538 world, class_2338 pos, class_2338 pivot,
                class_3499.class_3501 original,
                class_3499.class_3501 current,
                class_3492 data) {
            var st = current.comp_1342();
            if (st.method_28501().contains(class_2741.field_12508)) {
                st = st.method_11657(class_2741.field_12508, false);
                return new class_3499.class_3501(current.comp_1341(), st, current.comp_1343());
            }
            return current;
        }
        @Override protected class_3828<?> method_16772() { return class_3828.field_16987; }
    }

    private static final class RemoveStructureBlocksProcessor extends class_3491 {
        static final RemoveStructureBlocksProcessor INSTANCE = new RemoveStructureBlocksProcessor();
        private RemoveStructureBlocksProcessor() {}
        @Override
        public class_3499.class_3501 method_15110(
                class_4538 world, class_2338 pos, class_2338 pivot,
                class_3499.class_3501 original,
                class_3499.class_3501 current,
                class_3492 data) {
            if (current.comp_1342().method_27852(class_2246.field_10465)) {
                return null;
            }
            return current;
        }
        @Override protected class_3828<?> method_16772() { return class_3828.field_16987; }
    }
}
