package com.boyonk.sharedadvancements;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.internal.Streams;
import com.google.gson.stream.JsonReader;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_4239;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;

public class SharedAdvancementsConfig {

	public static final Codec<SharedAdvancementsConfig> CODEC = RecordCodecBuilder.create(instance ->
			instance.group(
					Codec.BOOL
							.fieldOf("merge_broadcast_message")
							.forGetter(SharedAdvancementsConfig::mergeBroadcastMessage),
					Codec.BOOL
							.fieldOf("share_displayless")
							.forGetter(SharedAdvancementsConfig::shareDisplayless),
					Codec.BOOL
							.fieldOf("creative_disabled")
							.forGetter(SharedAdvancementsConfig::creativeDisabled),
					Codec.BOOL
							.fieldOf("global_group")
							.forGetter(SharedAdvancementsConfig::globalGroup)
			).apply(instance, SharedAdvancementsConfig::new)
	);

	private static final Logger LOGGER = SharedAdvancements.LOGGER;
	private static final Gson GSON = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().create();

	private final boolean mergeBroadcastMessage;
	private final boolean shareDisplayless;
	private final boolean creativeDisabled;
	private final boolean globalGroup;

	private SharedAdvancementsConfig(boolean mergeBroadcastMessage, boolean shareDisplayless, boolean creativeDisabled, boolean globalGroup) {
		this.mergeBroadcastMessage = mergeBroadcastMessage;
		this.shareDisplayless = shareDisplayless;
		this.creativeDisabled = creativeDisabled;
		this.globalGroup = globalGroup;
	}

	private static SharedAdvancementsConfig createDefault() {
		return new SharedAdvancementsConfig(true, false, true, true);
	}

	public static SharedAdvancementsConfig load() {
		Path path = FabricLoader.getInstance().getConfigDir().resolve(SharedAdvancements.NAMESPACE + ".json");

		if (Files.isRegularFile(path)) {
			try (JsonReader reader = new JsonReader(Files.newBufferedReader(path, StandardCharsets.UTF_8))) {
				reader.setLenient(false);
				JsonElement json = Streams.parse(reader);
				return CODEC.parse(JsonOps.INSTANCE, json).getOrThrow(JsonParseException::new);
			} catch (JsonParseException exception) {
				LOGGER.error("Couldn't parse config {}", path.getFileName(), exception);
			} catch (IOException exception) {
				LOGGER.error("Couldn't access config {}", path.getFileName(), exception);
			}
			return createDefault();
		} else {
			SharedAdvancementsConfig config = createDefault();
			try {
				class_4239.method_47525(path.getParent());
				try (Writer writer = Files.newBufferedWriter(path, StandardCharsets.UTF_8)) {
					GSON.toJson(CODEC.encodeStart(JsonOps.INSTANCE, config).getOrThrow(IOException::new), writer);
				}
			} catch (IOException exception) {
				LOGGER.error("Failed to save config {}", path, exception);
			}
			return config;
		}
	}

	public boolean mergeBroadcastMessage() {
		return mergeBroadcastMessage;
	}

	public boolean shareDisplayless() {
		return shareDisplayless;
	}

	public boolean creativeDisabled() {
		return this.creativeDisabled;
	}

	public boolean globalGroup() {
		return this.globalGroup;
	}

}
