#if !defined(__AMBIENT__)
#define __AMBIENT__

#include "../program/config.glsl"
#include "coord.glsl"

uniform float rainStrength;
uniform float wetness;

uniform float blindness;
uniform float darknessFactor;
uniform float darknessLightFactor;
uniform float nightVision;

uniform vec3 skyColor;
uniform vec3 fogColor;
uniform int fogMode;
uniform int fogShape;
uniform int isEyeInWater;

uniform int moonPhase;
uniform float sunAngle;

vec3 getLight(vec3 sunrise, vec3 noon, vec3 sunset, vec3 evening, vec3 night, vec3 midnight) {
    float state = (sunAngle * 6.0) - 0.5;
    float stateStep = floor(state);
    state -= stateStep;

    if (stateStep == 0) {
        return mix(sunrise, noon, state);
    }
    else if (stateStep == 1) {
        return mix(noon, sunset, state);
    }
    else if (stateStep == 2) {
        return mix(sunset, evening, state);
    }
    else if (stateStep == 3) {
        return mix(evening, night, state);
    }
    else if (stateStep == 4) {
        return mix(night, midnight, state);
    }
    else {
        return mix(midnight, sunrise, state);
    }
}


float getSunLightPower(vec2 lightmapData) {
    return lightmapData.g * mix(1.0, 0.0, rainStrength);
}

vec3 getAmbientLightColor() {
    return ambientLightColor;
}

vec3 getSkyColor() {
    return skyColor;
}

vec3 getSkyLightColor() {
    vec3 curSkyColor = getSkyColor();

    vec3 sunrise = curSkyColor;
    vec3 noon = curSkyColor;
    vec3 sunset = curSkyColor;
    vec3 evening = curSkyColor + vec3(0.1);
    vec3 night = curSkyColor + vec3(0.1);
    vec3 midnight = curSkyColor + vec3(0.1);

    return getLight(sunrise, noon, sunset, evening, night, midnight);
}

vec3 getSunLightColor() {
    vec3 sunrise = vec3(0.7, 0.2, 0.0);
    vec3 noon = vec3(0.7, 0.45, 0.1);
    vec3 sunset = vec3(0.7, 0.2, 0.0);
    vec3 evening = vec3(0.0, 0.2, 0.5);
    vec3 night = vec3(0.0, 0.2, 0.5);
    vec3 midnight = vec3(0.0, 0.2, 0.5);

    vec3 rainSunLightColor = skyColor;
    return mix(getLight(sunrise, noon, sunset, evening, night, midnight), rainSunLightColor, rainStrength);
}

vec3 getBlockLightColor() {
    return blockLightColor;
}

vec3 getFogColor() {
    return fogColor;
}


bool isNight() {
    return sunAngle > 0.5;
}


vec3 getSun(vec3 color, sampler2D suntexture, vec2 coord, float visible) {
    float sunGlow = length(coord) / 1.0;
    sunGlow = 1.0 - sunGlow;
    color = mix(color, vec3(1, 0.6, 0), clamp(sunGlow, 0, visible));

    if (coord.x > -1 && coord.x < 1 && coord.y > -1 && coord.y < 1) {
        vec4 suntexcolor = texture(suntexture, coord * 0.5 + 0.5);
        color.rgb += suntexcolor.rgb * visible;
    }

    return color;
}


vec3 getMoon(vec3 color, sampler2D moontexture, vec2 coord, float visible) {
    if (coord.x > -1 && coord.x < 1 && coord.y > -1 && coord.y < 1) {
        ivec2 index = ivec2(moonPhase % 4, moonPhase / 4);
        
        vec4 suntexcolor = texture(moontexture, ((coord * 0.5 + 0.5) + index) * vec2(0.25, 0.5));
        color.rgb += suntexcolor.rgb * visible;
    }

    return color;
}


vec3 getSky(vec3 direction, sampler2D suntexture, sampler2D moontexture) {
    float factor = direction.y * 2.0;
    factor = clamp(factor, 0, 1);

    vec3 curSkyColor = getSkyColor();
    float sunVisible = rainStrength;
    if (blindness != 0) {
        sunVisible = blindness;
        curSkyColor = mix(curSkyColor, fogColor, blindness);
    }
    else if (darknessFactor != 0) {
        sunVisible = darknessFactor;
        curSkyColor = mix(curSkyColor, fogColor, darknessFactor);
    }

    sunVisible = 1.0 - sunVisible;

    



    vec3 rotatedDirection = (shadowModelView * vec4(direction, 0.0)).xyz;
    vec2 sunCoord = rotatedDirection.yx / 0.3;
    vec2 moonCoord = rotatedDirection.yx / 0.2;

    vec3 color = mix(getFogColor(), curSkyColor, factor);

    if (rotatedDirection.z > 0) {
        if (isNight()) {
            color = getMoon(color, moontexture, moonCoord, sunVisible);
        }
        else {
            color = getSun(color, suntexture, sunCoord, sunVisible);
        }
    }
    else {
        sunCoord = sunCoord * vec2(1, -1);
        moonCoord = moonCoord * vec2(1, -1);
        if (isNight()) {
            color = getSun(color, suntexture, sunCoord, sunVisible);
        }
        else {
            color = getMoon(color, moontexture, moonCoord, sunVisible);
        }
    }
    

    return color;
}



float getFogEnd() {
    if (fogEnd > far) {
        return far;
    }
    return fogEnd;
}

float getFogStart() {
    if (fogStart == 0) {
        return near;
    }
    return fogStart;
}

float getFog(vec3 position) {
    float dist = 0.0;
    if (fogShape == 0) {
        dist = length(position);
    }
    else {
        dist = length(position.xz);
    }

    float start = getFogStart();
    float end = getFogEnd();
    if (start > dist) {
        return 0;
    }
    else if (end < dist) {
        return 1;
    }

    float factor = clamp(dist, start, end) - start;
    factor /= end - start;

    if (isEyeInWater == 0) {
        factor = exp(-5 * (1.0 - factor));
    }

    return factor;
}

vec4 getCloudVoxel(sampler2D cloudtexture, ivec3 voxelPos) {
    if (voxelPos.y != 0) return vec4(0);
    return texelFetch(cloudtexture, voxelPos.xz % 256, 0);
}

#endif