package net.nikdo53.tinymultiblocklib.block;

import com.mojang.datafixers.util.Pair;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1750;
import net.minecraft.class_1799;
import net.minecraft.class_1922;
import net.minecraft.class_1936;
import net.minecraft.class_1937;
import net.minecraft.class_1941;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2586;
import net.minecraft.class_265;
import net.minecraft.class_2680;
import net.minecraft.class_2753;
import net.minecraft.class_2758;
import net.minecraft.class_3726;
import net.minecraft.class_4538;
import net.minecraft.class_5425;
import net.minecraft.world.level.*;
import net.minecraft.world.level.block.*;
import net.minecraft.world.level.block.state.properties.*;
import net.nikdo53.tinymultiblocklib.blockentities.IMultiBlockEntity;
import net.nikdo53.tinymultiblocklib.components.IBlockPosOffsetEnum;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import static net.nikdo53.tinymultiblocklib.Constants.*;
import static net.nikdo53.tinymultiblocklib.block.AbstractMultiBlock.CENTER;

public interface IMultiBlock extends IMBStateSyncer {

    /** Returns a BlockPos Stream of every block in this multiblock.
     * <p>
     * Should only be used for overriding
     * @see #getFullBlockShapeNoCache(class_2338, class_2680)
     * */
    List<class_2338> makeFullBlockShape(@Nullable class_2350 direction, class_2338 center, class_2680 state);

    /**
     * Mojangs BetweenClosed methods return a mutable BlockPos, which breaks everything.
     * Use this helper method to convert them safely
     * */
    static List<class_2338> posStreamToList(Stream<class_2338> posStream){
        return new ArrayList<>(posStream.map(class_2338::method_10062).toList());
    }

    /**
     * Returns the multiblocks DirectionProperty.
     * <p>
     * Only used for multiblocks that can be rotated, otherwise returns null
     * */
    default @Nullable class_2753 getDirectionProperty(){
        return null; // null if block doesn't have directions
    }

    default @Nullable class_2350 getDirection(class_2680 state){
        if (getDirectionProperty() != null){
            return state.method_11654(getDirectionProperty());
        }
        return null;
    }

    default List<class_2338> getFullBlockShapeNoCache(class_2338 center, class_2680 state){
        List<class_2338> list;

        if (getDirectionProperty() == null) {
            list = makeFullBlockShape(null, center, state);
        } else {
            list = makeFullBlockShape(state.method_11654(getDirectionProperty()), center, state);
        }

        // Warn everyone of Mo-jank
        Set<class_2338> set = new HashSet<>(list);
        if (set.size() < list.size()) {
            LOGGER.error("Multiblock {} at {} has overlapping blocks in it's shape,"
                    + " this is likely caused by the BlockPos being mutable."
                    + " Either map them to BlockPos::immutable or use IMultiBlock.posStreamToList()",
                    state.toString(), center);
        }


        return list;
    }


    default List<class_2338> getFullBlockShape(class_2338 pos, class_2680 state, class_1922 level){
        class_2338 center = getCenter(level, pos);

        if (!(level.method_8321(center) instanceof IMultiBlockEntity blockEntity)){
            return getFullBlockShapeNoCache(center, state);
        }

        if (blockEntity.getFullBlockShapeCache().isEmpty()){
            List<class_2338> blockPosList = getFullBlockShapeNoCache(center, state);
            blockPosList.forEach(class_2338::method_10062);

            blockEntity.setFullBlockShapeCache(blockPosList);
            return blockPosList;
        }

        return blockEntity.getFullBlockShapeCache();
    }

    static List<class_2338> getFullShape(class_1937 level, class_2338 pos){
        class_2680 state = level.method_8320(pos);
        if (state.method_26204() instanceof IMultiBlock multiBlock){
            return multiBlock.getFullBlockShape(pos, state, level);
        }

        return List.of(pos);
    }

    static void invalidateCaches(class_1922 level, class_2338 pos){
        if (level.method_8321(getCenter(level, pos)) instanceof IMultiBlockEntity blockEntity){
            blockEntity.invalidateCaches();
        }
    }

    /**
     * Changes the BlockState for each Block based on its offset from center
     * @see IBlockPosOffsetEnum#fromOffset(Class, class_2338, class_2350, Enum)
     * */
    default class_2680 getStateFromOffset(class_2680 state, class_2338 offset){
        return state;
    }

    default void onPlaceHelper(class_2680 state, class_1937 level, class_2338 pos, class_2680 oldState) {
        boolean isPlaced = IMultiBlockEntity.isPlaced(level, pos);

        if (isPlaced) syncBlockStates(level, pos, state);

        if (isCenter(state)) {
            if (!isPlaced) place(level, pos, state);
        }
    }

    /**
     * Places the multiblock, sets its BlockStates and BlockEntity center
     * */
    default void place(class_1937 level, class_2338 centerPos, class_2680 stateOriginal){
        prepareForPlace(level, centerPos, stateOriginal).forEach(pair -> {
            int flags = 66;

            class_2680 stateNew = pair.getSecond();
            class_2338 posNew = pair.getFirst();

            // Don't replace identical blocks
            if (!level.method_8320(posNew).equals(stateNew)) {
                level.method_8652(posNew, stateNew, flags);
            }

            if(level.method_8321(posNew) instanceof IMultiBlockEntity entity && !entity.getCenter().equals(centerPos)) {
                entity.setCenter(centerPos);
                entity.getBlockEntity().method_5431();
            }
        });
    }

    /**
     * Prepares all blocks to be Placed
     * */
    default List<Pair<class_2338, class_2680>> prepareForPlace(class_1937 level, class_2338 centerPos, class_2680 stateOriginal){
        List<Pair<class_2338, class_2680>> list = new ArrayList<>();

        getFullBlockShape(centerPos, stateOriginal, level).forEach(posNew -> {
            posNew = posNew.method_10062();

            class_2680 stateNew = stateOriginal.method_11657(AbstractMultiBlock.CENTER, centerPos.equals(posNew));
            stateNew = getStateFromOffset(stateNew, posNew.method_10059(centerPos));

            list.add(new Pair<>(posNew, stateNew));
        });

        return list;
    }


    default class_2680 getStateForPlacementHelper(class_1750 context) {
        return getStateForPlacementHelper(context, context.method_8042());
    }

    /**
     * Helper for {@link class_2248#method_9605(class_1750)}
     * @param direction The direction the block will have when placed, ignored when {@link #getDirectionProperty()} is null
     * */
    default class_2680 getStateForPlacementHelper(class_1750 context, class_2350 direction) {
        class_4538 level = context.method_8045();
        class_2338 pos = context.method_8037();
        class_2680 state = self().method_9564().method_11657(CENTER, true);

        if (getDirectionProperty() != null){
            state = state.method_11657(getDirectionProperty(), direction);
        }

        return canPlace(level, pos, state, context.method_8036(), false) ? state : null;
    }

    default boolean canPlace(class_4538 level, class_2338 center, class_2680 state, @Nullable class_1297 player, boolean ignoreEntities) {
        return getFullBlockShape(center, state, level).stream().allMatch(blockPos ->
                level.method_8320(blockPos).method_45474()
                        && extraSurviveRequirements(level, blockPos, state)
                        && (entityUnobstructed(level, blockPos, state, player) || ignoreEntities));
    }

    default boolean entityUnobstructed(class_1941 level, class_2338 pos, class_2680 state, @Nullable class_1297 player) {
        class_3726 context = player == null ? class_3726.method_16194() : class_3726.method_16195(player);

        return getFullBlockShape(pos, state, level).stream().allMatch(blockPos -> level.method_8628(state, blockPos, context));
    }

    default void destroy(class_2338 center, class_1937 level, class_2680 state, boolean dropBlock){
        if (level.method_8608()) return;
        List<class_2338> blocks = getFullBlockShape(center, state, level);

        level.method_22352(center, false);

        blocks.forEach(pos ->{
            class_2680 blockState = level.method_8320(pos);
            class_2248 block = state.method_26204();
            if (blockState.method_27852(block)) {
                level.method_22352(pos, dropBlock);
            }
        });
    }

    default boolean allBlocksPresent(class_4538 level, class_2338 pos, class_2680 state){
        if (level.method_8608()) return true;
        class_2338 center = getCenter(level, pos);

        boolean ret = getFullBlockShape(center, state, level).stream().allMatch(blockPos -> level.method_8320(blockPos).method_27852(self()));

        boolean isMultiblock = isMultiblock(level, pos);
        if (ret && level.method_8321(pos) instanceof IMultiBlockEntity entity && !entity.isPlaced() && isMultiblock) {
            getFullBlockShape(center, state, level).forEach(blockPos -> IMultiBlockEntity.setPlaced(level, blockPos, true));
        }

        return ret;
    }

    /**
     * Helper for Block.updateShape()
     * <p>
     * Destroys the multiblock if canSurvive returns false
     * */
    default class_2680 updateShapeHelper(class_2680 state, class_1936 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity){
            class_2338 centerPos = getCenter(level, pos);

            boolean canSurvive = state.method_26184(level, centerPos);
            if (!canSurvive){
                destroy(entity.getCenter(), (class_1937) level, state, true);
                return class_2246.field_10124.method_9564();
            }
        }else {
            return class_2246.field_10124.method_9564();
        }

        return state;
    }

    /**
     * Helper for Block.canSurvive()
     * */
    default boolean canSurviveHelper(class_2680 state, class_4538 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity){
            //survive logic
            boolean extraSurvive = getFullBlockShape(pos, state, level).stream().allMatch(blockPos -> extraSurviveRequirements(level, blockPos, state));
            return (allBlocksPresent(level, pos, state) || !entity.isPlaced()) && extraSurvive;
        } else {
            //placement logic
            return canPlace(level, pos, state, null, false);
        }
    }

    /**
     * Extra requirements for the block to survive or be placed, runs for every single block in the multiblock
     * */
    default boolean extraSurviveRequirements(class_4538 level, class_2338 pos, class_2680 state){
        return true;
    }

    /**
     * Should be added into {@link class_2248#method_9556(class_1937, class_1657, class_2338, class_2680, class_2586, class_1799)}
     * */
    default void preventCreativeDrops(class_1657 player, class_1937 level, class_2338 pos){
        if (player.method_7337() && level.method_8321(pos) instanceof IMultiBlockEntity) {
            class_2338 center = getCenter(level, pos);

            destroy(center, level, level.method_8320(pos), false);
        }
    }


    /**
     * Prevents desyncs and ghost blocks when multiblocks are used in structures
     * */
    default void fixInStructures(class_2680 state, class_5425 level, class_2338 pos){
        if (isCenter(state)) {
            level.method_39279(pos, state.method_26204(), 3);
        }
    }

    /**
     * Tries to fix the multiblock, called after {@link #fixInStructures(class_2680, class_5425, class_2338)}
     * */
    default void fixTick(class_2680 state, class_1937 level, class_2338 pos){
        if (isCenter(state)){

            getFullBlockShape(pos, state, level).forEach(posNew -> {
                if (level.method_8321(posNew) instanceof IMultiBlockEntity entity) {
                    entity.setCenter(pos);

                    entity.getBlockEntity().method_5431();
                    level.method_8413(posNew, state, state, 2);
                }
            });
        }
    }

    /**
     * Checks if the multiblock needs fixing by  {@link #fixTick(class_2680, class_1937, class_2338)}
     * */
    default boolean isBroken(class_4538 level, class_2338 pos, class_2680 state){
        if (!isCenter(state)) return false;

        return getFullBlockShape(pos, state, level).stream().anyMatch(blockPos -> {
            if (level.method_8321(blockPos) instanceof IMultiBlockEntity entity){
                return !(entity.getCenter().equals(pos) && !isCenter(level.method_8320(blockPos)));
            }
            return true;
        });
    }

    /**
     * Returns the center BlockPos of the multiblock
     * */
    static class_2338 getCenter(class_1922 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity){
            return entity.getCenter();
        }
        return pos;
    }

    static boolean isCenter(class_4538 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity) {
            return entity.getCenter().equals(pos);
        }
        return false;
    }

    static boolean isCenter(class_2680 state){
        return state.method_11654(CENTER);
    }

    static boolean isMultiblock(class_2680 state){
        return state.method_26204() instanceof IMultiBlock;
    }

    static boolean isMultiblock(class_1922 level, class_2338 pos){
        return isMultiblock(level.method_8320(pos));
    }

    static int getXOffset(class_1922 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity) {
            return pos.method_10263() - entity.getCenter().method_10263();
        }
        return 0;
    }

    static int getYOffset(class_1922 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity) {
            return pos.method_10264() - entity.getCenter().method_10264();
        }
        return 0;
    }

    static int getZOffset(class_1922 level, class_2338 pos){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity) {
            return pos.method_10260() - entity.getCenter().method_10260();
        }
        return 0;
    }

    default class_265 voxelShapeHelper(class_2680 state, class_1922 level, class_2338 pos, class_265 shape){
        return voxelShapeHelper(state,level,pos,shape, 0, 0, 0);
    }

    default class_265 voxelShapeHelper(class_2680 state, class_1922 level, class_2338 pos, class_265 shape, float xOffset, float yOffset, float zOffset){
        return voxelShapeHelper(state,level,pos,shape, xOffset, yOffset, zOffset, false);
    }

    /**
     * Offsets each Blocks VoxelShape to the center, allowing for VoxelShapes larger than 1 block
     * @param hasDirectionOffsets Larger directional multiblocks may have their center in a different point for every rotation, this offsets the VoxelShapes accordingly
     * */
    default class_265 voxelShapeHelper(class_2680 state, class_1922 level, class_2338 pos, class_265 shape, float xOffset, float yOffset, float zOffset, boolean hasDirectionOffsets){
        if (level.method_8321(pos) instanceof IMultiBlockEntity entity) {
            var x = entity.getCenter().method_10263() - pos.method_10263() + xOffset;
            var y = entity.getCenter().method_10264() - pos.method_10264() + yOffset;
            var z = entity.getCenter().method_10260() - pos.method_10260() + zOffset;

            if (getDirectionProperty() != null && hasDirectionOffsets) {
                switch (state.method_11654(getDirectionProperty())) {
                    case field_11034 -> x += 1;
                    case field_11043 -> {
                        x += 1;
                        z -= 1;
                    }
                    case field_11039 -> z -= 1;
                }
            }
            return shape.method_1096(x,y,z);
        }
        return shape;
    }

    /**
     * Increases age in each block at the same time
     * <p>
     * If used in the randomTick method, don't forget to check {@link #isCenter(class_2680)} first,
     * otherwise the block will grow significantly faster (each block tick separately)
     * @deprecated Use synced blockStates instead
     * */
    @Deprecated
    default void growHelper(class_1937 level, class_2338 blockPos, class_2758 ageProperty){
        class_2248 block = self();
            getFullBlockShape(blockPos, level.method_8320(blockPos), level).forEach(pos -> {
                if(level.method_8320(pos).method_27852(block)) {

                    class_2680 blockState = level.method_8320(pos);
                    int age = blockState.method_11654(ageProperty);
                    if (blockState.method_11654(ageProperty) >= getMaxAge(ageProperty)) return;

                    level.method_8652(pos, blockState.method_11657(ageProperty,age + 1), 2);

                }else {
                    level.method_22352(pos, false);
                }
            });
    }

    default int getMaxAge(class_2758 ageProperty) {
        return ageProperty.method_11898().stream().toList().get(ageProperty.method_11898().size() - 1);
    }

    static boolean isSameMultiblock(class_1937 level, class_2680 state1, class_2680 state2, class_2338 center, class_2338 posNew){
        return state1.method_26204().equals(state2.method_26204()) && level.method_8321(posNew) instanceof IMultiBlockEntity entity && entity.getCenter().equals(center);
    }

    private class_2248 self(){
        if (this instanceof class_2248 block){
            return block;
        } else {
            throw new RuntimeException(this.getClass().getSimpleName() + " is not implemented on a Block");
        }
    }

}
