package gloomyfolken.hooklib.asm;

import gloomyfolken.hooklib.asm.HookInjectorFactory.*;
import org.objectweb.asm.*;

import java.util.*;

import static org.objectweb.asm.Opcodes.*;
import static org.objectweb.asm.Type.*;

/**
 * Класс, отвечающий за установку одного хука в один метод.
 * Терминология:
 * hook (хук) - вызов вашего статического метода из стороннего кода (майнкрафта, форджа, других модов)
 * targetMethod (целевой метод) - метод, куда вставляется хук
 * targetClass (целевой класс) - класс, где находится метод, куда вставляется хук
 * hookMethod (хук-метод) - ваш статический метод, который вызывается из стороннего кода
 * hookClass (класс с хуком) - класс, в котором содержится хук-метод
 */
public class AsmHook implements Cloneable, Comparable<AsmHook> {
	
	public static final HookInjectorFactory ON_ENTER_FACTORY = MethodEnter.INSTANCE;
	public static final HookInjectorFactory ON_EXIT_FACTORY = MethodExit.INSTANCE;
	private String targetClassName; // через точки
	private String targetMethodName;
	private List<Type> targetMethodParameters = new ArrayList<>(2);
	private Type targetMethodReturnType; //если не задано, то не проверяется
	private String hooksClassName; // через точки
	private String hookMethodName;
	// -1 - значение return
	private List<Integer> transmittableVariableIds = new ArrayList<>(2);
	private List<Type> hookMethodParameters = new ArrayList<>(2);
	private Type hookMethodReturnType = Type.VOID_TYPE;
	private boolean hasReturnValueParameter; // если в хук-метод передается значение из return
	private ReturnCondition returnCondition = ReturnCondition.NEVER;
	private ReturnValue returnValue = ReturnValue.VOID;
	private Object primitiveConstant;
	private HookInjectorFactory injectorFactory = ON_ENTER_FACTORY;
	private HookPriority priority = HookPriority.NORMAL;
	// может быть без возвращаемого типа
	private String targetMethodDescription;
	private String hookMethodDescription;
	private String returnMethodName;
	// может быть без возвращаемого типа
	private String returnMethodDescription;
	private boolean isAbstract;
	private boolean isStatic;
	private String superClass;
	private boolean createMethod;
	private boolean isMandatory;
	
	public static Builder newBuilder() {
		return new AsmHook().new Builder();
	}
	
	protected String getTargetClassName() {
		return targetClassName;
	}
	
	protected boolean isTargetMethod(String name, String desc) {
		return (targetMethodReturnType == null && desc.startsWith(targetMethodDescription) ||
				desc.equals(targetMethodDescription)) && name.equals(targetMethodName);
	}
	
	protected boolean isAbstract() {
		return this.isAbstract;
	}
	
	protected boolean isStatic() {
		return isStatic;
	}
	
	protected String getSuperClass() {
		return superClass;
	}
	
	protected boolean getCreateMethod() {
		return createMethod;
	}
	
	protected boolean isMandatory() {
		return isMandatory;
	}
	
	protected HookInjectorFactory getInjectorFactory() {
		return injectorFactory;
	}
	
	protected void createMethod(HookInjectorClassVisitor classVisitor, boolean isStatic, boolean isAbstract, String superClass) {
		if (isStatic && isAbstract) throw new IllegalArgumentException("Cannot make static abstract class");
		
		ClassMetadataReader.MethodReference superMethod = classVisitor.transformer.classMetadataReader.findVirtualMethod(getTargetClassInternalName(), targetMethodName, targetMethodDescription);
		// юзаем название суперметода, потому что findVirtualMethod может вернуть метод с другим названием
		MethodVisitor mv;
		
		int access = Opcodes.ACC_PUBLIC;
		if (isStatic) {
			access |= Opcodes.ACC_STATIC;
		} else if (isAbstract) {
			access |= Opcodes.ACC_ABSTRACT;
		}
		
		mv = classVisitor.visitMethod(access, superMethod == null ? this.targetMethodName : superMethod.name, this.targetMethodDescription, null, null);
		
		if (!(mv instanceof HookInjectorMethodVisitor)) {
			throw new IllegalArgumentException("Hook injector not created");
		}
		
		HookInjectorMethodVisitor inj = (HookInjectorMethodVisitor) mv;
		
		if (!isAbstract) {
			inj.visitCode();
			if (superClass.length() > 0) {
				inj.visitVarInsn(ALOAD, 0);
//				inj.visitMethodInsn(INVOKESPECIAL, superClass, "<init>", "()V", false);
				// KAIIIAK
				String[] superClassInfoList = superClass.split("\\.");
				if (superClassInfoList.length == 1) { // только owner
					inj.visitMethodInsn(INVOKESPECIAL, superClass, "<init>", "()V", false);
				} else if (superClassInfoList.length == 2) { // owner и desc // бесполезно, ибо не подгружает параметры
					inj.visitMethodInsn(INVOKESPECIAL, superClassInfoList[0], "<init>", superClassInfoList[1], false);
				} else if (superClassInfoList.length == 3) {
					throw new IllegalArgumentException("superClass arguments count must not be odd"); //не четные
				} else if (superClassInfoList.length > 3) {
					if ((superClassInfoList.length & 1) == 0) {
						for (int i = 0; i < superClassInfoList.length - 2; i += 2) {
							if (i + 2 != superClassInfoList.length - 1) {
								inj.visitVarInsn(Integer.parseInt(superClassInfoList[i + 1]), Integer.parseInt(superClassInfoList[i + 2]));
							} else {
								throw new IllegalArgumentException("? superClass is not correct ?"); //?никогда не сработает?
							}
						}
						inj.visitMethodInsn(INVOKESPECIAL, superClassInfoList[0], "<init>", superClassInfoList[superClassInfoList.length - 1], false);
					} else {
						throw new IllegalArgumentException("superClass is not even"); //не четные
					}
				}
				// KAIIIAK
			}
			inj.visitLabel(new Label());
			if (superMethod == null) {
				injectDefaultValue(inj, targetMethodReturnType);
			} else {
				injectSuperCall(inj, superMethod);
			}
			injectReturn(inj, targetMethodReturnType);
			inj.visitLabel(new Label());
			inj.visitMaxs(0, 0);
		}
		inj.visitEnd();
	}
	
	private String getTargetClassInternalName() {
		return targetClassName.replace('.', '/');
	}
	
	private void injectDefaultValue(HookInjectorMethodVisitor inj, Type targetMethodReturnType) {
		switch (targetMethodReturnType.getSort()) {
			case Type.VOID:
				break;
			case Type.BOOLEAN:
			case Type.CHAR:
			case Type.BYTE:
			case Type.SHORT:
			case Type.INT:
				inj.visitInsn(Opcodes.ICONST_0);
				break;
			case Type.FLOAT:
				inj.visitInsn(Opcodes.FCONST_0);
				break;
			case Type.LONG:
				inj.visitInsn(Opcodes.LCONST_0);
				break;
			case Type.DOUBLE:
				inj.visitInsn(Opcodes.DCONST_0);
				break;
			default:
				inj.visitInsn(Opcodes.ACONST_NULL);
				break;
		}
	}
	
	private void injectSuperCall(HookInjectorMethodVisitor inj, ClassMetadataReader.MethodReference method) {
		int variableId = 0;
		for (int i = 0; i <= targetMethodParameters.size(); i++) {
			Type parameterType = i == 0 ? TypeHelper.getType(targetClassName) : targetMethodParameters.get(i - 1);
			injectLoad(inj, parameterType, variableId);
			if (parameterType.getSort() == Type.DOUBLE || parameterType.getSort() == Type.LONG) {
				variableId += 2;
			} else {
				variableId++;
			}
		}
		inj.visitMethodInsn(INVOKESPECIAL, method.owner, method.name, method.desc, false);
	}
	
	private void injectReturn(HookInjectorMethodVisitor inj, Type targetMethodReturnType) {
		if (targetMethodReturnType == INT_TYPE || targetMethodReturnType == SHORT_TYPE ||
				targetMethodReturnType == BOOLEAN_TYPE || targetMethodReturnType == BYTE_TYPE
				|| targetMethodReturnType == CHAR_TYPE) {
			inj.visitInsn(IRETURN);
		} else if (targetMethodReturnType == LONG_TYPE) {
			inj.visitInsn(LRETURN);
		} else if (targetMethodReturnType == FLOAT_TYPE) {
			inj.visitInsn(FRETURN);
		} else if (targetMethodReturnType == DOUBLE_TYPE) {
			inj.visitInsn(DRETURN);
		} else if (targetMethodReturnType == VOID_TYPE) {
			inj.visitInsn(RETURN);
		} else {
			inj.visitInsn(ARETURN);
		}
	}
	
	private void injectLoad(HookInjectorMethodVisitor inj, Type parameterType, int variableId) {
		int opcode;
		if (parameterType == INT_TYPE || parameterType == BYTE_TYPE || parameterType == CHAR_TYPE ||
				parameterType == BOOLEAN_TYPE || parameterType == SHORT_TYPE) {
			opcode = ILOAD;
		} else if (parameterType == LONG_TYPE) {
			opcode = LLOAD;
		} else if (parameterType == FLOAT_TYPE) {
			opcode = FLOAD;
		} else if (parameterType == DOUBLE_TYPE) {
			opcode = DLOAD;
		} else {
			opcode = ALOAD;
		}
		inj.visitVarInsn(opcode, variableId);
	}
	
	protected void inject(HookInjectorMethodVisitor inj) {
		Type targetMethodReturnType = inj.methodType.getReturnType();
		
		// сохраняем значение, которое было передано return в локальную переменную
		int returnLocalId = -1;
		if (hasReturnValueParameter) {
			returnLocalId = inj.newLocal(targetMethodReturnType);
			inj.visitVarInsn(targetMethodReturnType.getOpcode(54), returnLocalId); //storeLocal
		}
		
		// вызываем хук-метод
		int hookResultLocalId = -1;
		if (hasHookMethod()) {
			injectInvokeStatic(inj, returnLocalId, hookMethodName, hookMethodDescription);
			
			if (returnValue == ReturnValue.HOOK_RETURN_VALUE || returnCondition.requiresCondition) {
				hookResultLocalId = inj.newLocal(hookMethodReturnType);
				inj.visitVarInsn(hookMethodReturnType.getOpcode(54), hookResultLocalId); //storeLocal
			}
		}
		
		// вызываем return
		if (returnCondition != ReturnCondition.NEVER) {
			Label label = inj.newLabel();
			
			// вставляем GOTO-переход к label'у после вызова return
			if (returnCondition != ReturnCondition.ALWAYS) {
				inj.visitVarInsn(hookMethodReturnType.getOpcode(21), hookResultLocalId); //loadLocal
				if (returnCondition == ReturnCondition.ON_TRUE) {
					inj.visitJumpInsn(IFEQ, label);
				} else if (returnCondition == ReturnCondition.ON_NULL) {
					inj.visitJumpInsn(IFNONNULL, label);
				} else if (returnCondition == ReturnCondition.ON_NOT_NULL) {
					inj.visitJumpInsn(IFNULL, label);
				}
			}
			
			// вставляем в стак значение, которое необходимо вернуть
			if (returnValue == ReturnValue.NULL) {
				inj.visitInsn(Opcodes.ACONST_NULL);
			} else if (returnValue == ReturnValue.PRIMITIVE_CONSTANT) {
				inj.visitLdcInsn(primitiveConstant);
			} else if (returnValue == ReturnValue.HOOK_RETURN_VALUE) {
				inj.visitVarInsn(hookMethodReturnType.getOpcode(21), hookResultLocalId); //loadLocal
			} else if (returnValue == ReturnValue.ANOTHER_METHOD_RETURN_VALUE) {
				String returnMethodDescription = this.returnMethodDescription;
				// если не был определён заранее нужный возвращаемый тип, то добавляем его к описанию
				if (returnMethodDescription.endsWith(")")) {
					returnMethodDescription += targetMethodReturnType.getDescriptor();
				}
				injectInvokeStatic(inj, returnLocalId, returnMethodName, returnMethodDescription);
			}
			
			// вызываем return
			injectReturn(inj, targetMethodReturnType);
			
			// вставляем label, к которому идет GOTO-переход
			inj.visitLabel(label);
		}
		
		//кладем в стек значение, которое шло в return
		if (hasReturnValueParameter) {
			injectLoad(inj, targetMethodReturnType, returnLocalId);
		}
	}
	
	private boolean hasHookMethod() {
		return hookMethodName != null && hooksClassName != null;
	}
	
	private void injectInvokeStatic(HookInjectorMethodVisitor inj, int returnLocalId, String name, String desc) {
		for (int i = 0; i < hookMethodParameters.size(); i++) {
			Type parameterType = hookMethodParameters.get(i);
			int variableId = transmittableVariableIds.get(i);
			if (inj.isStatic) {
				// если попытка передачи this из статического метода, то передаем null
				if (variableId == 0) {
					inj.visitInsn(Opcodes.ACONST_NULL);
					continue;
				}
				// иначе сдвигаем номер локальной переменной
				if (variableId > 0) variableId--;
			}
			if (variableId == -1) variableId = returnLocalId;
			injectLoad(inj, parameterType, variableId);
		}
		
		inj.visitMethodInsn(INVOKESTATIC, getHookClassInternalName(), name, desc, false);
	}
	
	private String getHookClassInternalName() {
		return hooksClassName.replace('.', '/');
	}
	
	public String getPatchedMethodName() {
		return targetClassName + '#' + targetMethodName + targetMethodDescription;
	}
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("AsmHook: ");
		
		sb.append(targetClassName).append('#').append(targetMethodName);
		sb.append(targetMethodDescription);
		sb.append(" -> ");
		sb.append(hooksClassName).append('#').append(hookMethodName);
		sb.append(hookMethodDescription);
		
		sb.append(", ReturnCondition=").append(returnCondition);
		sb.append(", ReturnValue=").append(returnValue);
		if (returnValue == ReturnValue.PRIMITIVE_CONSTANT) sb.append(", Constant=").append(primitiveConstant);
		sb.append(", InjectorFactory: ").append(injectorFactory.getClass().getName());
		sb.append(", CreateMethod = ").append(createMethod);
		
		return sb.toString();
	}
	
	@Override
	public int compareTo(AsmHook o) {
		if (injectorFactory.isPriorityInverted && o.injectorFactory.isPriorityInverted) {
			return priority.ordinal() > o.priority.ordinal() ? -1 : 1;
		} else if (!injectorFactory.isPriorityInverted && !o.injectorFactory.isPriorityInverted) {
			return priority.ordinal() > o.priority.ordinal() ? 1 : -1;
		} else {
			return injectorFactory.isPriorityInverted ? 1 : -1;
		}
	}
	
	public class Builder extends AsmHook {
		
		private Builder() {
		
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ ---
		 * Определяет название класса, в который необходимо установить хук.
		 *
		 * @param className Название класса с указанием пакета, разделенное точками.
		 * Например: net.minecraft.world.World
		 */
		public Builder setTargetClass(String className) {
			AsmHook.this.targetClassName = className;
			return this;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ ---
		 * Определяет название метода, в который необходимо вставить хук.
		 * Если нужно пропатчить конструктор, то в названии метода нужно указать <init>.
		 *
		 * @param methodName Название метода.
		 * Например: getBlockId
		 */
		public Builder setTargetMethod(String methodName) {
			AsmHook.this.targetMethodName = methodName;
			return this;
		}
		
		/**
		 * Добавляет один или несколько параметров к списку параметров целевого метода.
		 * Обёртка над addTargetMethodParameters(Type... parameterTypes), которая сама строит типы из названия.
		 *
		 * @param parameterTypeNames Названия классов параметров целевого метода.
		 * Например: net.minecraft.world.World
		 */
		
		public Builder addTargetMethodParameters(String... parameterTypeNames) {
			Type[] types = new Type[parameterTypeNames.length];
			for (int i = 0; i < parameterTypeNames.length; i++) {
				types[i] = TypeHelper.getType(parameterTypeNames[i]);
			}
			return addTargetMethodParameters(types);
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ У ЦЕЛЕВОГО МЕТОДА ЕСТЬ ПАРАМЕТРЫ ---
		 * Добавляет один или несколько параметров к списку параметров целевого метода.
		 * <p/>
		 * Эти параметры используются, чтобы составить описание целевого метода.
		 * Чтобы однозначно определить целевой метод, недостаточно только его названия - нужно ещё и описание.
		 * <p/>
		 * Примеры использования:
		 * import static gloomyfolken.hooklib.asm.TypeHelper.*
		 * //...
		 * addTargetMethodParameters(Type.INT_TYPE)
		 * Type worldType = getType("net.minecraft.world.World")
		 * Type playerType = getType("net.minecraft.entity.player.EntityPlayer")
		 * addTargetMethodParameters(worldType, playerType, playerType)
		 *
		 * @param parameterTypes Типы параметров целевого метода
		 *
		 * @see TypeHelper
		 */
		public Builder addTargetMethodParameters(Type... parameterTypes) {
			AsmHook.this.targetMethodParameters.addAll(Arrays.asList(parameterTypes));
			return this;
		}
		
		/**
		 * Изменяет тип, возвращаемый целевым методом.
		 * Обёртка над setTargetMethodReturnType(Type returnType)
		 *
		 * @param returnType Название класса, экземпляр которого возвращает целевой метод
		 */
		public Builder setTargetMethodReturnType(String returnType) {
			return setTargetMethodReturnType(TypeHelper.getType(returnType));
		}
		
		/**
		 * Изменяет тип, возвращаемый целевым методом.
		 * Вовращаемый тип используется, чтобы составить описание целевого метода.
		 * Чтобы однозначно определить целевой метод, недостаточно только его названия - нужно ещё и описание.
		 * По умолчанию хук применяется ко всем методам, подходящим по названию и списку параметров.
		 *
		 * @param returnType Тип, возвращаемый целевым методом
		 *
		 * @see TypeHelper
		 */
		public Builder setTargetMethodReturnType(Type returnType) {
			AsmHook.this.targetMethodReturnType = returnType;
			return this;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ НУЖЕН ХУК-МЕТОД, А НЕ ПРОСТО return SOME_CONSTANT ---
		 * Определяет название класса, в котором находится хук-метод.
		 *
		 * @param className Название класса с указанием пакета, разделенное точками.
		 * Например: net.myname.mymod.asm.MyHooks
		 */
		public Builder setHookClass(String className) {
			AsmHook.this.hooksClassName = className;
			return this;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ НУЖЕН ХУК-МЕТОД, А НЕ ПРОСТО return SOME_CONSTANT ---
		 * Определяет название хук-метода.
		 * ХУК-МЕТОД ДОЛЖЕН БЫТЬ СТАТИЧЕСКИМ, А ПРОВЕРКИ НА ЭТО НЕТ. Будьте внимательны.
		 *
		 * @param methodName Название хук-метода.
		 * Например: myFirstHook
		 */
		public Builder setHookMethod(String methodName) {
			AsmHook.this.hookMethodName = methodName;
			return this;
		}
		
		/**
		 * Добавляет параметр в список параметров целевого метода.
		 * Обёртка над addHookMethodParameter(Type parameterType, int variableId)
		 *
		 * @param parameterTypeName Название типа параметра хук-метода.
		 * Например: net.minecraft.world.World
		 * @param variableId ID значения, передаваемого в хук-метод
		 */
		public Builder addHookMethodParameter(String parameterTypeName, int variableId) {
			return addHookMethodParameter(TypeHelper.getType(parameterTypeName), variableId);
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ У ХУК-МЕТОДА ЕСТЬ ПАРАМЕТРЫ ---
		 * Добавляет параметр в список параметров хук-метода.
		 * В байткоде не сохраняются названия параметров. Вместо этого приходится использовать их номера.
		 * Например, в классе EntityLivingBase есть метод attackEntityFrom(DamageSource damageSource, float damage).
		 * В нём будут использоваться такие номера параметров:
		 * 1 - damageSource
		 * 2 - damage
		 * ВАЖНЫЙ МОМЕНТ: LONG И DOUBLE "ЗАНИМАЮТ" ДВА НОМЕРА.
		 * Теоретически, кроме параметров в хук-метод можно передать и локальные переменные, но их
		 * номера сложнее посчитать.
		 * Например, в классе Entity есть метод setPosition(double x, double y, double z).
		 * В нём будут такие номера параметров:
		 * 1 - x
		 * 2 - пропущено
		 * 3 - y
		 * 4 - пропущено
		 * 5 - z
		 * 6 - пропущено
		 * <p/>
		 * Код этого метода таков:
		 * //...
		 * float f = ...;
		 * float f1 = ...;
		 * //...
		 * В таком случае у f будет номер 7, а у f1 - 8.
		 * <p/>
		 * Если целевой метод static, то не нужно начинать отсчет локальных переменных с нуля, номера
		 * будут смещены автоматически.
		 *
		 * @param parameterType Тип параметра хук-метода
		 * @param variableId ID значения, передаваемого в хук-метод
		 *
		 * @throws IllegalStateException если не задано название хук-метода или класса, который его содержит
		 */
		public Builder addHookMethodParameter(Type parameterType, int variableId) {
			if (!AsmHook.this.hasHookMethod()) {
				throw new IllegalStateException("Hook method is not specified, so can not append " +
													"parameter to its parameters list.");
			}
			AsmHook.this.hookMethodParameters.add(parameterType);
			AsmHook.this.transmittableVariableIds.add(variableId);
			return this;
		}
		
		/**
		 * Добавляет в список параметров хук-метода целевой класс и передает хук-методу this.
		 * Если целевой метод static, то будет передано null.
		 *
		 * @throws IllegalStateException если не задан хук-метод
		 */
		public Builder addThisToHookMethodParameters() {
			if (!AsmHook.this.hasHookMethod()) {
				throw new IllegalStateException("Hook method is not specified, so can not append " +
													"parameter to its parameters list.");
			}
			AsmHook.this.hookMethodParameters.add(TypeHelper.getType(targetClassName));
			AsmHook.this.transmittableVariableIds.add(0);
			return this;
		}
		
		/**
		 * Добавляет в список параметров хук-метода тип, возвращаемый целевым методом и
		 * передает хук-методу значение, которое вернёт return.
		 * Более формально, при вызове хук-метода указывает в качестве этого параметра верхнее значение в стеке.
		 * На практике основное применение -
		 * Например, есть такой код метода:
		 * int foo = bar();
		 * return foo;
		 * Или такой:
		 * return bar()
		 * <p/>
		 * В обоих случаях хук-методу можно передать возвращаемое значение перед вызовом return.
		 *
		 * @throws IllegalStateException если целевой метод возвращает void
		 * @throws IllegalStateException если не задан хук-метод
		 */
		public Builder addReturnValueToHookMethodParameters() {
			if (!AsmHook.this.hasHookMethod()) {
				throw new IllegalStateException("Hook method is not specified, so can not append " +
													"parameter to its parameters list.");
			}
			if (AsmHook.this.targetMethodReturnType == Type.VOID_TYPE) {
				throw new IllegalStateException("Target method's return type is void, it does not make sense to " +
													"transmit its return value to hook method.");
			}
			AsmHook.this.hookMethodParameters.add(AsmHook.this.targetMethodReturnType);
			AsmHook.this.transmittableVariableIds.add(-1);
			AsmHook.this.hasReturnValueParameter = true;
			return this;
		}
		
		/**
		 * Задает условие, при котором после вызова хук-метода вызывается return.
		 * По умолчанию return не вызывается вообще.
		 * Кроме того, этот метод изменяет тип возвращаемого значения хук-метода:
		 * NEVER -> void
		 * ALWAYS -> void
		 * ON_TRUE -> boolean
		 * ON_NULL -> Object
		 * ON_NOT_NULL -> Object
		 *
		 * @param condition Условие выхода после вызова хук-метода
		 *
		 * @throws IllegalArgumentException если condition == ON_TRUE, ON_NULL или ON_NOT_NULL, но не задан хук-метод.
		 * @see ReturnCondition
		 */
		public Builder setReturnCondition(ReturnCondition condition) {
			if (condition.requiresCondition && AsmHook.this.hookMethodName == null) {
				throw new IllegalArgumentException("Hook method is not specified, so can not use return " +
													   "condition that depends on hook method.");
			}
			
			AsmHook.this.returnCondition = condition;
			Type returnType;
			switch (condition) {
				case NEVER:
				case ALWAYS:
					returnType = VOID_TYPE;
					break;
				case ON_TRUE:
					returnType = BOOLEAN_TYPE;
					break;
				default:
					returnType = getType(Object.class);
					break;
			}
			AsmHook.this.hookMethodReturnType = returnType;
			return this;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ ЦЕЛЕВОЙ МЕТОД ВОЗВРАЩАЕТ НЕ void, И ВЫЗВАН setReturnCondition ---
		 * Задает значение, которое возвращается при вызове return после вызова хук-метода.
		 * Следует вызывать после setReturnCondition.
		 * По умолчанию возвращается void.
		 * Кроме того, если value == ReturnValue.HOOK_RETURN_VALUE, то этот метод изменяет тип возвращаемого
		 * значения хук-метода на тип, указанный в setTargetMethodReturnType()
		 *
		 * @param value возвращаемое значение
		 *
		 * @throws IllegalStateException если returnCondition == NEVER (т. е. если setReturnCondition() не вызывался).
		 * Нет смысла указывать возвращаемое значение, если return не вызывается.
		 * @throws IllegalArgumentException если value == ReturnValue.HOOK_RETURN_VALUE, а тип возвращаемого значения
		 * целевого метода указан как void (или setTargetMethodReturnType ещё не вызывался).
		 * Нет смысла использовать значение, которое вернул хук-метод, если метод возвращает void.
		 */
		public Builder setReturnValue(ReturnValue value) {
			if (AsmHook.this.returnCondition == ReturnCondition.NEVER) {
				throw new IllegalStateException("Current return condition is ReturnCondition.NEVER, so it does not " +
													"make sense to specify the return value.");
			}
			Type returnType = AsmHook.this.targetMethodReturnType;
			if (value != ReturnValue.VOID && returnType == VOID_TYPE) {
				throw new IllegalArgumentException("Target method return value is void, so it does not make sense to " +
													   "return anything else.");
			}
			if (value == ReturnValue.VOID && returnType != VOID_TYPE) {
				throw new IllegalArgumentException("Target method return value is not void, so it is impossible " +
													   "to return VOID.");
			}
			if (value == ReturnValue.PRIMITIVE_CONSTANT && returnType != null && !isPrimitive(returnType)) {
				throw new IllegalArgumentException("Target method return value is not a primitive, so it is " +
													   "impossible to return PRIVITIVE_CONSTANT.");
			}
			if (value == ReturnValue.NULL && returnType != null && isPrimitive(returnType)) {
				throw new IllegalArgumentException("Target method return value is a primitive, so it is impossible " +
													   "to return NULL.");
			}
			if (value == ReturnValue.HOOK_RETURN_VALUE && !hasHookMethod()) {
				throw new IllegalArgumentException("Hook method is not specified, so can not use return " +
													   "value that depends on hook method.");
			}
			
			AsmHook.this.returnValue = value;
			if (value == ReturnValue.HOOK_RETURN_VALUE) {
				AsmHook.this.hookMethodReturnType = AsmHook.this.targetMethodReturnType;
			}
			return this;
		}
		
		private boolean isPrimitive(Type type) {
			return type.getSort() > 0 && type.getSort() < 9;
		}
		
		/**
		 * Возвращает тип возвращаемого значения хук-метода, если кому-то сложно "вычислить" его самостоятельно.
		 *
		 * @return тип возвращаемого значения хук-метода
		 */
		public Type getHookMethodReturnType() {
			return hookMethodReturnType;
		}
		
		/**
		 * Напрямую указывает тип, возвращаемый хук-методом.
		 *
		 * @param type тип
		 */
		protected void setHookMethodReturnType(Type type) {
			AsmHook.this.hookMethodReturnType = type;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ ВОЗВРАЩАЕМОЕ ЗНАЧЕНИЕ УСТАНОВЛЕНО НА PRIMITIVE_CONSTANT ---
		 * Следует вызывать после setReturnValue(ReturnValue.PRIMITIVE_CONSTANT)
		 * Задает константу, которая будет возвращена при вызове return.
		 * Класс заданного объекта должен соответствовать примитивному типу.
		 * Например, если целевой метод возвращает int, то в этот метод должен быть передан объект класса Integer.
		 *
		 * @param constant Объект, класс которого соответствует примитиву, который следует возвращать.
		 *
		 * @throws IllegalStateException если возвращаемое значение не установлено на PRIMITIVE_CONSTANT
		 * @throws IllegalArgumentException если класс объекта constant не является обёрткой
		 * для примитивного типа, который возвращает целевой метод.
		 */
		public Builder setPrimitiveConstant(Object constant) {
			if (AsmHook.this.returnValue != ReturnValue.PRIMITIVE_CONSTANT) {
				throw new IllegalStateException("Return value is not PRIMITIVE_CONSTANT, so it does not make sence" +
													"to specify that constant.");
			}
			Type returnType = AsmHook.this.targetMethodReturnType;
			if (returnType == BOOLEAN_TYPE && !(constant instanceof Boolean) ||
					returnType == CHAR_TYPE && !(constant instanceof Character) ||
					returnType == BYTE_TYPE && !(constant instanceof Byte) ||
					returnType == SHORT_TYPE && !(constant instanceof Short) ||
					returnType == INT_TYPE && !(constant instanceof Integer) ||
					returnType == LONG_TYPE && !(constant instanceof Long) ||
					returnType == FLOAT_TYPE && !(constant instanceof Float) ||
					returnType == DOUBLE_TYPE && !(constant instanceof Double)) {
				throw new IllegalArgumentException("Given object class does not math target method return type.");
			}
			
			AsmHook.this.primitiveConstant = constant;
			return this;
		}
		
		/**
		 * --- ОБЯЗАТЕЛЬНО ВЫЗВАТЬ, ЕСЛИ ВОЗВРАЩАЕМОЕ ЗНАЧЕНИЕ УСТАНОВЛЕНО НА ANOTHER_METHOD_RETURN_VALUE ---
		 * Следует вызывать после setReturnValue(ReturnValue.ANOTHER_METHOD_RETURN_VALUE)
		 * Задает метод, результат вызова которого будет возвращён при вызове return.
		 *
		 * @param methodName название метода, результат вызова которого следует возвращать
		 *
		 * @throws IllegalStateException если возвращаемое значение не установлено на ANOTHER_METHOD_RETURN_VALUE
		 */
		public Builder setReturnMethod(String methodName) {
			if (AsmHook.this.returnValue != ReturnValue.ANOTHER_METHOD_RETURN_VALUE) {
				throw new IllegalStateException("Return value is not ANOTHER_METHOD_RETURN_VALUE, " +
													"so it does not make sence to specify that method.");
			}
			
			AsmHook.this.returnMethodName = methodName;
			return this;
		}
		
		/**
		 * Задает фабрику, которая создаст инжектор для этого хука.
		 * Если говорить более человеческим языком, то этот метод определяет, где будет вставлен хук:
		 * в начале метода, в конце или где-то ещё.
		 * Если не создавать своих инжекторов, то можно использовать две фабрики:
		 * AsmHook.ON_ENTER_FACTORY (вставляет хук на входе в метод, используется по умолчанию)
		 * AsmHook.ON_EXIT_FACTORY (вставляет хук на выходе из метода)
		 *
		 * @param factory Фабрика, создающая инжектор для этого хука
		 */
		public Builder setInjectorFactory(HookInjectorFactory factory) {
			AsmHook.this.injectorFactory = factory;
			return this;
		}
		
		/**
		 * Задаёт статический модификатор создаваемого метода
		 */
		public Builder setIsStatic(boolean isStatic) {
			AsmHook.this.isStatic = isStatic;
			return this;
		}
		
		/**
		 * Задаёт абстрактный модификатор создаваемого метода
		 */
		public AsmHook.Builder setIsAbstract(boolean isAbstract) {
			AsmHook.this.isAbstract = isAbstract;
			return this;
		}
		
		/**
		 * Задает суперкласс для вызова в создаваемом конструкторе
		 */
		public Builder setSuperClass(String superClass) {
			AsmHook.this.superClass = superClass;
			return this;
		}
		
		/**
		 * Задает приоритет хука.
		 * Хуки с большим приоритетом вызаваются раньше.
		 */
		public Builder setPriority(HookPriority priority) {
			AsmHook.this.priority = priority;
			return this;
		}
		
		/**
		 * Позволяет не только вставлять хуки в существующие методы, но и добавлять новые. Это может понадобиться,
		 * когда нужно переопределить метод суперкласса. Если супер-метод найден, то тело генерируемого метода
		 * представляет собой вызов супер-метода. Иначе это просто пустой метод или return false/0/null в зависимости
		 * от возвращаемого типа.
		 */
		public Builder setCreateMethod(boolean createMethod) {
			AsmHook.this.createMethod = createMethod;
			return this;
		}
		
		/**
		 * Позволяет объявить хук "обязательным" для запуска игры. В случае неудачи во время вставки такого хука
		 * будет не просто выведено сообщение в лог, а крашнется игра.
		 */
		public Builder setMandatory(boolean isMandatory) {
			AsmHook.this.isMandatory = isMandatory;
			return this;
		}
		
		/**
		 * Создает хук по заданным параметрам.
		 *
		 * @return полученный хук
		 *
		 * @throws IllegalStateException если не был вызван какой-либо из обязательных методов
		 */
		public AsmHook build() {
			AsmHook hook = AsmHook.this;
			
			if (hook.createMethod && hook.targetMethodReturnType == null) {
				hook.targetMethodReturnType = hook.hookMethodReturnType;
			}
			hook.targetMethodDescription = getMethodDesc(hook.targetMethodReturnType, hook.targetMethodParameters);
			
			if (hook.hasHookMethod()) {
				hook.hookMethodDescription = Type.getMethodDescriptor(hook.hookMethodReturnType,
					hook.hookMethodParameters.toArray(new Type[0]));
			}
			if (hook.returnValue == ReturnValue.ANOTHER_METHOD_RETURN_VALUE) {
				hook.returnMethodDescription = getMethodDesc(hook.targetMethodReturnType, hook.hookMethodParameters);
			}
			
			try {
				hook = (AsmHook) AsmHook.this.clone();
			} catch (CloneNotSupportedException ignored) {}
			
			if (hook.targetClassName == null) {
				throw new IllegalStateException("Target class name is not specified. " +
													"Call setTargetClassName() before build().");
			}
			
			if (hook.targetMethodName == null) {
				throw new IllegalStateException("Target method name is not specified. " +
													"Call setTargetMethodName() before build().");
			}
			
			if (hook.returnValue == ReturnValue.PRIMITIVE_CONSTANT && hook.primitiveConstant == null) {
				throw new IllegalStateException("Return value is PRIMITIVE_CONSTANT, but the constant is not " +
													"specified. Call setReturnValue() before build().");
			}
			
			if (hook.returnValue == ReturnValue.ANOTHER_METHOD_RETURN_VALUE && hook.returnMethodName == null) {
				throw new IllegalStateException("Return value is ANOTHER_METHOD_RETURN_VALUE, but the method is not " +
													"specified. Call setReturnMethod() before build().");
			}
			
			if (!(hook.injectorFactory instanceof MethodExit) && hook.hasReturnValueParameter) {
				throw new IllegalStateException("Can not pass return value to hook method " +
													"because hook location is not return insn.");
			}
			
			return hook;
		}
		
		private String getMethodDesc(Type returnType, List<Type> paramTypes) {
			Type[] paramTypesArray = paramTypes.toArray(new Type[0]);
			if (returnType == null) {
				String voidDesc = Type.getMethodDescriptor(Type.VOID_TYPE, paramTypesArray);
				return voidDesc.substring(0, voidDesc.length() - 1);
			} else {
				return Type.getMethodDescriptor(returnType, paramTypesArray);
			}
		}
	}
}
