package dev.creatormind.respawntimeout;

import dev.creatormind.respawntimeout.commands.*;
import dev.creatormind.respawntimeout.enums.PlayerStatus;
import dev.creatormind.respawntimeout.state.PlayerState;
import dev.creatormind.respawntimeout.state.ServerState;
import dev.creatormind.respawntimeout.utils.TimeFormatter;
import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.fabricmc.fabric.api.entity.event.v1.ServerLivingEntityEvents;
import net.fabricmc.fabric.api.networking.v1.ServerPlayConnectionEvents;
import net.minecraft.class_1934;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Random;
import java.util.concurrent.TimeUnit;


public class RespawnTimeoutMod implements ModInitializer {

    public static final String MOD_ID = "respawn-timeout";
    public static final String MOD_VERSION = "1.2.0";
    public static final Logger LOGGER = LoggerFactory.getLogger(MOD_ID);


    @Override
    public void onInitialize() {
        LOGGER.info("[Respawn Timeout] Initializing");

        // Commands.
        CommandRegistrationCallback.EVENT.register((dispatcher, registryAccess, environment) -> {
            if (!environment.field_25423)
                return;

            SetCommand.register(dispatcher);
            SetRandomCommand.register(dispatcher);
            GetCommand.register(dispatcher);
            ClearCommand.register(dispatcher);
            RespawnCommand.register(dispatcher);
            VersionCommand.register(dispatcher);
        });

        // Automatic respawn verification if needed when joining.
        ServerPlayConnectionEvents.JOIN.register((handler, sender, server) -> {
            final class_3222 player = handler.method_32311();
            final ServerState serverState = ServerState.getServerState(server);
            final PlayerState playerState = ServerState.getPlayerState(player);
            final PlayerStatus playerStatus = getPlayerStatus(player);

            if (playerStatus == PlayerStatus.TIMED_OUT) {
                final long remainingTime = serverState.timeUnit.toMillis(playerState.respawnTimeout) - (System.currentTimeMillis() - playerState.deathTimestamp);

                player.method_7353(class_2561.method_43469(
                    "respawn-timeout.info.self.status",
                    TimeFormatter.format(remainingTime, TimeUnit.MILLISECONDS)
                ), false);
            }
            else if (playerStatus == PlayerStatus.AWAITING_RESPAWN) {
                respawnPlayer(player);

                player.method_7353(class_2561.method_43471("respawn-timeout.info.self.respawned"), false);
            }
        });

        // Timing out process for when a player dies, if there's an applicable timeout.
        ServerLivingEntityEvents.AFTER_DEATH.register((entity, damageSource) -> {
            if (!(entity instanceof class_3222 player))
                return;

            final MinecraftServer server = player.method_5682();

            if (server == null)
                throw new NullPointerException("[Respawn Timeout] Server is null!");

            final ServerState serverState = ServerState.getServerState(server);

            // If there's no timeout there's no need to perform any action.
            if (serverState.respawnTimeout == 0L)
                return;

            final PlayerState playerState = ServerState.getPlayerState(player);

            // A negative timeout indicates that the timeout must be random.
            if (serverState.respawnTimeout < 0L)
                playerState.respawnTimeout = new Random().nextLong((serverState.maxRandomTimeout - serverState.minRandomTimeout) + 1) + serverState.minRandomTimeout;
            else
                playerState.respawnTimeout = serverState.respawnTimeout;

            playerState.deathTimestamp = System.currentTimeMillis();

            serverState.players.put(player.method_5667(), playerState);
            serverState.method_80();

            player.method_7336(class_1934.field_9219);
            player.method_7353(class_2561.method_43469(
                "respawn-timeout.info.self.status",
                TimeFormatter.format(playerState.respawnTimeout, serverState.timeUnit)
            ), false);
        });
    }


    public static PlayerStatus getPlayerStatus(class_3222 player) {
        final MinecraftServer server = player.method_5682();

        if (server == null)
            throw new NullPointerException("[Respawn Timeout] Server is null!");

        final ServerState serverState = ServerState.getServerState(server);
        final PlayerState playerState = ServerState.getPlayerState(player);

        // The player must be on spectator mode (timed out) and have a registered death time.
        if (playerState.deathTimestamp == 0L || !player.method_7325())
            return PlayerStatus.ALIVE;

        final long respawnTimeoutInSeconds = serverState.timeUnit.toSeconds(playerState.respawnTimeout);
        final long timeSinceDeathInSeconds = (System.currentTimeMillis() - playerState.deathTimestamp) / 1000L;

        // Time elapsed since death should be less than the defined timeout for a player to be timed out.
        if (timeSinceDeathInSeconds < respawnTimeoutInSeconds)
            return PlayerStatus.TIMED_OUT;

        return PlayerStatus.AWAITING_RESPAWN;
    }

    public static void respawnPlayer(class_3222 player) {
        final MinecraftServer server = player.method_5682();

        if (server == null)
            throw new NullPointerException("[Respawn Timeout] Server is null!");

        final ServerState serverState = ServerState.getServerState(server);
        final PlayerState playerState = ServerState.getPlayerState(player);

        class_3218 spawnWorld = server.method_3847(player.method_26281());
        class_2338 spawnPosition = player.method_26280();

        // Use the player's spawn point if it's valid, otherwise use world's spawn.
        if (spawnWorld == null || spawnPosition == null) {
            spawnWorld = server.method_30002();
            spawnPosition = spawnWorld.method_43126();
        }

        final int x = spawnPosition.method_10263();
        final int y = spawnPosition.method_10264();
        final int z = spawnPosition.method_10260();

        player.method_48105(
            spawnWorld,
            x,
            y,
            z,
            new HashSet<>(),
            player.method_36454(),
            player.method_36455(),
            true
        );

        player.method_7336(server.method_3790());

        // Resets the timeout status.
        playerState.deathTimestamp = 0L;

        serverState.players.put(player.method_5667(), playerState);
        serverState.method_80();
    }

}
