/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.deepspace.planet;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.mojang.logging.LogUtils;
import java.awt.Color;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.core.registries.Registries;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.Vec3;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import net.neoforged.neoforge.event.server.ServerStoppedEvent;
import net.neoforged.neoforge.network.PacketDistributor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import world.landfall.deepspace.network.PlanetSyncPacket;
import world.landfall.deepspace.planet.Planet;
import world.landfall.deepspace.planet.Sun;

@EventBusSubscriber(modid="deepspace")
public class PlanetRegistry {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String PLANETS_CONFIG_FILE = "planets.json";
    private static final Map<String, Planet> planets = new ConcurrentHashMap<String, Planet>();
    private static volatile Sun sun;
    private static final Map<ResourceKey<Level>, Planet> planetsByDimension;
    private static final Object registryLock;
    private static volatile Path configPath;

    public static void initialize(@NotNull Path configDir) {
        Objects.requireNonNull(configDir, "Config directory cannot be null");
        configPath = configDir.resolve(PLANETS_CONFIG_FILE);
        LOGGER.info("Planet registry initialized with config path: {}", (Object)configPath);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void loadPlanets() {
        Path currentConfigPath = configPath;
        if (currentConfigPath == null) {
            LOGGER.error("Planet registry not initialized - cannot load planets");
            return;
        }
        Object object = registryLock;
        synchronized (object) {
            try {
                if (!Files.exists(currentConfigPath, new LinkOption[0])) {
                    LOGGER.info("No planets configuration file found, creating default configuration");
                    PlanetRegistry.createDefaultConfigurationUnsafe();
                    return;
                }
                String json = Files.readString(currentConfigPath);
                PlanetsConfig config = (PlanetsConfig)GSON.fromJson(json, PlanetsConfig.class);
                if (config == null || config.planets == null) {
                    LOGGER.warn("Invalid planets configuration file, creating default configuration");
                    PlanetRegistry.createDefaultConfigurationUnsafe();
                    return;
                }
                planets.clear();
                planetsByDimension.clear();
                for (PlanetConfig planetConfig : config.planets) {
                    try {
                        Planet planet = PlanetRegistry.createPlanetFromConfig(planetConfig);
                        PlanetRegistry.registerPlanetUnsafe(planet);
                        LOGGER.debug("Loaded planet: {}", (Object)planet.getId());
                    }
                    catch (Exception e) {
                        LOGGER.error("Failed to load planet configuration: {}", (Object)planetConfig.id, (Object)e);
                    }
                }
                LOGGER.info("Loaded {} planets from configuration", (Object)planets.size());
                try {
                    sun = PlanetRegistry.createSunFromConfig(config.sun);
                }
                catch (Exception e) {
                    LOGGER.error("Failed to load sun configuration: ", (Throwable)e);
                }
            }
            catch (IOException e) {
                LOGGER.error("Failed to read planets configuration file", (Throwable)e);
                PlanetRegistry.createDefaultConfigurationUnsafe();
            }
            catch (JsonSyntaxException e) {
                LOGGER.error("Invalid JSON in planets configuration file", (Throwable)e);
                PlanetRegistry.createDefaultConfigurationUnsafe();
            }
        }
    }

    private static void createDefaultConfigurationUnsafe() {
        planets.clear();
        planetsByDimension.clear();
        try {
            Planet overworld = new Planet("overworld", "Overworld", (ResourceKey<Level>)Level.OVERWORLD, new Vec3(-100.0, -100.0, -100.0), new Vec3(100.0, 100.0, 100.0), List.of(new Planet.PlanetDecoration("atmosphere", 1.05f, Color.WHITE.getRGB())), "The main world where players spawn");
            PlanetRegistry.registerPlanetUnsafe(overworld);
            Planet nether = new Planet("nether", "The Nether", (ResourceKey<Level>)Level.NETHER, new Vec3(200.0, -50.0, -100.0), new Vec3(400.0, 150.0, 100.0), List.of(new Planet.PlanetDecoration("atmosphere", 1.0f, Color.RED.getRGB()), new Planet.PlanetDecoration("rings", 1.2f, Color.RED.getRGB())), "A hellish dimension filled with lava and dangerous creatures");
            PlanetRegistry.registerPlanetUnsafe(nether);
            Planet end = new Planet("end", "The End", (ResourceKey<Level>)Level.END, new Vec3(-200.0, 100.0, -100.0), new Vec3(0.0, 300.0, 100.0), List.of(new Planet.PlanetDecoration("atmosphere", 1.0f, Color.WHITE.getRGB())), "The final dimension, home to the Ender Dragon");
            PlanetRegistry.registerPlanetUnsafe(end);
            sun = new Sun(new Vec3(-500.0, 0.0, 0.0), new Vec3(-300.0, 200.0, 200.0), 1500.0);
            LOGGER.info("Created default planet configuration with {} planets", (Object)planets.size());
        }
        catch (Exception e) {
            LOGGER.error("Failed to create default planet configuration", (Throwable)e);
        }
    }

    @NotNull
    private static Sun createSunFromConfig(@NotNull SunConfig config) {
        Objects.requireNonNull(config, "Sun config cannot be null");
        if (config.boundingBoxMin == null || config.boundingBoxMin.length != 3) {
            throw new IllegalArgumentException("Invalid bounding box minimum coordinates");
        }
        if (config.boundingBoxMax == null || config.boundingBoxMax.length != 3) {
            throw new IllegalArgumentException("Invalid bounding box maximum coordinates");
        }
        return new Sun(new Vec3(config.boundingBoxMin[0], config.boundingBoxMin[1], config.boundingBoxMin[2]), new Vec3(config.boundingBoxMax[0], config.boundingBoxMax[1], config.boundingBoxMax[2]), config.hurtRadius);
    }

    @NotNull
    private static Planet createPlanetFromConfig(@NotNull PlanetConfig config) {
        Objects.requireNonNull(config, "Planet config cannot be null");
        if (config.id == null || config.id.trim().isEmpty()) {
            throw new IllegalArgumentException("Planet ID cannot be null or empty");
        }
        if (config.name == null || config.name.trim().isEmpty()) {
            throw new IllegalArgumentException("Planet name cannot be null or empty");
        }
        if (config.dimension == null || config.dimension.trim().isEmpty()) {
            throw new IllegalArgumentException("Planet dimension cannot be null or empty");
        }
        if (config.boundingBoxMin == null || config.boundingBoxMin.length != 3) {
            throw new IllegalArgumentException("Invalid bounding box minimum coordinates");
        }
        if (config.boundingBoxMax == null || config.boundingBoxMax.length != 3) {
            throw new IllegalArgumentException("Invalid bounding box maximum coordinates");
        }
        ResourceKey dimension = ResourceKey.create((ResourceKey)Registries.DIMENSION, (ResourceLocation)ResourceLocation.parse((String)config.dimension));
        Vec3 boundingBoxMin = new Vec3(config.boundingBoxMin[0], config.boundingBoxMin[1], config.boundingBoxMin[2]);
        Vec3 boundingBoxMax = new Vec3(config.boundingBoxMax[0], config.boundingBoxMax[1], config.boundingBoxMax[2]);
        return new Planet(config.id, config.name, (ResourceKey<Level>)dimension, boundingBoxMin, boundingBoxMax, config.decorations, config.description);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void registerPlanet(@NotNull Planet planet) {
        Objects.requireNonNull(planet, "Planet cannot be null");
        Object object = registryLock;
        synchronized (object) {
            PlanetRegistry.registerPlanetUnsafe(planet);
        }
    }

    private static void registerPlanetUnsafe(@NotNull Planet planet) {
        planets.put(planet.getId(), planet);
        planetsByDimension.put(planet.getDimension(), planet);
        LOGGER.debug("Registered planet: {} for dimension: {}", (Object)planet.getId(), (Object)planet.getDimension().location());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public static Planet unregisterPlanet(@NotNull String planetId) {
        Objects.requireNonNull(planetId, "Planet ID cannot be null");
        Object object = registryLock;
        synchronized (object) {
            Planet planet = planets.remove(planetId);
            if (planet != null) {
                planetsByDimension.remove(planet.getDimension());
                LOGGER.debug("Unregistered planet: {}", (Object)planetId);
            }
            return planet;
        }
    }

    @Nullable
    public static Planet getPlanet(@NotNull String planetId) {
        Objects.requireNonNull(planetId, "Planet ID cannot be null");
        return planets.get(planetId);
    }

    @Nullable
    public static Sun getSun() {
        return sun;
    }

    public static boolean setSun(Sun _sun) {
        sun = _sun;
        return true;
    }

    @Nullable
    public static Planet getPlanetByDimension(@NotNull ResourceKey<Level> dimension) {
        Objects.requireNonNull(dimension, "Dimension cannot be null");
        return planetsByDimension.get(dimension);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public static Collection<Planet> getAllPlanets() {
        Object object = registryLock;
        synchronized (object) {
            return Collections.unmodifiableCollection(new ArrayList<Planet>(planets.values()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public static List<Planet> getPlanetsAtPosition(@NotNull Vec3 position) {
        Objects.requireNonNull(position, "Position cannot be null");
        Object object = registryLock;
        synchronized (object) {
            return planets.values().stream().filter(planet -> planet.isWithinBounds(position)).toList();
        }
    }

    public static boolean hasPlanet(@NotNull String planetId) {
        Objects.requireNonNull(planetId, "Planet ID cannot be null");
        return planets.containsKey(planetId);
    }

    public static int getPlanetCount() {
        return planets.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void clear() {
        Object object = registryLock;
        synchronized (object) {
            planets.clear();
            planetsByDimension.clear();
            sun = null;
            LOGGER.debug("Cleared all planets from registry");
        }
    }

    public static void init() {
        Path serverConfigDir = Paths.get("config", new String[0]);
        PlanetRegistry.initialize(serverConfigDir);
        PlanetRegistry.loadPlanets();
        LOGGER.info("Planet registry loaded on server start");
    }

    @SubscribeEvent
    public static void onServerStopped(ServerStoppedEvent event) {
        LOGGER.info("Planet registry server stopped");
    }

    @SubscribeEvent
    public static void onPlayerLoggedIn(PlayerEvent.PlayerLoggedInEvent event) {
        Player player;
        if (!event.getEntity().level().isClientSide && (player = event.getEntity()) instanceof ServerPlayer) {
            ServerPlayer serverPlayer = (ServerPlayer)player;
            PlanetSyncPacket syncPacket = PlanetSyncPacket.createSyncPacket();
            PacketDistributor.sendToPlayer((ServerPlayer)serverPlayer, (CustomPacketPayload)syncPacket, (CustomPacketPayload[])new CustomPacketPayload[0]);
            LOGGER.debug("Synchronized planets to player: {}", (Object)serverPlayer.getName().getString());
        }
    }

    public static void syncToAllPlayers() {
        PlanetSyncPacket syncPacket = PlanetSyncPacket.createSyncPacket();
        PacketDistributor.sendToAllPlayers((CustomPacketPayload)syncPacket, (CustomPacketPayload[])new CustomPacketPayload[0]);
        LOGGER.info("Synchronized planets to all players");
    }

    static {
        planetsByDimension = new ConcurrentHashMap<ResourceKey<Level>, Planet>();
        registryLock = new Object();
    }

    public static class PlanetsConfig {
        public List<PlanetConfig> planets = new ArrayList<PlanetConfig>();
        public SunConfig sun;

        public PlanetsConfig() {
        }

        public PlanetsConfig(Collection<Planet> planets, Sun sun) {
            this.planets = planets.stream().map(PlanetConfig::new).toList();
            this.sun = new SunConfig(sun);
        }
    }

    public static class PlanetConfig {
        public String id;
        public String name;
        public String dimension;
        public double[] boundingBoxMin;
        public double[] boundingBoxMax;
        public Collection<Planet.PlanetDecoration> decorations;
        public String description = "";

        public PlanetConfig() {
        }

        public PlanetConfig(Planet planet) {
            this.id = planet.getId();
            this.name = planet.getName();
            this.dimension = planet.getDimension().location().toString();
            this.boundingBoxMin = new double[]{planet.getBoundingBoxMin().x, planet.getBoundingBoxMin().y, planet.getBoundingBoxMin().z};
            this.boundingBoxMax = new double[]{planet.getBoundingBoxMax().x, planet.getBoundingBoxMax().y, planet.getBoundingBoxMax().z};
            this.decorations = planet.getDecorations().isPresent() ? (Collection<Object>)planet.getDecorations().get() : List.of();
            this.description = planet.getDescription();
        }
    }

    public static class SunConfig {
        public double[] boundingBoxMin;
        public double[] boundingBoxMax;
        public double hurtRadius;

        public SunConfig(Sun sun) {
            Vec3 boundingBoxMin = sun.getBoundingBoxMin();
            this.boundingBoxMin = new double[]{boundingBoxMin.x, boundingBoxMin.y, boundingBoxMin.z};
            Vec3 boundingBoxMax = sun.getBoundingBoxMax();
            this.boundingBoxMax = new double[]{boundingBoxMax.x, boundingBoxMax.y, boundingBoxMax.z};
            this.hurtRadius = sun.getHurtRadius();
        }
    }
}

