/*
 * Decompiled with CFR 0.152.
 */
package org.bxteam.quark;

import java.io.InputStream;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Pattern;
import org.bxteam.quark.classloader.IsolatedClassLoader;
import org.bxteam.quark.dependency.Dependency;
import org.bxteam.quark.dependency.DependencyResolver;
import org.bxteam.quark.logger.LogAdapter;
import org.bxteam.quark.logger.LogLevel;
import org.bxteam.quark.logger.Logger;
import org.bxteam.quark.relocation.Relocation;
import org.bxteam.quark.relocation.RelocationHandler;
import org.bxteam.quark.repository.LocalRepository;
import org.bxteam.quark.repository.Repository;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class LibraryManager
implements AutoCloseable {
    private static final String DEFAULT_LIBS_DIRECTORY = "libs";
    protected final Logger logger;
    protected final Path dataDirectory;
    protected final LocalRepository localRepository;
    protected final Set<Repository> globalRepositories = ConcurrentHashMap.newKeySet();
    private final AtomicBoolean mavenCentralWarningShown = new AtomicBoolean(false);
    private boolean includeDependencyManagement = false;
    private int maxTransitiveDepth = Integer.MAX_VALUE;
    private final Set<String> excludedGroupIds = ConcurrentHashMap.newKeySet();
    private final List<Pattern> excludedArtifactPatterns = Collections.synchronizedList(new ArrayList());
    private boolean skipOptionalDependencies = true;
    private boolean skipTestDependencies = true;
    protected volatile DependencyResolver dependencyResolver;
    private volatile RelocationHandler relocationHandler;
    protected final IsolatedClassLoader globalIsolatedClassLoader;
    protected final Map<String, IsolatedClassLoader> isolatedClassLoaders = new ConcurrentHashMap<String, IsolatedClassLoader>();
    protected final Map<Dependency, Path> loadedDependencies = new ConcurrentHashMap<Dependency, Path>();

    protected LibraryManager(@NotNull LogAdapter logAdapter, @NotNull Path dataDirectory) {
        this(logAdapter, dataDirectory, DEFAULT_LIBS_DIRECTORY);
    }

    protected LibraryManager(@NotNull LogAdapter logAdapter, @NotNull Path dataDirectory, @NotNull String librariesDirectoryName) {
        this.logger = new Logger(Objects.requireNonNull(logAdapter, "Log adapter cannot be null"));
        this.dataDirectory = Objects.requireNonNull(dataDirectory, "Data directory cannot be null").toAbsolutePath();
        Path libsPath = this.dataDirectory.resolve(Objects.requireNonNull(librariesDirectoryName, "Libraries directory name cannot be null"));
        this.localRepository = new LocalRepository(libsPath);
        this.localRepository.ensureExists();
        this.updateDependencyResolver();
        this.globalIsolatedClassLoader = this.createIsolatedClassLoader();
        this.logger.debug("Initialized LibraryManager with data directory: " + String.valueOf(dataDirectory));
    }

    private void updateDependencyResolver() {
        ArrayList<Repository> allRepositories = new ArrayList<Repository>();
        allRepositories.add(this.localRepository);
        allRepositories.addAll(this.globalRepositories);
        this.dependencyResolver = new DependencyResolver.Builder(this.logger, allRepositories, this.localRepository.getPath()).includeDependencyManagement(this.includeDependencyManagement).maxTransitiveDepth(this.maxTransitiveDepth).skipOptionalDependencies(this.skipOptionalDependencies).skipTestDependencies(this.skipTestDependencies).excludeGroupIds(this.excludedGroupIds.toArray(new String[0])).build();
    }

    protected abstract void addToClasspath(@NotNull Path var1);

    @NotNull
    protected abstract IsolatedClassLoader createIsolatedClassLoader();

    @Nullable
    protected abstract InputStream getResourceAsStream(@NotNull String var1);

    public void addRepository(@NotNull String repositoryUrl) {
        Objects.requireNonNull(repositoryUrl, "Repository URL cannot be null");
        if (this.isMavenCentralUrl(repositoryUrl) && this.mavenCentralWarningShown.compareAndSet(false, true)) {
            this.showMavenCentralWarning();
        }
        Repository repository = Repository.of(repositoryUrl);
        this.globalRepositories.add(repository);
        this.updateDependencyResolver();
        this.logger.debug("Added repository: " + repositoryUrl);
    }

    private boolean isMavenCentralUrl(@NotNull String url) {
        return url.equals("https://repo1.maven.org/maven2") || url.equals("https://repo1.maven.org/maven2/");
    }

    private void showMavenCentralWarning() {
        RuntimeException stackTrace = new RuntimeException("Plugin used Maven Central for library resolution");
        this.logger.warn("Use of Maven Central as a CDN is against the Maven Central Terms of Service. Use Repositories.GOOGLE_MAVEN_CENTRAL_MIRROR or LibraryManager#addGoogleMavenCentralMirror() instead.", stackTrace);
    }

    @Deprecated
    public void addMavenCentral() {
        this.addRepository("https://repo1.maven.org/maven2");
    }

    public void addGoogleMavenCentralMirror() {
        this.addRepository("https://maven-central.storage-download.googleapis.com/maven2/");
    }

    public void addSonatype() {
        this.addRepository("https://oss.sonatype.org/content/groups/public/");
    }

    public void addJitPack() {
        this.addRepository("https://jitpack.io/");
    }

    public void addMavenLocal() {
        this.addRepository(LocalRepository.mavenLocal().getUrl());
    }

    @NotNull
    public Collection<Repository> getRepositories() {
        return Collections.unmodifiableSet(this.globalRepositories);
    }

    public LibraryManager includeDependencyManagement(boolean include) {
        this.includeDependencyManagement = include;
        this.updateDependencyResolver();
        return this;
    }

    public LibraryManager maxTransitiveDepth(int depth) {
        this.maxTransitiveDepth = Math.max(0, depth);
        this.updateDependencyResolver();
        return this;
    }

    public LibraryManager excludeGroupIds(String ... groupIds) {
        if (groupIds != null) {
            this.excludedGroupIds.addAll(Arrays.asList(groupIds));
            this.updateDependencyResolver();
        }
        return this;
    }

    public LibraryManager excludeArtifacts(String ... patterns) {
        if (patterns != null) {
            for (String pattern : patterns) {
                String regex = pattern.replace(".", "\\.").replace("*", ".*");
                this.excludedArtifactPatterns.add(Pattern.compile(regex));
            }
            this.updateDependencyResolver();
        }
        return this;
    }

    public LibraryManager skipOptionalDependencies(boolean skip) {
        this.skipOptionalDependencies = skip;
        this.updateDependencyResolver();
        return this;
    }

    public LibraryManager skipTestDependencies(boolean skip) {
        this.skipTestDependencies = skip;
        this.updateDependencyResolver();
        return this;
    }

    public LibraryManager optimizeDependencyDownloads() {
        return this.maxTransitiveDepth(3).skipOptionalDependencies(true).skipTestDependencies(true).excludeGroupIds("javax.servlet");
    }

    public void loadDependency(@NotNull Dependency dependency) {
        Objects.requireNonNull(dependency, "Dependency cannot be null");
        this.loadDependencies(Collections.singletonList(dependency));
    }

    public void loadDependencies(@NotNull List<Dependency> dependencies) {
        this.loadDependencies(dependencies, Collections.emptyList());
    }

    public void loadDependencies(@NotNull List<Dependency> dependencies, @NotNull List<Relocation> relocations) {
        Objects.requireNonNull(dependencies, "Dependencies cannot be null");
        Objects.requireNonNull(relocations, "Relocations cannot be null");
        if (dependencies.isEmpty()) {
            return;
        }
        Instant startTime = Instant.now();
        try {
            DependencyResolver.ResolutionResult result = this.dependencyResolver.resolveDependencies(dependencies);
            if (result.hasErrors()) {
                this.logger.warn("Dependency resolution completed with " + result.errors().size() + " errors");
                if (this.logger.getLevel() == LogLevel.DEBUG) {
                    result.errors().forEach(error -> this.logger.debug("Resolution error: " + error));
                }
            }
            List<DependencyLoadEntry> loadEntries = this.applyRelocations(result.resolvedDependencies(), relocations);
            for (DependencyLoadEntry entry : loadEntries) {
                this.addToClasspath(entry.path());
                this.loadedDependencies.put(entry.dependency(), entry.path());
            }
            Duration elapsed = Duration.between(startTime, Instant.now());
            this.logger.info("Loaded " + loadEntries.size() + " dependencies in " + elapsed.toMillis() + " ms");
        }
        catch (Exception e) {
            Duration elapsed = Duration.between(startTime, Instant.now());
            this.logger.error("Failed to load dependencies after " + elapsed.toMillis() + " ms: " + e.getMessage());
            throw new LibraryLoadException("Failed to load dependencies", e);
        }
    }

    public void loadDependencies(@NotNull IsolatedClassLoader classLoader, @NotNull List<Dependency> dependencies, @NotNull List<Relocation> relocations) {
        Objects.requireNonNull(classLoader, "Class loader cannot be null");
        Objects.requireNonNull(dependencies, "Dependencies cannot be null");
        Objects.requireNonNull(relocations, "Relocations cannot be null");
        if (dependencies.isEmpty()) {
            return;
        }
        Instant startTime = Instant.now();
        try {
            this.logger.info("Resolving " + dependencies.size() + " dependencies for isolated class loader...");
            DependencyResolver.ResolutionResult result = this.dependencyResolver.resolveDependencies(dependencies);
            if (result.hasErrors()) {
                this.logger.warn("Dependency resolution completed with " + result.errors().size() + " errors:");
                result.errors().forEach(error -> this.logger.warn("  - " + error));
            }
            this.logger.info("Resolved " + result.getDependencyCount() + " total dependencies");
            List<DependencyLoadEntry> loadEntries = this.applyRelocations(result.resolvedDependencies(), relocations);
            for (DependencyLoadEntry entry : loadEntries) {
                classLoader.addPath(entry.path());
                this.loadedDependencies.put(entry.dependency(), entry.path());
            }
            Duration elapsed = Duration.between(startTime, Instant.now());
            this.logger.info("Loaded " + loadEntries.size() + " dependencies into isolated class loader in " + elapsed.toMillis() + " ms");
        }
        catch (Exception e) {
            Duration elapsed = Duration.between(startTime, Instant.now());
            this.logger.error("Failed to load dependencies into isolated class loader after " + elapsed.toMillis() + " ms: " + e.getMessage());
            throw new LibraryLoadException("Failed to load dependencies", e);
        }
    }

    @NotNull
    private List<DependencyLoadEntry> applyRelocations(@NotNull List<DependencyResolver.ResolvedDependency> resolvedDependencies, @NotNull List<Relocation> relocations) {
        ArrayList<DependencyLoadEntry> loadEntries = new ArrayList<DependencyLoadEntry>();
        for (DependencyResolver.ResolvedDependency resolved : resolvedDependencies) {
            Dependency dependency = resolved.dependency();
            Path jarPath = resolved.jarPath();
            Path existingPath = this.loadedDependencies.get(dependency);
            if (existingPath != null) {
                this.logger.debug("Using cached dependency: " + dependency.toShortString());
                loadEntries.add(new DependencyLoadEntry(dependency, existingPath));
                continue;
            }
            Path finalPath = this.applyRelocations(dependency, jarPath, relocations);
            loadEntries.add(new DependencyLoadEntry(dependency, finalPath));
        }
        return loadEntries;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    private Path applyRelocations(@NotNull Dependency dependency, @NotNull Path jarPath, @NotNull List<Relocation> relocations) {
        if (relocations.isEmpty()) {
            return jarPath;
        }
        if (this.relocationHandler == null) {
            LibraryManager libraryManager = this;
            synchronized (libraryManager) {
                if (this.relocationHandler == null) {
                    this.relocationHandler = RelocationHandler.create(this);
                }
            }
        }
        return this.relocationHandler.relocateDependency(this.localRepository, jarPath, dependency, relocations);
    }

    public void loadDependency(@NotNull String coordinates) {
        Objects.requireNonNull(coordinates, "Coordinates cannot be null");
        this.loadDependency(Dependency.fromCoordinates(coordinates));
    }

    public void loadDependency(@NotNull String groupId, @NotNull String artifactId, @NotNull String version) {
        Objects.requireNonNull(groupId, "Group ID cannot be null");
        Objects.requireNonNull(artifactId, "Artifact ID cannot be null");
        Objects.requireNonNull(version, "Version cannot be null");
        this.loadDependency(Dependency.of(groupId, artifactId, version));
    }

    @NotNull
    public IsolatedClassLoader createNamedIsolatedClassLoader(@NotNull String loaderId) {
        Objects.requireNonNull(loaderId, "Loader ID cannot be null");
        IsolatedClassLoader classLoader = this.createIsolatedClassLoader();
        this.isolatedClassLoaders.put(loaderId, classLoader);
        this.logger.debug("Created isolated class loader: " + loaderId);
        return classLoader;
    }

    public void loadDependenciesIsolated(@NotNull String loaderId, @NotNull List<Dependency> dependencies, @NotNull List<Relocation> relocations) {
        IsolatedClassLoader classLoader = this.isolatedClassLoaders.get(loaderId);
        if (classLoader == null) {
            classLoader = this.createNamedIsolatedClassLoader(loaderId);
        }
        this.loadDependencies(classLoader, dependencies, relocations);
    }

    @NotNull
    public Logger getLogger() {
        return this.logger;
    }

    @NotNull
    public LocalRepository getLocalRepository() {
        return this.localRepository;
    }

    @NotNull
    public LogLevel getLogLevel() {
        return this.logger.getLevel();
    }

    public void setLogLevel(@NotNull LogLevel level) {
        this.logger.setLevel(Objects.requireNonNull(level, "Log level cannot be null"));
    }

    @NotNull
    public IsolatedClassLoader getGlobalIsolatedClassLoader() {
        return this.globalIsolatedClassLoader;
    }

    @Nullable
    public IsolatedClassLoader getIsolatedClassLoader(@NotNull String loaderId) {
        return this.isolatedClassLoaders.get(Objects.requireNonNull(loaderId, "Loader ID cannot be null"));
    }

    @NotNull
    public Map<Dependency, Path> getLoadedDependencies() {
        return Collections.unmodifiableMap(this.loadedDependencies);
    }

    public boolean isDependencyLoaded(@NotNull Dependency dependency) {
        return this.loadedDependencies.containsKey(Objects.requireNonNull(dependency, "Dependency cannot be null"));
    }

    @NotNull
    public LibraryManagerStats getStats() {
        return new LibraryManagerStats(this.globalRepositories.size(), this.loadedDependencies.size(), this.isolatedClassLoaders.size());
    }

    @Override
    public void close() {
        this.logger.debug("Shutting down LibraryManager...");
        try {
            if (this.relocationHandler != null) {
                this.relocationHandler.close();
            }
            this.globalIsolatedClassLoader.close();
            this.isolatedClassLoaders.values().forEach(IsolatedClassLoader::close);
            this.isolatedClassLoaders.clear();
        }
        catch (Exception e) {
            this.logger.error("Error during LibraryManager shutdown", e);
        }
        this.logger.debug("LibraryManager shutdown complete");
    }

    public record DependencyLoadEntry(@NotNull Dependency dependency, @NotNull Path path) {
        public DependencyLoadEntry(@NotNull Dependency dependency, @NotNull Path path) {
            Objects.requireNonNull(dependency, "Dependency cannot be null");
            Objects.requireNonNull(path, "Path cannot be null");
        }
    }

    public static class LibraryLoadException
    extends RuntimeException {
        public LibraryLoadException(String message) {
            super(message);
        }

        public LibraryLoadException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public record LibraryManagerStats(int repositoryCount, int loadedDependencyCount, int isolatedClassLoaderCount) {
        @Override
        public String toString() {
            return String.format("LibraryManagerStats{repositories=%d, loadedDependencies=%d, isolatedClassLoaders=%d}", this.repositoryCount, this.loadedDependencyCount, this.isolatedClassLoaderCount);
        }
    }
}

