package mods.thecomputerizer.theimpossiblelibrary.api.common.entity;

import lombok.Getter;
import mods.thecomputerizer.theimpossiblelibrary.api.common.effect.EffectInstanceAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.core.Hacks;
import mods.thecomputerizer.theimpossiblelibrary.api.core.TILRef;
import mods.thecomputerizer.theimpossiblelibrary.api.core.annotation.IndirectCallers;
import mods.thecomputerizer.theimpossiblelibrary.api.registry.RegistryAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.registry.RegistryEntryAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.resource.ResourceLocationAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.shapes.Box;
import mods.thecomputerizer.theimpossiblelibrary.api.shapes.vectors.Vector3;
import mods.thecomputerizer.theimpossiblelibrary.api.tag.CompoundTagAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.util.GenericUtils;
import mods.thecomputerizer.theimpossiblelibrary.api.world.BlockPosAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.world.DimensionAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.world.PosHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.world.WorldAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.wrappers.AbstractWrapped;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;

import static java.lang.Double.MAX_VALUE;
import static mods.thecomputerizer.theimpossiblelibrary.api.wrappers.WrapperHelper.WrapperType.EFFECT_INSTANCE;

@Getter
public abstract class EntityAPI<E,V> extends AbstractWrapped<V> implements RegistryEntryAPI<V> {
    
    protected final Function<Object,Collection<?>> effectsGetter;
    
    protected ResourceLocationAPI<?> registryName;

    protected E entity;
    private boolean erroredEffects;
    
    protected EntityAPI(Object entity, Object type) {
        this(entity,type,Hacks::invokeMethodObj);
    }

    protected EntityAPI(Object entity, Object type, BiFunction<E,String,Collection<?>> effectsGetter) {
        super(type);
        this.entity = GenericUtils.cast(entity);
        this.effectsGetter = e -> effectsGetter.apply(GenericUtils.cast(e),getActiveEffectsMethodName());
    }
    
    @IndirectCallers public abstract boolean canTarget();
    
    @Override public boolean equals(Object other) {
        if(super.equals(other) && other instanceof EntityAPI<?,?>) {
            Object entity = getEntity();
            Object otherEntity = ((EntityAPI<?,?>)other).getEntity();
            return Objects.isNull(otherEntity) ? Objects.isNull(entity) : entity.equals(otherEntity);
        }
        return false;
    }

    @IndirectCallers
    public Collection<EffectInstanceAPI<?>> getActiveEffects() {
        if(this.erroredEffects || Objects.isNull(this.entity) || !isLiving()) return Collections.emptyList();
        try {
            return EFFECT_INSTANCE.wrapCollectionGetter(this.entity,this.effectsGetter);
        } catch(Throwable t) {
            TILRef.logError("Failed to retrieve active effects for entity! This check will be disable for the "+
                            "current entity",t);
            this.erroredEffects = true;
        }
        return Collections.emptyList();
    }
    
    protected String getActiveEffectsMethodName() {
        return "getActiveEffects";
    }
    
    @IndirectCallers public abstract EntityAPI<?,?> getAttackTarget();
    public abstract Box getBoundingBox();
    public abstract CompoundTagAPI<?> getData();
    public abstract DimensionAPI<?> getDimension();

    @IndirectCallers
    public double getDistanceTo(EntityAPI<?,?> entity) {
        return Objects.nonNull(entity) ? getPos().distanceTo(entity.getPos()) : MAX_VALUE;
    }
    
    @IndirectCallers
    public double getDistanceTo(BlockPosAPI<?> pos) {
        return Objects.nonNull(pos) ? getPos().distanceTo(pos) : MAX_VALUE;
    }
    
    @IndirectCallers
    public double getDistanceTo(Vector3 pos) {
        return Objects.nonNull(pos) ? getPos().distanceTo(pos) : MAX_VALUE;
    }

    public abstract String getName();
    public abstract BlockPosAPI<?> getPos();

    public Vector3 getPosExact() {
        return new Vector3(x(),y(),z());
    }
    
    /**
     * Rounds to the CENTER of the block which is the relative position (0.5,0.5,0.5)
     */
    @IndirectCallers public BlockPosAPI<?> getPosRounded() {
        double x = PosHelper.roundToCenter(x());
        double y = PosHelper.roundToCenter(y());
        double z = PosHelper.roundToCenter(z());
        return PosHelper.getPos(x,y,z);
    }
    
    @Override public ResourceLocationAPI<?> getRegistryName() {
        if(Objects.isNull(this.registryName) && Objects.nonNull(this.wrapped)) {
            RegistryAPI<?> registry = getRegistry();
            if(Objects.nonNull(registry)) this.registryName = registry.getKey(unwrap());
        }
        return this.registryName;
    }
    
    public abstract EntityAPI<?,?> getRootVehicle();
    public abstract @Nullable EntityAPI<?,?> getVehicle();
    public abstract WorldAPI<?> getWorld();
    @IndirectCallers public abstract boolean isAlive();
    @IndirectCallers public abstract boolean isAnimal();
    @IndirectCallers public abstract boolean isHostile();
    public abstract boolean isLiving();
    public abstract boolean isPlayer();
    @IndirectCallers public abstract boolean isOwnedBy(EntityAPI<?,?> owner);
    
    protected void setLocalRegistryName(ResourceLocationAPI<?> registryName) {
        this.registryName = registryName;
    }
    
    public void setPosition(BlockPosAPI<?> pos) {
        setPosition(pos.x(),pos.y(),pos.z());
    }
    
    public void setPosition(Vector3 vec) {
        setPosition(vec.dX(),vec.dY(),vec.dZ());
    }
    
    public void setPosition(int x, int y, int z) {
        setPosition((double)x,y,z);
    }
    
    public abstract void setPosition(double x, double y, double z);
    
    public <T> T unwrapEntity() {
        return GenericUtils.cast(getEntity());
    }
    
    public abstract double x();
    public abstract double y();
    public abstract double z();
}
