package mods.thecomputerizer.theimpossiblelibrary.api.core;

import lombok.Setter;
import mods.thecomputerizer.theimpossiblelibrary.api.client.ClientAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.client.SharedHandlesClient;
import mods.thecomputerizer.theimpossiblelibrary.api.common.SharedHandlesCommon;
import mods.thecomputerizer.theimpossiblelibrary.api.core.annotation.IndirectCallers;
import mods.thecomputerizer.theimpossiblelibrary.api.resource.ResourceLocationAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.common.CommonAPI;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

import static org.apache.logging.log4j.Level.*;

/**
 * Base reference API for global constants specific to this library
 */
public class TILRef {
    
    public static final String DATA_DIRECTORY = "impossible_data";
    public static final Logger LOGGER = createLogger("The Impossible Library");
    public static final String BASE_PACKAGE = "mods.thecomputerizer.theimpossiblelibrary";
    public static final String DEFAULT_LICENSE = "LGPL V3"; //Default for this library or when the license it not set
    public static final String DESCRIPTION = "Multiversion API & mod loader with helpers to do things deemed impossible";
    public static final String MODID = "theimpossiblelibrary";
    public static final String NAME = "The Impossible Library";
    public static final String PROVIDERID = "multiversionprovider";
    public static final String VERSION = "0.4.6";
    @Setter private static CommonAPI API;
    /**
     * Enable to disable server stuff
     */
    public static boolean CLIENT_ONLY;
    private static Reference INSTANCE;
    
    /**
     * Should only exist on the client
     */
    public static @Nullable ClientAPI getClientAPI() {
        CoreAPI core = CoreAPI.getInstance();
        if(core.getSide().isClient()) {
            if(Objects.isNull(API)) core.initAPI();
            return (ClientAPI)API;
        }
        logError("The client API can only be retrieved from the client side!");
        return null;
    }
    
    public static <A> @Nullable A getClientSubAPI(Function<ClientAPI,A> getter) {
        if(CoreAPI.isClient()) return getter.apply(getClientAPI());
        else logError("Cannot get client sub API {} since this is not the client side!");
        return null;
    }
    
    public static CommonAPI getCommonAPI() {
        if(Objects.isNull(API)) CoreAPI.getInstance().initAPI();
        return API;
    }
    
    public static <A> A getCommonSubAPI(Function<CommonAPI,A> getter) {
        return getter.apply(getCommonAPI());
    }
    
    public static String getNetworkVersion() {
        return VERSION;
    }
    
    public static SharedHandlesClient getClientHandles() {
        return getClientSubAPI(ClientAPI::getSharedHandlesClient);
    }
    
    public static SharedHandlesCommon getCommonHandles() {
        return getCommonSubAPI(CommonAPI::getSharedHandlesCommon);
    }
    
    /**
     * Initializes the base reference API
     */
    public static Reference instance(Supplier<Boolean> isClient, String dependencies) {
        if(Objects.isNull(INSTANCE)) INSTANCE = new Reference(isClient.get(),dependencies,MODID,NAME,VERSION);
        return INSTANCE;
    }
    
    @IndirectCallers
    public static void log(Level level, String msg, Object ... args) {
        logNullable(level,msg,args);
    }
    
    public static void logDebug(String msg, Object ... args) {
        logNullable(DEBUG,msg,args);
    }
    
    public static void logError(String msg, Object ... args) {
        logNullable(ERROR,msg,args);
    }
    
    public static void logFatal(String msg, Object ... args) {
        logNullable(FATAL,msg,args);
    }
    
    public static void logInfo(String msg, Object ... args) {
        logNullable(INFO,msg,args);
    }
    
    private static void logNullable(Level level, String msg, Object ... args) {
        if(Objects.nonNull(INSTANCE)) INSTANCE.log(level,msg,args);
        else LOGGER.log(level,msg,args);
    }
    
    @IndirectCallers
    public static void logTrace(String msg, Object ... args) {
        logNullable(TRACE,msg,args);
    }
    
    public static void logWarn(String msg, Object ... args) {
        logNullable(WARN,msg,args);
    }
    
    @IndirectCallers
    public static Logger createLogger(Class<?> c) {
        return LogManager.getLogger(c);
    }
    
    public static Logger createLogger(String loggerName) {
        return LogManager.getLogger(loggerName);
    }
    
    public static ResourceLocationAPI<?> res(String path) {
        if(Objects.nonNull(INSTANCE)) return INSTANCE.getResource(path);
        throw new RuntimeException("Cannot get a ResourceLocation until the reference API has been initialized!");
    }
}
