package mods.thecomputerizer.theimpossiblelibrary.api.text;

import lombok.Getter;
import mods.thecomputerizer.theimpossiblelibrary.api.core.TILRef;
import mods.thecomputerizer.theimpossiblelibrary.api.common.CommonAPI;
import mods.thecomputerizer.theimpossiblelibrary.api.core.annotation.IndirectCallers;
import mods.thecomputerizer.theimpossiblelibrary.api.util.GenericUtils;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

public class TextHelper {
    
    @IndirectCallers
    public static String arrayToString(Object ... array) {
        return arrayToString(0,System.lineSeparator(),array);
    }
    
    @IndirectCallers
    public static String arrayToString(int limit, Object ... array) {
        return arrayToString(limit,System.lineSeparator(),array);
    }
    
    @IndirectCallers
    public static String arrayToString(String split, Object ... array) {
        return arrayToString(0,split,array);
    }

    public static String arrayToString(int limit, String split, Object ... array) {
        if(Objects.isNull(array) || array.length==0) return null;
        limit = limit>0 ? limit : Integer.MAX_VALUE;
        StringBuilder builder = new StringBuilder();
        int index = 1;
        for(Object element : array) {
            if(Objects.nonNull(element)) {
                String asString = element.toString();
                if(!asString.trim().isEmpty()) builder.append(asString);
            }
            if(index<array.length && index<limit) {
                builder.append(split);
                index++;
            }
            else if(index==limit) return builder.toString();
        }
        return builder.toString();
    }

    public static String capitalize(String original) {
        if(Objects.isNull(original) || original.isEmpty()) return original;
        return String.valueOf(original.charAt(0)).toUpperCase()+original.substring(1).toLowerCase();
    }

    /**
     * Converts a collection of generic objects into a single string of the format [ "element1" "element2" "element3" ... ]
     */
    public static String compileCollection(Collection<?> collection) {
        return compileCollection(collection.toArray(new Object[0]));
    }

    /**
     * Converts an array of generic objects into a single string of the format [ element1 element2 element3 ... ]
     * If the objects are strings, the elements will be in the format [ "element1" "element2" "element3" ... ]
     */
    public static String compileCollection(Object ... generics) {
        StringBuilder builder = new StringBuilder();
        builder.append("[ ");
        for(Object element : generics) {
            if(element instanceof String) builder.append("\"").append(element).append("\" ");
            else builder.append(element);
        }
        builder.append("]");
        return builder.toString();
    }
    
    public static int count(String s, char c) {
        if(isEmpty(s)) return 0;
        int count = 0;
        for(int i=0;i<s.length();i++)
            if(s.charAt(i)==c) count++;
        return count;
    }
    
    @IndirectCallers
    public static boolean endsWithAny(String s, String ... endings) {
        if(isEmpty(s)) return false;
        for(String ending : endings)
            if(s.endsWith(ending)) return true;
        return false;
    }
    
    /**
     * Converts an iterable of strings to a single string with newline characters with an optional limiter.
     * The limit input determines the maximum number of elements that can be read in before it gets cut off.
     * Setting the limit to zero or below will disable it.
     * Returns null if the input list is empty or null.
     * If an element in the input list is empty, null, or has only whitespace,
     * it will be replaced with a newline character or be removed if it is the final element.
     */
    public static String fromIterable(Iterable<?> itr) {
        return fromIterable(itr,0,System.lineSeparator());
    }
    
    @IndirectCallers
    public static String fromIterable(Iterable<?> itr, int limit) {
        return fromIterable(itr,limit,System.lineSeparator());
    }
    
    public static String fromIterable(Iterable<?> itr, String split) {
        return fromIterable(itr,0,split);
    }
    
    public static String fromIterable(Iterable<?> itr, int limit, String split) {
        if(Objects.isNull(itr)) return null;
        int count = 0;
        StringJoiner joiner = new StringJoiner(split);
        for(Object value : itr) {
            joiner.add(String.valueOf(value));
            if(limit>0) {
                count++;
                if(count>=limit) break;
            }
        }
        String val = joiner.toString();
        return TextHelper.isNotEmpty(val) ? val : null;
    }

    public static <S> TextHelperAPI<S> getHelper() {
        return GenericUtils.cast(TILRef.getCommonSubAPI(CommonAPI::getTextHelper));
    }

    public static <S> TextStringAPI<S> getLiteral(String text) {
        return GenericUtils.cast(getHelper().getLiteral(text));
    }
    
    public static <S> TextTranslationAPI<S> getTranslated(String key, Object ... args) {
        return GenericUtils.cast(getHelper().getTranslated(key,args));
    }
    
    @IndirectCallers
    public static List<String> hangingIndent(List<String> lines) {
        return hangingIndent(lines,"    ");
    }
    
    public static List<String> hangingIndent(List<String> lines, String prefix) {
        for(int i=1;i<lines.size();i++) lines.set(i,prefix+lines.get(i));
        return lines;
    }
    
    public static boolean isBlank(@Nullable String s) {
        if(Objects.isNull(s) || s.isEmpty()) return true;
        for(int i=0;i<s.length();i++)
            if(!Character.isWhitespace(s.charAt(i))) return false;
        return true;
    }
    
    public static boolean isEmpty(@Nullable String s) {
        return Objects.isNull(s) || s.isEmpty();
    }
    
    public static boolean isNotBlank(@Nullable String s) {
        return !isBlank(s);
    }
    
    public static boolean isNotEmpty(@Nullable String s) {
        return !isEmpty(s);
    }

    /**
     * Assumes the input string is camel case
     */
    @IndirectCallers
    public static String makeCaseTypeFromCamel(String original, TextCasing type) {
        String[] words = TextCasing.CAMEL.split(original);
        if(type==TextCasing.CAMEL) return TextCasing.CAMEL.combine(words);
        if(type==TextCasing.PASCAL) return TextCasing.PASCAL.combine(words);
        if(type==TextCasing.SNAKE) return TextCasing.SNAKE.combine(words);
        return TextCasing.KEBAB.combine(words);
    }

    /**
     * Assumes the input string is kebab case
     */
    @IndirectCallers
    public static String makeCaseTypeFromKebab(String original, TextCasing type) {
        String[] words = TextCasing.KEBAB.split(original);
        if(type==TextCasing.CAMEL) return TextCasing.CAMEL.combine(words);
        if(type==TextCasing.PASCAL) return TextCasing.PASCAL.combine(words);
        if(type==TextCasing.SNAKE) return TextCasing.SNAKE.combine(words);
        return TextCasing.KEBAB.combine(words);
    }

    /**
     * Assumes the input string is pascal case
     */
    @IndirectCallers
    public static String makeCaseTypeFromPascal(String original, TextCasing type) {
        String[] words = TextCasing.PASCAL.split(original);
        if(type==TextCasing.CAMEL) return TextCasing.CAMEL.combine(words);
        if(type==TextCasing.PASCAL) return TextCasing.PASCAL.combine(words);
        if(type==TextCasing.SNAKE) return TextCasing.SNAKE.combine(words);
        return TextCasing.KEBAB.combine(words);
    }

    /**
     * Assumes the input string is snake case
     */
    @IndirectCallers
    public static String makeCaseTypeFromSnake(String original, TextCasing type) {
        String[] words = TextCasing.SNAKE.split(original);
        if(type==TextCasing.CAMEL) return TextCasing.CAMEL.combine(words);
        if(type==TextCasing.PASCAL) return TextCasing.PASCAL.combine(words);
        if(type==TextCasing.SNAKE) return TextCasing.SNAKE.combine(words);
        return TextCasing.KEBAB.combine(words);
    }

    /**
     * Splits a string into a list of strings based on the system line separator
     */
    @IndirectCallers
    public static List<String> newLineSplit(String original) {
        return Arrays.stream(original.split(System.lineSeparator())).collect(Collectors.toList());
    }

    /**
     * Same as the above method with a limit on the maximum number of elements.
     */
    @IndirectCallers
    public static List<String> newLineSplit(String original, int limit) {
        return Arrays.stream(original.split(System.lineSeparator(),limit)).collect(Collectors.toList());
    }

    /**
     * Implementation of String#repeat for the versions that rely on Java 8
     */
    @IndirectCallers
    public static String repeat(String base, int num) {
        StringBuilder builder = new StringBuilder();
        for(int i=0;i<num;i++) builder.append(base);
        return builder.toString();
    }

    /**
     * Splits a string into a list of strings based on the input separator
     */
    @IndirectCallers
    public static List<String> splitToList(String original, String splitBy) {
        return Arrays.stream(original.split(splitBy)).collect(Collectors.toList());
    }

    /**
     * Same as the above method with a limit on the maximum number of elements.
     */
    @IndirectCallers
    public static List<String> splitToList(String original, String splitBy, int limit) {
        return Arrays.stream(original.split(splitBy, limit)).collect(Collectors.toList());
    }

    /**
     * Returns the input string with the specified number of leading tabs
     */
    public static String withTabs(String original, int tabs) {
        StringBuilder builder = new StringBuilder();
        for(int i=0;i<tabs;i++) builder.append("\t");
        return builder.append(original).toString();
    }

    public enum TextCasing {

        CAMEL("camel",input -> input.split("(?=\\p{Upper})"),words -> {
            StringBuilder builder = new StringBuilder();
            if(Objects.isNull(words) || words.length==0) return builder.toString();
            for(int i=0;i<words.length;i++) {
                if(i==0) builder.append(words[i].toLowerCase());
                else builder.append(capitalize(words[i]));
            }
            return builder.toString();
        }),
        PASCAL("pascal",input -> input.split("(?<=.)(?=\\p{Upper})"),words -> {
            StringBuilder builder = new StringBuilder();
            if(Objects.isNull(words)) return builder.toString();
            for(String word : words)
                builder.append(capitalize(word));
            return builder.toString();
        }),
        SNAKE("snake",input -> input.split("_"),words -> {
            StringBuilder builder = new StringBuilder();
            if(Objects.isNull(words) || words.length==0) return builder.toString();
            for(int i=0;i<words.length;i++) {
                builder.append(words[i].toLowerCase());
                if(i<words.length-1) builder.append("_");
            }
            return builder.toString();
        }),
        KEBAB("kebab",input -> input.split("-"),words -> {
            StringBuilder builder = new StringBuilder();
            if(Objects.isNull(words) || words.length==0) return builder.toString();
            for(int i=0;i<words.length;i++) {
                builder.append(words[i].toLowerCase());
                if(i<words.length-1) builder.append("-");
            }
            return builder.toString();
        });

        @Getter private final String name;
        private final Function<String,String[]> splitFunc;
        private final Function<String[],String> combineFunc;
        TextCasing(String name, Function<String,String[]> splitFunc, Function<String[],String> combineFunc) {
            this.name = name;
            this.splitFunc = splitFunc;
            this.combineFunc = combineFunc;
        }

        public String[] split(String input) {
            return this.splitFunc.apply(input);
        }

        public String combine(String ... words) {
            return this.combineFunc.apply(words);
        }

        public static final Map<String, TextCasing> BY_NAME = new HashMap<>();

        static {
            for(TextCasing casing : values()) BY_NAME.putIfAbsent(casing.name,casing);
        }
    }
}