package mods.thecomputerizer.theimpossiblelibrary.api.toml;

import io.netty.buffer.ByteBuf;
import lombok.Getter;
import lombok.Setter;
import mods.thecomputerizer.theimpossiblelibrary.api.core.TILDev;
import mods.thecomputerizer.theimpossiblelibrary.api.core.TILRef;
import mods.thecomputerizer.theimpossiblelibrary.api.io.FileHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.parameter.Parameter;
import mods.thecomputerizer.theimpossiblelibrary.api.parameter.ParameterHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.toml.TomlReader.TableBuilder;
import mods.thecomputerizer.theimpossiblelibrary.api.core.ArrayHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.io.IOUtils;
import mods.thecomputerizer.theimpossiblelibrary.api.iterator.IterableHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.network.NetworkHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.text.TextHelper;
import mods.thecomputerizer.theimpossiblelibrary.api.util.GenericUtils;
import mods.thecomputerizer.theimpossiblelibrary.api.util.Matching;
import mods.thecomputerizer.theimpossiblelibrary.api.util.Sorting;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.*;
import java.util.Map.Entry;

/**
 Represents a TOML table. The root is considered a table.
 */
@SuppressWarnings({"unused","UnusedReturnValue"})
public class Toml {
    
    public static Toml getEmpty() {
        return new Toml("root");
    }
    
    public static Toml readBuf(ByteBuf buf) throws TomlParsingException {
        return readString(NetworkHelper.readString(buf));
    }
    
    public static Toml readFile(File file) throws TomlParsingException, IOException {
        return readFile(file,new TomlReader());
    }
    
    public static Toml readFile(File file, TomlReader reader) throws IOException, TomlParsingException {
        try(FileInputStream stream = new FileInputStream(file)) { //Close the stream without catching the exception
            return readStream(stream,reader);
        }
    }
    
    public static Toml readFile(String filePath) throws TomlParsingException, IOException {
        return readFile(filePath,new TomlReader());
    }
    
    public static Toml readFile(String filePath, TomlReader reader) throws TomlParsingException, IOException {
        try(FileInputStream stream = new FileInputStream(filePath)) { //Close the stream without catching the exception
            TILDev.logInfo("Reading toml from file path {}",filePath);
            return readStream(stream,reader);
        }
    }
    
    public static Toml readLines(Iterable<String> lines) throws TomlParsingException {
        return readLines(lines,new TomlReader());
    }
    
    public static Toml readLines(Iterable<String> lines, TomlReader reader) throws TomlParsingException {
        return readString(TextHelper.fromIterable(lines),reader);
    }
    
    public static Toml readPath(Path path, OpenOption ... options) throws TomlParsingException, IOException {
        return readPath(path,new TomlReader(),options);
    }
    
    public static Toml readPath(Path path, TomlReader reader, OpenOption ... options)
            throws IOException, TomlParsingException {
        try(InputStream stream = Files.newInputStream(path,options)) { //Close the stream without catching the exception
            return readStream(stream,reader);
        }
    }
    
    public static Toml readStream(InputStream stream) throws IOException, TomlParsingException {
        return readStream(stream,new TomlReader());
    }
    
    public static Toml readStream(InputStream stream, TomlReader reader) throws IOException, TomlParsingException {
        return readString(IOUtils.streamToString(stream),reader);
    }
    
    public static Toml readString(String tomlString) throws TomlParsingException {
        return readString(tomlString,new TomlReader());
    }
    
    public static Toml readString(String tomlString, TomlReader reader) throws TomlParsingException {
        reader.read(tomlString);
        return new Toml(reader.rootBuilder,"root");
    }
    
    public static Toml readURI(URI uri, OpenOption ... options) throws TomlParsingException, IOException {
        return readURI(uri,new TomlReader(),options);
    }
    
    public static Toml readURI(URI uri, TomlReader reader, OpenOption ... options)
            throws TomlParsingException, IOException {
        return readPath(FileHelper.toPath(uri),reader,options);
    }
    
    public static Toml readURL(URL url) throws TomlParsingException, IOException {
        return readURL(url,new TomlReader());
    }
    
    public static Toml readURL(URL url, TomlReader reader) throws IOException, TomlParsingException {
        try(InputStream stream = url.openStream()) { //Close the stream without catching the exception
            return readStream(stream,reader);
        }
    }
    
    @Getter private String name;
    private final Map<String,Toml[]> tables;
    private final Map<String,TomlEntry<?>> entries;
    private String[] comments; //Table comments only - entry comments are under the TableEntry class
    @Getter private Toml parent;
    @Setter @Getter private Sorting[] sorters; //TODO Make this functional
    
    Toml(String name) {
        this(null,name);
    }
    
    Toml(@Nullable TableBuilder builder, String name) {
        this.name = name;
        this.entries = new LinkedHashMap<>();
        this.tables = new LinkedHashMap<>();
        this.comments = new String[]{};
        if(Objects.nonNull(builder)) {
            setRootEntries(builder.entries);
            setTables(builder.tables);
            setComments(builder.tableComments,builder.entryComments);
        }
    }
    
    Toml(ByteBuf buf) {
        this.name = NetworkHelper.readString(buf);
        boolean comments = buf.readBoolean();
        this.tables = NetworkHelper.readMapEntries(buf,() -> {
            String name = NetworkHelper.readString(buf);
            return IterableHelper.getMapEntry(name,NetworkHelper.readList(buf,() -> {
                Toml table = new Toml(buf);
                table.parent = this;
                return table;
            }).toArray(new Toml[0]));
        });
        this.entries = NetworkHelper.readMapEntries(buf,() -> {
            String key = NetworkHelper.readString(buf);
            return IterableHelper.getMapEntry(key,new TomlEntry<>(key,buf,comments));
        });
        this.comments = comments ? NetworkHelper.readList(buf,() ->
                NetworkHelper.readString(buf)).toArray(new String[0]) : new String[]{};
    }
    
    public void addComment(String comment) {
        if(TextHelper.isNotEmpty(comment)) this.comments = ArrayHelper.append(this.comments,comment,true);
    }
    
    public void addComments(Iterable<String> comments) {
        for(String comment : comments) addComment(comment);
    }
    
    public void addComments(String ... comments) {
        for(String comment : comments) addComment(comment);
    }
    
    public <V> TomlEntry<V> addEntry(String key, V value) {
        TomlEntry<V> entry = new TomlEntry<>(key,value);
        addEntry(entry);
        return entry;
    }
    
    public <V> void addEntry(@Nullable TomlEntry<V> entry) {
        if(Objects.nonNull(entry)) this.entries.put(entry.getKey(),entry);
    }
    
    public void addEntryComment(String key, String comment) {
        TomlEntry<?> entry = getEntry(key);
        if(Objects.nonNull(entry)) entry.addComment(comment);
        else TILRef.logWarn("Tried to add comment to non existent entry {} (comment -> `{}`)",key,comment);
    }
    
    public void addEntryComments(String key, Iterable<String> comments) {
        TomlEntry<?> entry = getEntry(key);
        if(Objects.nonNull(entry)) {
            for(String comment : comments) entry.addComment(comment);
        } else TILRef.logWarn("Tried to add {} comments to non existent entry {}",IterableHelper.count(comments),key);
    }
    
    public void addEntryComments(String key, String ... comments) {
        TomlEntry<?> entry = getEntry(key);
        if(Objects.nonNull(entry)) {
            for(String comment : comments) entry.addComment(comment);
        } else TILRef.logWarn("Tried to add {} comments to non existent entry {}",comments.length,key);
    }
    
    public Toml addTable(String name, boolean array) throws TomlWritingException {
        Toml toml = new Toml(name);
        Toml[] tomls = this.tables.get(name);
        if(Objects.nonNull(tomls)) {
            if(array) ArrayHelper.append(tomls,toml,true);
            else throw new TomlWritingException("Cannot add table ["+name+"] that already exists");
        } else this.tables.put(name,new Toml[]{toml});
        toml.parent = this;
        return toml;
    }
    
    /**
     Renames the input table and adds it
     */
    public void addTable(String name, Toml table) {
        table.name = name;
        this.tables.put(name,ArrayHelper.append(this.tables.get(name),table,true));
        table.parent = this;
    }
    
    public void clear() {
        clear(true,true,true);
    }
    
    public void clear(boolean tables) {
        clear(true,true,tables);
    }
    
    public void clear(boolean entries, boolean tables) {
        clear(true,entries,tables);
    }
    
    public void clear(boolean comments, boolean entries, boolean tables) {
        if(!comments && !entries && !tables) {
            TILRef.logInfo("Toml#clear called but comments, entries, & tables are all false? "+
                           "Nothing will be cleared");
            return;
        }
        if(comments) clearAllComments();
        if(entries) clearAllEntries();
        if(tables) clearAllTables();
    }
    
    public void clearAllComments() {
        clearComments();
        clearAllEntryComments();
    }
    
    public void clearAllEntries() {
        this.entries.clear();
    }
    
    public void clearAllEntryComments() {
        for(TomlEntry<?> entry : this.entries.values()) entry.clearComments();
    }
    
    public void clearAllTables() {
        this.tables.clear();
    }
    
    public void clearAnyMatching(String toMatch, Matching ... matchers) {
        clearAnyMatching(toMatch,true,true,true,matchers);
    }
    
    public void clearAnyMatching(String toMatch, boolean tables, Matching ... matchers) {
        clearAnyMatching(toMatch,true,true,tables,matchers);
    }
    
    public void clearAnyMatching(String toMatch, boolean entries, boolean tables, Matching ... matchers) {
        clearAnyMatching(toMatch,true,entries,tables,matchers);
    }
    
    public void clearAnyMatching(String toMatch, boolean comments, boolean entries, boolean tables,
            Matching ... matchers) {
        if(!comments && !entries && !tables) {
            TILRef.logInfo("Toml#clearAnyMatching called but comments, entries, & tables are all false? "+
                           "Nothing will be cleared");
            return;
        }
        if(comments) clearAnyCommentsMatching(toMatch,matchers);
        if(entries) clearEntriesMatching(toMatch,matchers);
        if(tables) clearTablesMatching(toMatch,matchers);
    }
    
    public void clearAnyCommentsMatching(String toMatch, Matching ... matchers) {
        clearCommentsMatching(toMatch,matchers);
        for(TomlEntry<?> entry : this.entries.values()) entry.clearCommentsMatching(toMatch,matchers);
    }
    
    public void clearComments() {
        this.comments = new String[]{};
    }
    
    public void clearCommentsMatching(String toMatch, Matching ... matchers) {
        this.comments = ArrayHelper.removeMatching(this.comments,toMatch,comment ->
                Matching.matchesAny(comment,toMatch,matchers));
    }
    
    public void clearEntriesMatching(String toMatch, Matching ... matchers) {
        Collection<String> removals = Matching.matchingValuesAny(this.entries.keySet(),toMatch,matchers);
        for(String removal : removals) this.entries.remove(removal);
    }
    
    public void clearEntryCommentsMatching(String key, String toMatch, Matching ... matchers) {
        TomlEntry<?> entry = getEntry(key);
        if(Objects.nonNull(entry)) entry.clearCommentsMatching(toMatch,matchers);
    }
    
    public void clearEntryComments(String key) {
        TomlEntry<?> entry = getEntry(key);
        if(Objects.nonNull(entry)) entry.clearComments();
    }
    
    public void clearTablesMatching(String toMatch, Matching ... matchers) {
        Collection<String> removals = Matching.matchingValuesAny(this.tables.keySet(),toMatch,matchers);
        for(String removal : removals) this.tables.remove(removal);
    }
    
    public Collection<TomlEntry<?>> getAllEntries() {
        return this.entries.values();
    }
    
    public List<Toml> getAllTables() {
        List<Toml> tables = new ArrayList<>();
        for(Toml[] tomls : this.tables.values()) tables.addAll(Arrays.asList(tomls));
        return Collections.unmodifiableList(tables);
    }
    
    public TomlEntry<?> getEntry(String name) {
        return this.entries.getOrDefault(name,null);
    }
    
    public TomlEntry<List<?>> getEntryArray(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof List<?> ? GenericUtils.cast(entry) : null;
    }
    
    public TomlEntry<Boolean> getEntryBool(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof Boolean ? GenericUtils.cast(entry) : null;
    }
    
    public TomlEntry<Float> getEntryFloat(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof Float ? GenericUtils.cast(entry) : null;
    }
    
    public TomlEntry<Integer> getEntryInt(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof Integer ? GenericUtils.cast(entry) : null;
    }
    
    public TomlEntry<Number> getEntryNumber(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof Number ? GenericUtils.cast(entry) : null;
    }
    
    public TomlEntry<String> getEntryString(String name) {
        TomlEntry<?> entry = getEntry(name);
        return Objects.nonNull(entry) && entry.value instanceof String ? GenericUtils.cast(entry) : null;
    }
    
    public Map<String,Object> getEntryValuesAsMap() {
        Map<String,Object> map =  new LinkedHashMap<>(); //Preserve insertion order
        for(TomlEntry<?> entry : this.entries.values()) map.put(entry.key,entry.value);
        return map;
    }
    
    /**
     * Return a potentially empty generic optional depending on whether the value is present
     */
    public <T> Optional<T> getOptional(String name) {
        TomlEntry<T> entry = GenericUtils.cast(getEntry(name));
        return Objects.nonNull(entry) ? Optional.of(entry.value) : Optional.empty();
    }
    
    /**
     * Return a generic optional of the value if it is present or the defVal input.
     */
    public <T> Optional<T> getOptional(String name, @Nullable T defVal) {
        TomlEntry<T> entry = GenericUtils.cast(getEntry(name));
        return Optional.ofNullable(Objects.nonNull(entry) ? entry.value : defVal);
    }
    
    /**
     * Return a potentially empty list optional depending on whether the value is present
     */
    public Optional<List<?>> getOptionalArray(String name) {
        TomlEntry<List<?>> entry = getEntryArray(name);
        return Objects.nonNull(entry) ? Optional.of(entry.value) : Optional.empty();
    }
    
    /**
     * Return a list optional of the value if it is present or the defVal input.
     */
    public Optional<List<?>> getOptionalArray(String name, @Nullable List<?> defVal) {
        TomlEntry<List<?>> entry = getEntryArray(name);
        return Optional.ofNullable(Objects.nonNull(entry) ? entry.value : defVal);
    }
    
    /**
     * Return a potentially empty boolean optional depending on whether the value is present
     */
    public Optional<Boolean> getOptionalBool(String name) {
        TomlEntry<Boolean> entry = getEntryBool(name);
        return Objects.nonNull(entry) ? Optional.of(entry.value) : Optional.empty();
    }
    
    /**
     * Return a boolean optional of the value if it is present or the defVal input.
     */
    public Optional<Boolean> getOptionalBool(String name, boolean defVal) {
        TomlEntry<Boolean> entry = getEntryBool(name);
        return Optional.of(Objects.nonNull(entry) ? entry.value : defVal);
    }
    
    /**
     * Return a potentially empty byte optional depending on whether the value is present
     */
    public Optional<Byte> getOptionalByte(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.byteValue()) : Optional.empty();
    }
    
    /**
     * Return a byte optional of the value if it is present or the defVal input.
     */
    public Optional<Byte> getOptionalByte(String name, byte defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.byteValue() : defVal);
    }
    
    /**
     * Return a potentially empty double optional depending on whether the value is present
     */
    public Optional<Double> getOptionalDouble(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.doubleValue()) : Optional.empty();
    }
    
    /**
     * Return a double optional of the value if it is present or the defVal input.
     */
    public Optional<Double> getOptionalDouble(String name, double defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.doubleValue() : defVal);
    }
    
    /**
     * Return a potentially empty float optional depending on whether the value is present
     */
    public Optional<Float> getOptionalFloat(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.floatValue()) : Optional.empty();
    }
    
    /**
     * Return a float optional of the value if it is present or the defVal input.
     */
    public Optional<Float> getOptionalFloat(String name, float defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.floatValue() : defVal);
    }
    
    /**
     * Return a potentially empty int optional depending on whether the value is present
     */
    public Optional<Integer> getOptionalInt(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.intValue()) : Optional.empty();
    }
    
    /**
     * Return an int optional of the value if it is present or the defVal input.
     */
    public Optional<Integer> getOptionalInt(String name, int defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.intValue() : defVal);
    }
    
    /**
     * Return a potentially empty long optional depending on whether the value is present
     */
    public Optional<Long> getOptionalLong(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.longValue()) : Optional.empty();
    }
    
    /**
     * Return a long optional of the value if it is present or the defVal input.
     */
    public Optional<Long> getOptionalLong(String name, long defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.longValue() : defVal);
    }
    
    /**
     * Return a potentially empty number optional depending on whether the value is present
     */
    public Optional<Number> getOptionalNumber(String name) {
        return Optional.ofNullable(getValueNumber(name));
    }
    
    /**
     * Return a number optional of the value if it is present or the defVal input.
     */
    public Optional<Number> getOptionalNumber(String name, @Nullable Number defVal) {
        Number number = getValueNumber(name);
        return Optional.ofNullable(Objects.nonNull(number) ? number : defVal);
    }
    
    /**
     * Return a potentially empty short optional depending on whether the value is present
     */
    public Optional<Short> getOptionalShort(String name) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? Optional.of(number.shortValue()) : Optional.empty();
    }
    
    /**
     * Return a short optional of the value if it is present or the defVal input.
     */
    public Optional<Short> getOptionalShort(String name, short defVal) {
        Number number = getValueNumber(name);
        return Optional.of(Objects.nonNull(number) ? number.shortValue() : defVal);
    }
    
    /**
     * Return a potentially empty string optional depending on whether the value is present
     */
    public Optional<String> getOptionalString(String name) {
        TomlEntry<String> entry = getEntryString(name);
        return Objects.nonNull(entry) ? Optional.of(entry.value) : Optional.empty();
    }
    
    /**
     * Return a string optional of the value if it is present or the defVal input.
     */
    public Optional<String> getOptionalString(String name, @Nullable String defVal) {
        TomlEntry<String> entry = getEntryString(name);
        return Optional.ofNullable(Objects.nonNull(entry) ? entry.value : defVal);
    }
    
    public Optional<Toml> getOptionalTable(String name) {
        Toml[] tomls = this.tables.get(name);
        return ArrayHelper.isNotEmpty(tomls) ? Optional.of(tomls[0]) : Optional.empty();
    }
    
    public Optional<Toml> getOptionalTable(String name, @Nullable Toml defVal) {
        Toml[] tomls = this.tables.get(name);
        return Optional.ofNullable(ArrayHelper.isNotEmpty(tomls) ? tomls[0] : defVal);
    }
    
    public Optional<Toml[]> getOptionalTables(String name) {
        Toml[] tomls = this.tables.get(name);
        return Optional.of(ArrayHelper.isNotEmpty(tomls) ? tomls : new Toml[]{});
    }
    
    public Optional<Toml[]> getOptionalTables(String name, @Nullable Toml[] defVal) {
        Toml[] tomls = this.tables.get(name);
        return Optional.ofNullable(ArrayHelper.isNotEmpty(tomls) ? tomls : defVal);
    }
    
    public <V> V getOrSetValue(String key, V def) {
        return hasEntry(key) ? getValue(key) : addEntry(key,def).value;
    }
    
    /**
     * Returns the fully qualified path of this table including all non-root parent tables.
     * Note that this is assumed to be called for writing purposes and will encapsulate the name in quotes if necessary
     */
    public String getPath() {
        if("root".equals(this.name)) return "";
        String path = Objects.nonNull(this.parent) ? this.parent.getPath() : "";
        return path.isEmpty() ? TomlHelper.encapsulateTableName(this.name) :
                path+"."+TomlHelper.encapsulateTableName(this.name);
    }
    
    public Toml getTable(String name) {
        Toml[] tomls = this.tables.get(name);
        return ArrayHelper.isNotEmpty(tomls) ? tomls[0] : null;
    }
    
    public Toml[] getTableArray(String name) {
        return this.tables.get(name);
    }
    
    public <T> T getValue(String name) {
        return getValue(name,null);
    }
    
    public <T> T getValue(String name, @Nullable T defVal) {
        TomlEntry<T> entry = GenericUtils.cast(getEntry(name));
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public List<?> getValueArray(String name) {
        return getValueArray(name,null);
    }
    
    public List<?> getValueArray(String name, @Nullable List<?> defVal) {
        TomlEntry<List<?>> entry = getEntryArray(name);
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public List<?> getValueArrayOrEmpty(String name) {
        return getValueArray(name,new ArrayList<>());
    }
    
    public boolean getValueBool(String name) {
        return getValueBool(name,false);
    }
    
    public boolean getValueBool(String name, boolean defVal) {
        TomlEntry<Boolean> entry = getEntryBool(name);
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public byte getValueByte(String name) {
        return getValueByte(name,(byte)0);
    }
    
    public byte getValueByte(String name, byte defVal) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? number.byteValue() : defVal;
    }
    
    public double getValueDouble(String name) {
        return getValueDouble(name,0d);
    }
    
    public double getValueDouble(String name, double defVal) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? number.doubleValue() : defVal;
    }
    
    public float getValueFloat(String name) {
        return getValueFloat(name,0f);
    }
    
    public float getValueFloat(String name, float defVal) {
        TomlEntry<Float> entry = getEntryFloat(name);
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public int getValueInt(String name) {
        return getValueInt(name,0);
    }
    
    public int getValueInt(String name, int defVal) {
        TomlEntry<Integer> entry = getEntryInt(name);
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public long getValueLong(String name) {
        return getValueLong(name,0L);
    }
    
    public long getValueLong(String name, long defVal) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? number.longValue() : defVal;
    }
    
    public Number getValueNumber(String name) {
        return getValueNumber(name,null);
    }
    
    public Number getValueNumber(String name, @Nullable Number defVal) {
        TomlEntry<Number> entry = getEntryNumber(name);
        return Objects.nonNull(entry) ? entry.value: defVal;
    }
    
    public short getValueShort(String name) {
        return getValueShort(name,(short)0);
    }
    
    public short getValueShort(String name, short defVal) {
        Number number = getValueNumber(name);
        return Objects.nonNull(number) ? number.shortValue() : defVal;
    }
    
    public String getValueString(String name) {
        return getValueString(name,null);
    }
    
    public String getValueString(String name, @Nullable String defVal) {
        TomlEntry<String> entry = getEntryString(name);
        return Objects.nonNull(entry) ? entry.value : defVal;
    }
    
    public boolean hasEntry(String name) {
        return this.entries.containsKey(name);
    }
    
    public boolean hasTable(String name) {
        return this.tables.containsKey(name);
    }
    
    public Map<String,Parameter<?>> parameterizeEntries() {
        Map<String,Parameter<?>> map =  new LinkedHashMap<>(); //Preserve insertion order
        for(TomlEntry<?> entry : this.entries.values()) map.put(entry.key,entry.parameterize());
        return map;
    }
    
    /**
     Removes all tables with the given name regardless of whether they are singular or in an array
     */
    public void removeTables(String name) {
        removeTable(name,-1);
    }
    
    /**
     Set index to -1 to remove all elements in a table array with the input name
     */
    public void removeTable(String name, int index) {
        if(index==-1) this.tables.remove(name);
        else {
            Toml[] tomls = ArrayHelper.removeElement(this.tables.get(name),index);
            if(Objects.isNull(tomls)) return;
            if(tomls.length==0) this.tables.remove(name);
            else this.tables.put(name,tomls);
        }
    }
    
    public void removeEntry(String name) {
        this.entries.remove(name);
    }
    
    public void remapTables(String original, String remapped) {
        remapTable(original,remapped,-1);
    }
    
    /**
     Set the index to -1 to remap all tables with the original name
     */
    public void remapTable(String original, String remapped, int index) {
        if(Objects.isNull(remapped)) {
            this.tables.remove(original);
            return;
        }
        Toml[] tables = this.tables.get(original);
        if(Objects.nonNull(tables)) {
            this.tables.remove(original);
            for(Toml table : tables) table.name = remapped;
            this.tables.put(remapped,tables);
        }
    }
    
    void setComments(List<String> comments, Map<String,List<String>> entryComments) {
        this.comments = ArrayHelper.fromIterable(comments,String.class);
        for(Entry<String,List<String>> entryComment : entryComments.entrySet()) {
            TomlEntry<?> entry = this.entries.get(entryComment.getKey());
            if(Objects.nonNull(entry)) entry.setComments(entryComment.getValue());
        }
    }
    
    void setRootEntries(Map<String,Object> entries) {
        for(Entry<String,Object> entry : entries.entrySet()) {
            String key = entry.getKey();
            this.entries.put(key,new TomlEntry<>(key,entry.getValue()));
        }
    }
    
    void setTables(Map<String,List<TableBuilder>> builders) {
        for(Entry<String,List<TableBuilder>> entry : builders.entrySet()) {
            String name = entry.getKey();
            List<TableBuilder> tables = entry.getValue();
            Toml[] tomls = new Toml[tables.size()];
            for(int i=0;i<tomls.length;i++) {
                Toml table = new Toml(tables.get(i),name);
                table.parent = this;
                tomls[i] = table;
            }
            this.tables.put(name,tomls);
        }
    }
    
    public String toString() {
        StringBuilder builder = new StringBuilder("\n");
        write(builder,0);
        return builder.toString();
    }
    
    /**
     * Write this table to a StringBuilder with optional formatting and comments enabled.
     * @param builder A StringBuilder output of the written table
     * @param tabs The number of tabs to use when writing the table. Set -1 to disable formatting entirely
     */
    public void write(StringBuilder builder, int tabs) {
        write(builder,tabs,true);
    }
    
    /**
     * Write this table to a StringBuilder with optional formatting.
     * @param builder A StringBuilder output of the written table
     * @param tabs The number of tabs to use when writing the table. Set -1 to disable formatting entirely
     * @param comments Enables the writing of comments
     */
    public void write(StringBuilder builder, int tabs, boolean comments) {
        for(String comment : this.comments)
            builder.append(tabs==-1 ? "#"+comment : TextHelper.withTabs("# "+comment,tabs)).append("\n");
        if(this.comments.length!=0) builder.append("\n");
        List<String> sorted = new ArrayList<>(this.entries.keySet());
        Collections.sort(sorted);
        for(String key : sorted) {
            TomlEntry<?> entry = this.entries.get(key);
            String entryLine = key+" = "+entry.writeValue();
            for(String comment : entry.comments)
                builder.append(tabs==-1 ? "#"+comment : TextHelper.withTabs("# "+comment,tabs)).append("\n");
            builder.append(tabs==-1 ? entryLine : TextHelper.withTabs(entryLine,tabs)).append("\n");
        }
        if("root".equals(this.name) && tabs!=-1 && !sorted.isEmpty()) builder.append("\n");
        sorted = new ArrayList<>(this.tables.keySet());
        Collections.sort(sorted);
        for(String key : sorted) {
            Toml[] tomls = this.tables.get(key);
            if(ArrayHelper.isNotEmpty(tomls)) {
                boolean array = tomls.length>1;
                for(Toml toml : tomls) {
                    String tableName = TomlHelper.encapsulateTablePath(toml.getPath(),array);
                    builder.append(tabs!=-1 ? tableName : TextHelper.withTabs(tableName,tabs)).append("\n");
                    toml.write(builder,tabs==-1 ? -1 : tabs+1,comments);
                    if("root".equals(this.name) && tabs!=-1) builder.append("\n");
                }
            }
        }
    }
    
    /**
     * Write this table to a collection of strings with optional formatting and comments enabled.
     * @param lines A collections of strings where each entry is assumed to be a separate line
     * @param tabs The number of tabs to use when writing the table. Set -1 to disable formatting entirely
     */
    public void write(Collection<String> lines, int tabs) {
        write(lines,tabs,true);
    }
    
    /**
     * Write this table to a collection of strings with optional formatting.
     * @param lines A collections of strings where each entry is assumed to be a separate line
     * @param tabs The number of tabs to use when writing the table. Set -1 to disable formatting entirely
     * @param comments Enables the writing of comments
     */
    public void write(Collection<String> lines, int tabs, boolean comments) {
        for(String comment : this.comments)
            lines.add(tabs==-1 ? "#"+comment : TextHelper.withTabs("# "+comment,tabs));
        if(this.comments.length!=0 && tabs!=-1) lines.add("");
        List<String> sorted = new ArrayList<>(this.entries.keySet());
        Collections.sort(sorted);
        for(String key : sorted) {
            TomlEntry<?> entry = getEntry(key);
            String entryLine = key+" = "+entry.writeValue();
            for(String comment : entry.comments)
                lines.add(tabs==-1 ? "#"+comment : TextHelper.withTabs("# "+comment,tabs));
            lines.add(tabs==-1 ? entryLine : TextHelper.withTabs(entryLine,tabs));
        }
        if("root".equals(this.name) && tabs!=-1 && !sorted.isEmpty()) lines.add("");
        sorted = new ArrayList<>(this.tables.keySet());
        Collections.sort(sorted);
        for(String key : sorted) {
            Toml[] tomls = this.tables.get(key);
            if(ArrayHelper.isNotEmpty(tomls)) {
                boolean array = tomls.length>1;
                for(Toml toml : tomls) {
                    String tableName = TomlHelper.encapsulateTablePath(toml.getPath(),array);
                    lines.add(tabs==-1 ? tableName : TextHelper.withTabs(tableName,tabs));
                    toml.write(lines,tabs==-1 ? -1 : tabs+1,comments);
                    if("root".equals(this.name) && tabs!=-1) lines.add("");
                }
            }
        }
    }
    
    /**
     Write this table to a ByteBuf with comments disabled.
     * @param buf The buffer to be written
     */
    public void write(ByteBuf buf) {
        write(buf,false);
    }
    
    /**
     Write this table to a ByteBuf with the option of enabling comments to be written.
     * @param buf The buffer to be written
     * @param comments Enables the writing of comments
     */
    public void write(ByteBuf buf, boolean comments) {
        NetworkHelper.writeString(buf,toString());
    }
    
    @Getter
    public static class TomlEntry<V> {
        
        private final String key;
        private final V value;
        private String[] comments;
        
        public TomlEntry(String key, V value) {
            this.key = key;
            this.value = value;
            this.comments = new String[]{};
        }
        
        @SuppressWarnings("unchecked") private TomlEntry(String key, ByteBuf buf, boolean comments) {
            this.key = key;
            String type = NetworkHelper.readString(buf);
            switch(type) {
                case "bool": {
                    this.value = (V)(Boolean)buf.readBoolean();
                    break;
                }
                case "float": {
                    this.value = (V)(Float)buf.readFloat();
                    break;
                }
                case "int": {
                    this.value = (V)(Integer)buf.readInt();
                    break;
                }
                default: {
                    this.value = (V)NetworkHelper.readString(buf);
                    break;
                }
            }
            this.comments = comments ? NetworkHelper.readList(buf,() ->
                    NetworkHelper.readString(buf)).toArray(new String[0]) : new String[]{};
        }
        
        public void addComment(String comment) {
            if(TextHelper.isNotEmpty(comment)) this.comments = ArrayHelper.append(this.comments,comment,true);
        }
        
        public void clearComments() {
            this.comments = new String[]{};
        }
        
        public void clearCommentsMatching(String toMatch, Matching ... matchers) {
            this.comments = ArrayHelper.removeMatching(this.comments,toMatch,comment ->
                    Matching.matchesAny(comment,toMatch,matchers));
        }
        
        Parameter<?> parameterize() {
            return ParameterHelper.parameterize(this.value.getClass(),this.value);
        }
        
        void setComments(List<String> comments) {
            this.comments = ArrayHelper.fromIterable(comments,String.class);
        }
        
        @Override public String toString() {
            return "<"+this.key+" = "+writeValue()+">";
        }
        
        String writeValue() {
            return writeValue(this.value);
        }
        
        String writeValue(Object value) { //TODO Split long arrays to multiple lines?
            if(value instanceof List<?>) {
                StringJoiner joiner = new StringJoiner(",");
                for(Object element : (List<?>)value) joiner.add(" "+writeValue(element));
                return "["+joiner+" ]";
            }
            return value instanceof String ? "\""+value+"\"" : (Objects.nonNull(value) ? value.toString() : "null");
        }
    }
}