package mods.thecomputerizer.theimpossiblelibrary.forge.core.modules;

import lombok.Setter;
import mods.thecomputerizer.theimpossiblelibrary.api.core.annotation.IndirectCallers;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ClassLoaderAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ConfigurationAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleDescriptorAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleHolder;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleLayerAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleReferenceAccess;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ModuleSystemAccessor;
import mods.thecomputerizer.theimpossiblelibrary.api.core.modules.ResolvedModuleAccess;
import org.jetbrains.annotations.Nullable;

import java.net.URI;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;

import static mods.thecomputerizer.theimpossiblelibrary.forge.core.ForgeCoreLoader.SECURE_CLASSLOADER_FORMAT;

/**
 * [1.18.2+] cpw.mods.cl.ModuleClassLoader
 * [1.20.4+] extends net.minecraftforge.securemodules.SecureModuleClassLoader
 */
public class ModuleClassLoaderAccess extends ClassLoaderAccess implements ModuleHolder {
    
    static final String packageLookupField = ForgeModuleAccess.changing("packageToOurModules","packageLookup");
    static final String parentLoadersField = ForgeModuleAccess.changing("packageToParentLoader","parentLoaders");
    static final String resolvedRootsField = ForgeModuleAccess.changing("ourModules","resolvedRoots");
    
    @Setter String layerName;
    
    ModuleClassLoaderAccess(ClassLoader loader, Object accessorOrLogger) {
        super(loader,accessorOrLogger);
    }
    
    @IndirectCallers
    public void addPackage(String pkg, ResolvedModuleAccess resolvedModule) {
        addPackage(pkg,resolvedModule.access());
    }
    
    void addPackage(String pkg, Object resolvedModule) {
        packageLookup().put(pkg,resolvedModule);
    }
    
    @IndirectCallers
    public void addPackages(ResolvedModuleAccess resolvedModule) {
        addPackages(resolvedModule.reference().descriptor(),resolvedModule);
    }
    
    public void addPackages(ModuleDescriptorAccess moduleDescriptor, ResolvedModuleAccess resolvedModule) {
        addPackages(moduleDescriptor.packages(),resolvedModule);
    }
    
    public void addPackages(Collection<String> pkgs, ResolvedModuleAccess resolvedModule) {
        addPackages(pkgs,resolvedModule.access());
    }
    
    void addPackages(Collection<String> pkgs, Object resolvedModule) {
        Map<String,Object> packageLookup = packageLookup();
        for(String pkg : pkgs) packageLookup.put(pkg,resolvedModule);
    }
    
    @IndirectCallers
    public void addPackagesFrom(Collection<String> pkgs, ModuleClassLoaderAccess source, String moduleName) {
        Object module = source.getConfigModuleDirect(moduleName);
        addPackages(pkgs,module);
    }
    
    @IndirectCallers
    public void addParentLoaders(ResolvedModuleAccess resolvedModule, ModuleClassLoaderAccess loader) {
        addParentLoaders(resolvedModule.reference().descriptor(),loader);
    }
    
    public void addParentLoaders(ModuleDescriptorAccess moduleDescriptor, ModuleClassLoaderAccess loader) {
        addParentLoaders(moduleDescriptor.packages(),loader);
    }
    
    public void addParentLoaders(Collection<String> pkgs, ModuleClassLoaderAccess loader) {
        addParentLoaders(pkgs,loader.access());
    }
    
    void addParentLoaders(Collection<String> pkgs, Object loader) {
        Map<String,Object> parentLoaders = parentLoaders();
        for(String pkg : pkgs) parentLoaders.put(pkg,loader);
    }
    
    @IndirectCallers
    public void addParentLoader(String pkg, ModuleClassLoaderAccess loader) {
        addParentLoader(pkg,loader.access());
    }
    
    void addParentLoader(String pkg, Object loader) {
        parentLoaders().put(pkg,loader);
    }
    
    @IndirectCallers
    public void addRoot(ResolvedModuleAccess resolvedModule) {
        addRoot(resolvedModule.reference());
    }
    
    public void addRoot(ModuleReferenceAccess moduleReference) {
        addRoot(moduleReference.name(),moduleReference.access());
    }
    
    @IndirectCallers
    public void addRoot(String name, ModuleReferenceAccess moduleReference) {
        addRoot(name,moduleReference.access());
    }
    
    public void addRoot(String name, Object moduleReference) {
        resolvedRoots().put(name,moduleReference);
    }
    
    public void addSecureModule(Object secureModule, String ... names) {
        if(SECURE_CLASSLOADER_FORMAT)
            for(String name : names) ourModulesSecure().put(name,secureModule);
    }
    
    @Override public void cloneModule(String moduleName, String newModuleName) {
        clonePackages(moduleName,newModuleName);
        cloneModuleMap(resolvedRoots(),moduleName,newModuleName);
        if(SECURE_CLASSLOADER_FORMAT)
            cloneModuleMap(ourModulesSecure(),moduleName,newModuleName);
    }
    
    void cloneModuleMap(Map<String,Object> map, String moduleName, String newModuleName) {
        if(map.containsKey(moduleName)) {
            Object o = map.get(moduleName);
            map.remove(moduleName);
            if(!map.containsKey(newModuleName)) {
                getModuleReference(o).setName(newModuleName);
                map.put(newModuleName,o);
            }
        }
    }
    
    void clonePackages(String moduleName, String newModuleName) {
        ResolvedModuleAccess existingModule = lookupResolvedModule(newModuleName);
        boolean existed = Objects.nonNull(existingModule);
        Set<String> packages = existed ? new HashSet<>() : null;
        Map<String,Object> packageLookup = packageLookup();
        for(Entry<String,Object> packageEntry : packageLookup.entrySet()) {
            ResolvedModuleAccess moduleAccess = getAsResolvedModule(packageEntry.getValue());
            if(moduleName.equals(moduleAccess.name())) {
                if(existed) packages.add(packageEntry.getKey());
                else moduleAccess.setName(newModuleName);
            }
        }
        if(existed)
            for(String pkg : packages) packageLookup.put(pkg,existingModule.access());
    }
    
    /**
     * Assumes the combined module already has a defined root
     */
    public void combineModules(URI combinedLocation, String combinedName, String ... others) {
        ConfigurationAccess configuration = configuration();
        ResolvedModuleAccess module = configuration.getModule(combinedName);
        if(Objects.isNull(module)) {
            this.logger.error("Cannot combined modules {} into module {} that does not exist!",others,
                              combinedLocation);
            return;
        }
        ModuleLayerAccess layer = getModuleLayer();
        ModuleReferenceAccess reference = getRoot(combinedName);
        if(Objects.nonNull(reference)) reference.setLocation(combinedLocation);
        module.inheritFrom(configuration,others);
        Map<String,Object> parentLoaders = parentLoaders();
        Map<String,Object> packageLookup = packageLookup();
        for(String pkg : module.packages()) {
            packageLookup.put(pkg,module.accessAs());
            parentLoaders.remove(pkg);
        }
        removeRoots(others);
        configuration.removeModules(others);
        layer.combineModules(combinedName,others);
    }
    
    public ConfigurationAccess configuration() {
        Object configuration = getDirect("configuration");
        if(Objects.nonNull(configuration)) return getConfiguration(configuration);
        logOrPrintError("Configuration field not found in ModuleClassLoader "+this.access());
        return null;
    }
    
    @Override public Collection<ModuleHolder> getAllReferents() {
        return Arrays.asList(this,configuration(),getModuleLayer());
    }
    
    ResolvedModuleAccess getAsResolvedModule(Object resolvedModule) {
        if(Objects.isNull(resolvedModule)) return null;
        if(resolvedModule instanceof ResolvedModuleAccess) return (ResolvedModuleAccess)resolvedModule;
        return ModuleSystemAccessor.getResolvedModule(resolvedModule,this);
    }
    
    public <T> T getConfigModuleDirect(String name) {
        return configuration().getModuleDirect(name);
    }
    
    public @Nullable ModuleDescriptorAccess getModuleDescriptor(String name) {
        ModuleAccess module = getModuleLayer().getModule(name);
        return Objects.nonNull(module) ? module.getDescriptor() : null;
    }
    
    public @Nullable Object getModuleDescriptorDirect(String name) {
        ModuleDescriptorAccess descriptorAccess = getModuleDescriptor(name);
        return Objects.nonNull(descriptorAccess) ? descriptorAccess.accessAs() : null;
    }
    
    public ModuleLayerAccess getModuleLayer() {
        if(Objects.isNull(this.layerName)) {
            logOrPrintError("Cannot get ModuleLayer! (ModuleClassLoaderAccess#layerName is null)");
            return null;
        }
        return ForgeModuleAccess.getModuleLayer(this.layerName,this);
    }
    
    public ResolvedModuleAccess getResolvedModule(String pkg) {
        return getAsResolvedModule(packageLookup().get(pkg));
    }
    
    @IndirectCallers
    public String getResolvedModuleName(Object resolvedModule) {
        return getAsResolvedModule(resolvedModule).name();
    }
    
    public ModuleReferenceAccess getRoot(String name) {
        Object moduleReference = resolvedRoots().get(name);
        return Objects.nonNull(moduleReference) ? getModuleReference(moduleReference) : null;
    }
    
    public Object getRootDirect(String name) {
        return resolvedRoots().get(name);
    }
    
    @IndirectCallers
    public ModuleLayerHandlerAccess handler() {
        return ForgeModuleAccess.getModuleLayerHandler(this);
    }
    
    /**
     * Get a ResolvedModule from the packageLookup map that matches the input moduleName
     * Remove all packages associated with the module from the packageLookup
     */
    public void lookupAndRemovePackagesFor(String moduleName) {
        removePackages(lookupResolvedModule(moduleName));
    }
    
    /**
     * Get a ResolvedModule from the packageLookup map that matches the input moduleName
     */
    public ResolvedModuleAccess lookupResolvedModule(String moduleName) {
        for(Object resolvedModule : packageLookup().values()) {
            ResolvedModuleAccess moduleAccess = getAsResolvedModule(resolvedModule);
            if(moduleName.equals(moduleAccess.name())) return moduleAccess;
        }
        return null;
    }
    
    /**
     * Assumes layerName has been set for both this and the targetLoader
     */
    public void moveModuleTo(ModuleClassLoaderAccess targetLoader, String moduleName) {
        ResolvedModuleAccess resolvedModule = lookupResolvedModule(moduleName);
        Set<String> packages = resolvedModule.packages(false);
        Object resolveModuleAccess = resolvedModule.access();
        Object secureModule = SECURE_CLASSLOADER_FORMAT ? secureModuleDirect(moduleName) : null;
        resolvedModule.configuration().moveModuleTo(targetLoader.configuration(),resolvedModule);
        moveRoots(targetLoader,moduleName);
        if(Objects.nonNull(secureModule)) {
            targetLoader.addSecureModule(secureModule,moduleName);
            removeSecureModule(moduleName);
        }
        movePackageLookup(targetLoader,packages,resolveModuleAccess);
    }
    
    private void movePackageLookup(ModuleClassLoaderAccess targetLoader, Set<String> packages,
            Object resolvedModuleAccess) {
        movePackageParent(targetLoader,packages);
        Map<String,Object> packageLookup = packageLookup();
        Map<String,Object> targetPackageLookup = targetLoader.packageLookup();
        for(String pkg : packages) {
            packageLookup.remove(pkg);
            targetPackageLookup.put(pkg,resolvedModuleAccess);
        }
    }
    
    private void movePackageParent(ModuleClassLoaderAccess targetLoader, Set<String> packages) {
        if(packages.isEmpty()) return;
        if(!"BOOT".equals(this.layerName)) addParentLoaders(packages,targetLoader);
        targetLoader.removeParentLoaders(packages);
    }
    
    private void moveRoots(ModuleClassLoaderAccess targetLoader, String moduleName) {
        Object ref = getRootDirect(moduleName);
        if(Objects.nonNull(ref)) {
            removeRoot(moduleName);
            targetLoader.addRoot(moduleName,ref);
        }
    }
    
    @IndirectCallers
    public void moveServicesTo(ModuleLayerAccess target, ModuleAccess module) {
        getModuleLayer().moveServicesTo(target,module);
    }
    
    @IndirectCallers
    public ResolvedModuleAccess newResolvedModule(ModuleReferenceAccess moduleReference) {
        return newResolvedModule(Objects.nonNull(moduleReference) ? moduleReference.access() : null);
    }
    
    public ResolvedModuleAccess newResolvedModule(Object moduleReference) {
        ConfigurationAccess configuration = configuration();
        return newResolvedModule(Objects.nonNull(configuration) ? configuration.access() : null,moduleReference);
    }
    
    public ResolvedModuleAccess newResolvedModule(Object configuration, Object moduleReference) {
        if(Objects.isNull(configuration)) {
            logOrPrintError("Cannot create new ResolvedModule with null Configuration!");
            return null;
        }
        if(Objects.isNull(moduleReference)) {
            logOrPrintError("Cannot create new ResolvedModule with null ModuleReference!");
            return null;
        }
        return newResolvedModule(configuration,moduleReference);
    }
    
    /**
     * Only present in 1.20.4+ but still needs to be accounted for
     */
    public Map<String,Object> ourModulesSecure() {
        return SECURE_CLASSLOADER_FORMAT ? getDirect("ourModulesSecure") : Collections.emptyMap();
    }
    
    @IndirectCallers
    public Map<String,Object> packageLookup() {
        return getDirect(packageLookupField);
    }
    
    public Map<String,Object> packageToCodeSource() {
        return SECURE_CLASSLOADER_FORMAT ? getDirect("packageToCodeSource") : Collections.emptyMap();
    }
    
    @IndirectCallers
    public Map<String,Object> parentLoaders() {
        return getDirect(parentLoadersField);
    }
    
    public void removeModule(String moduleName) {
        removeRoot(moduleName);
        removeSecureModule(moduleName);
        lookupAndRemovePackagesFor(moduleName);
    }
    
    public void removePackages(ResolvedModuleAccess resolvedModule) {
        if(Objects.nonNull(resolvedModule)) removePackages(resolvedModule.packages());
    }
    
    public void removePackages(Collection<String> pkgs) {
        if(Objects.isNull(pkgs) || pkgs.isEmpty()) return;
        Collection<Map<String,Object>> maps = SECURE_CLASSLOADER_FORMAT ?
                Arrays.asList(packageLookup(),parentLoaders(),packageToCodeSource()) :
                Arrays.asList(packageLookup(),parentLoaders());
        for(Map<String,Object> map : maps)
            for(String pkg : pkgs) map.remove(pkg);
    }
    
    @IndirectCallers
    public void removePackage(String pkg) {
        removePackageLookup(pkg);
        removeParentLoader(pkg);
        removePackageToCodeSource(pkg);
    }
    
    public void removePackageLookup(String pkg) {
        packageLookup().remove(pkg);
    }
    
    public void removePackageToCodeSource(String pkg) {
        if(SECURE_CLASSLOADER_FORMAT) packageToCodeSource().remove(pkg);
    }
    
    @IndirectCallers
    public void removePackagesForModule(ResolvedModuleAccess resolvedModule) {
        removePackagesForModule(resolvedModule.access());
    }
    
    public void removePackagesForModule(Object resolvedModule) {
        packageLookup().entrySet().removeIf(entry -> entry.getValue().equals(resolvedModule));
    }
    
    public void removeParentLoaders(Collection<String> pkgs) {
        Map<String,Object> parentLoaders = parentLoaders();
        for(String pkg : pkgs) parentLoaders.remove(pkg);
    }
    
    public void removeParentLoader(String pkg) {
        parentLoaders().remove(pkg);
    }
    
    public void removeRoot(String root) {
        resolvedRoots().remove(root);
    }
    
    public void removeRoots(String ... roots) {
        Map<String,Object> resolvedRoots = resolvedRoots();
        for(String root : roots) resolvedRoots.remove(root);
    }
    
    public void removeSecureModule(String moduleName) {
        if(SECURE_CLASSLOADER_FORMAT) ourModulesSecure().remove(moduleName);
    }
    
    public void renameModule(String name, String newName) {
        this.logger.info("Renaming module from {} to {}",name,newName);
        ConfigurationAccess configuration = configuration();
        ResolvedModuleAccess module = configuration.getModule(name);
        if(Objects.isNull(module)) {
            this.logger.error("Cannot rename module {} that does not exist on layer {}!",name,this.layerName);
            return;
        }
        ModuleLayerAccess layer = getModuleLayer();
        module.setName(newName);
        Object root = getRootDirect(name);
        if(Objects.nonNull(root)) {
            removeRoot(name);
            addRoot(newName,root);
        }
        configuration.renameModule(name,newName);
        layer.renameModule(name,newName);
    }
    
    @IndirectCallers
    public Map<String,Object> resolvedRoots() {
        return getDirect(resolvedRootsField);
    }
    
    @SuppressWarnings("unchecked")
    public <T> T secureModuleDirect(String name) {
        return SECURE_CLASSLOADER_FORMAT ? (T)ourModulesSecure().get(name) : null;
    }
}