# MrUltimateShop API Documentation

## Overview
MrUltimateShop provides a comprehensive API for other plugins to interact with the shop system. This allows you to get item prices, check if items can be bought/sold, and retrieve various information about the shop system.

## Getting Started

### 1. Add MrUltimateShop as a dependency
In your plugin.yml, add MrUltimateShop as a dependency:
```yaml
depend: [MrUltimateShop]
```

Or as a soft dependency if your plugin can work without it:
```yaml
softdepend: [MrUltimateShop]
```

### 2. Check if the API is available
Before using the API, always check if it's available:
```java
if (MrUltimateShopAPI.isAvailable()) {
    // API is ready to use
}
```

## API Methods

### Item Price Methods

#### Get Sell Price
Get the sell price of an ItemStack:
```java
double sellPrice = MrUltimateShopAPI.getItemStackPriceSell(itemStack);
```

#### Get Buy Price
Get the buy price of an ItemStack:
```java
double buyPrice = MrUltimateShopAPI.getItemStackPriceBuy(itemStack);
```

### Item Availability Methods

#### Check if Item Can Be Sold
Check if an ItemStack can be sold:
```java
boolean canSell = MrUltimateShopAPI.canSellItem(itemStack);
```

#### Check if Item Can Be Bought
Check if an ItemStack can be bought:
```java
boolean canBuy = MrUltimateShopAPI.canBuyItem(itemStack);
```

### Player Inventory Methods

#### Get Total Sell Value
Get the total value of all matching sellable items in a player's inventory:
```java
double totalValue = MrUltimateShopAPI.getTotalSellValue(player, itemStack);
```

#### Get Sellable Item Count
Get the count of matching sellable items in a player's inventory:
```java
int itemCount = MrUltimateShopAPI.getSellableItemCount(player, itemStack);
```

### Utility Methods

#### Check API Availability
Check if the API is available and properly initialized:
```java
boolean isAvailable = MrUltimateShopAPI.isAvailable();
```

#### Get Plugin Version
Get the MrUltimateShop plugin version:
```java
String version = MrUltimateShopAPI.getVersion();
```

## Example Usage

### Example 1: Get Item Sell Price
```java
import eu.mrneznamy.mrultimateshop.api.MrUltimateShopAPI;
import org.bukkit.inventory.ItemStack;

public class ExamplePlugin extends JavaPlugin {
    
    public void onEnable() {
        if (MrUltimateShopAPI.isAvailable()) {
            getLogger().info("MrUltimateShop API is available! Version: " + MrUltimateShopAPI.getVersion());
        } else {
            getLogger().warning("MrUltimateShop is not available!");
        }
    }
    
    public double getSellPrice(ItemStack item) {
        if (MrUltimateShopAPI.isAvailable() && MrUltimateShopAPI.canSellItem(item)) {
            return MrUltimateShopAPI.getItemStackPriceSell(item);
        }
        return 0.0;
    }
}
```

### Example 2: Your Original Use Case
```java
// Your original example
if (provider.equalsIgnoreCase("MrUltimateShop") && Bukkit.getPluginManager().getPlugin("MrUltimateShop") != null) {
    return MrUltimateShopAPI.getItemStackPriceSell((ItemStack)item);
}
```

## Important Notes

1. **Always check availability**: Before calling any API method, check if the API is available using `MrUltimateShopAPI.isAvailable()`.

2. **Null safety**: All API methods handle null parameters gracefully and will return appropriate default values.

3. **NBT-aware matching**: The API uses the same NBT-aware item matching as the main plugin, ensuring consistency.

4. **Command items**: Items with type "COMMAND" cannot be sold and will return 0.0 for sell price and false for canSellItem().

5. **Performance**: API methods are optimized and cache results where appropriate, but avoid calling them in tight loops.

## Version Compatibility

This API is compatible with MrUltimateShop version 1.0.0 and above. Breaking changes will be clearly documented in release notes.