/*
 * Decompiled with CFR 0.152.
 */
package me.gamerduck.alwaysauth.api;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import me.gamerduck.alwaysauth.Platform;

public class AuthDatabase {
    private Connection connection;
    private final Platform platform;
    private final Gson gson;
    private final boolean isRemote;

    public AuthDatabase(File dbFile, Platform platform) {
        this.platform = platform;
        this.gson = new Gson();
        this.isRemote = false;
        try {
            Class.forName("org.h2.Driver");
            String url = "jdbc:h2:file:" + dbFile.getAbsolutePath().replace("\\", "/") + ";AUTO_SERVER=TRUE;MODE=MySQL";
            this.connection = DriverManager.getConnection(url, "sa", "");
            this.initializeTables();
            platform.sendLogMessage("H2 database initialized at: " + dbFile.getAbsolutePath() + ".mv.db");
        }
        catch (Exception e) {
            platform.sendSevereLogMessage("Failed to initialize H2 database: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public AuthDatabase(String host, int port, String database, String username, String password, String dbType, Platform platform) {
        this.platform = platform;
        this.gson = new Gson();
        this.isRemote = true;
        try {
            String driverClass;
            String url = switch (dbType.toLowerCase()) {
                case "mysql" -> {
                    driverClass = "com.mysql.cj.jdbc.Driver";
                    yield "jdbc:mysql://" + host + ":" + port + "/" + database + "?autoReconnect=true&useSSL=false";
                }
                case "mariadb" -> {
                    driverClass = "org.mariadb.jdbc.Driver";
                    yield "jdbc:mariadb://" + host + ":" + port + "/" + database;
                }
                default -> throw new IllegalArgumentException("Unsupported database type: " + dbType + ". Supported types: mysql, mariadb, postgresql");
            };
            Class.forName(driverClass);
            this.connection = DriverManager.getConnection(url, username, password);
            this.initializeTables();
            platform.sendLogMessage("Remote database connection established to " + host + ":" + port + "/" + database);
        }
        catch (Exception e) {
            platform.sendSevereLogMessage("Failed to initialize remote database: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void initializeTables() throws SQLException {
        String createTable = "    CREATE TABLE IF NOT EXISTS player_auth (\n        username VARCHAR(16) PRIMARY KEY,\n        uuid VARCHAR(36) NOT NULL,\n        last_ip VARCHAR(45) NOT NULL,\n        last_seen BIGINT NOT NULL,\n        profile_data TEXT NOT NULL\n    )\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createTable);
        }
    }

    public void cacheAuthentication(String username, String ip, JsonObject profile) {
        if (username == null || profile == null) {
            return;
        }
        try {
            String uuid = profile.get("id").getAsString();
            String profileJson = this.gson.toJson((JsonElement)profile);
            long timestamp = System.currentTimeMillis();
            String sql = this.isRemote ? "    INSERT INTO player_auth (username, uuid, last_ip, last_seen, profile_data)\n    VALUES (?, ?, ?, ?, ?)\n    ON DUPLICATE KEY UPDATE uuid = VALUES(uuid), last_ip = VALUES(last_ip),\n    last_seen = VALUES(last_seen), profile_data = VALUES(profile_data)\n" : "    MERGE INTO player_auth (username, uuid, last_ip, last_seen, profile_data)\n    KEY (username)\n    VALUES (?, ?, ?, ?, ?)\n";
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                pstmt.setString(1, username);
                pstmt.setString(2, uuid);
                pstmt.setString(3, ip != null ? ip : "unknown");
                pstmt.setLong(4, timestamp);
                pstmt.setString(5, profileJson);
                pstmt.executeUpdate();
            }
            this.platform.sendLogMessage("Cached authentication for " + username + " (IP: " + ip + ")");
        }
        catch (SQLException e) {
            this.platform.sendWarningLogMessage("Failed to cache authentication: " + e.getMessage());
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String getFallbackAuth(String username, String ip, int maxOfflineHours) {
        if (username == null) {
            return null;
        }
        try {
            String sql = "    SELECT profile_data, last_ip, last_seen\n    FROM player_auth\n    WHERE LOWER(username) = LOWER(?)\n";
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                pstmt.setString(1, username);
                try (ResultSet rs = pstmt.executeQuery();){
                    long hoursSinceLastSeen;
                    if (!rs.next()) return null;
                    String cachedIp = rs.getString("last_ip");
                    long lastSeen = rs.getLong("last_seen");
                    String profileData = rs.getString("profile_data");
                    if (ip != null && !ip.equals("unknown") && !ip.equals(cachedIp)) {
                        this.platform.sendWarningLogMessage("IP mismatch for " + username + " - cached: " + cachedIp + ", current: " + ip);
                        String string = null;
                        return string;
                    }
                    if (maxOfflineHours > 0 && (hoursSinceLastSeen = (System.currentTimeMillis() - lastSeen) / 3600000L) > (long)maxOfflineHours) {
                        this.platform.sendWarningLogMessage("Auth cache expired for " + username + " - last seen " + hoursSinceLastSeen + " hours ago");
                        String string = null;
                        return string;
                    }
                    String string = profileData;
                    return string;
                }
            }
        }
        catch (SQLException e) {
            this.platform.sendWarningLogMessage("Database error during fallback auth: " + e.getMessage());
        }
        return null;
    }

    public CacheStats getStats() {
        CacheStats stats = new CacheStats();
        try {
            String sql = "SELECT COUNT(*) as total FROM player_auth";
            try (Statement stmt = this.connection.createStatement();
                 ResultSet rs = stmt.executeQuery(sql);){
                if (rs.next()) {
                    stats.totalPlayers = rs.getInt("total");
                }
            }
            sql = "    SELECT COUNT(*) as recent\n    FROM player_auth\n    WHERE last_seen > ?\n";
            long oneDayAgo = System.currentTimeMillis() - 86400000L;
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                pstmt.setLong(1, oneDayAgo);
                try (ResultSet rs = pstmt.executeQuery();){
                    if (rs.next()) {
                        stats.recentPlayers = rs.getInt("recent");
                    }
                }
            }
        }
        catch (SQLException e) {
            this.platform.sendWarningLogMessage("Failed to get stats: " + e.getMessage());
        }
        return stats;
    }

    public int cleanOldEntries(int daysOld) {
        int n;
        block8: {
            long cutoffTime = System.currentTimeMillis() - (long)daysOld * 24L * 60L * 60L * 1000L;
            String sql = "DELETE FROM player_auth WHERE last_seen < ?";
            PreparedStatement pstmt = this.connection.prepareStatement(sql);
            try {
                pstmt.setLong(1, cutoffTime);
                int deleted = pstmt.executeUpdate();
                this.platform.sendLogMessage("Cleaned " + deleted + " old entries (older than " + daysOld + " days)");
                n = deleted;
                if (pstmt == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (pstmt != null) {
                        try {
                            pstmt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (SQLException e) {
                    this.platform.sendWarningLogMessage("Failed to clean old entries: " + e.getMessage());
                    return 0;
                }
            }
            pstmt.close();
        }
        return n;
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                this.platform.sendLogMessage("Database connection closed");
            }
        }
        catch (SQLException e) {
            this.platform.sendWarningLogMessage("Error closing database: " + e.getMessage());
        }
    }

    public static class CacheStats {
        public int totalPlayers = 0;
        public int recentPlayers = 0;

        public String toString() {
            return "Total cached players: " + this.totalPlayers + ", Active (24h): " + this.recentPlayers;
        }
    }
}

