package io.github.xrickastley.originsmath.util;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;

import java.util.function.BiConsumer;
import java.util.function.Function;
import net.minecraft.class_1297;
import net.minecraft.class_2540;
import io.github.apace100.apoli.data.ApoliDataTypes;
import io.github.apace100.apoli.power.PowerType;
import io.github.apace100.calio.ClassUtil;
import io.github.apace100.calio.data.SerializableDataType;
import io.github.xrickastley.originsmath.commands.ResourceCommand;

public class ResourceBacked<T extends Number> 
	extends Number 
	implements Comparable<Number>
{
	private final PowerType<?> powerType;
	private final T number;
	private class_1297 targetEntity;

	public static ResourceBacked<Double> fromPowerType(final PowerType<?> powerType) {
		return new ResourceBacked<Double>(powerType);
	}

	public static <T extends Number> ResourceBacked<T> fromPowerType(final PowerType<?> powerType, final Class<T> numberClass) {
		return new ResourceBacked<T>(powerType);
	}

	public static <T extends Number> ResourceBacked<T> fromNumber(final T number) {
		return new ResourceBacked<T>(number);
	}

	private ResourceBacked(final PowerType<?> powerType) {
		this.powerType = powerType;
		this.number = null;
	}

	private ResourceBacked(final T number) {
		this.powerType = null;
		this.number = number;
	}

	private Number getValue() {
		return this.powerType != null
			? this.targetEntity != null
				? ResourceCommand.getAbsoluteValue(powerType.get(targetEntity))
				: 0
			: this.number != null
				? this.number
				: 0;
	}

	public void setTargetEntity(class_1297 entity) {
		this.targetEntity = entity;
	}

	@Override
	public int intValue() {
		return getValue().intValue();
	}

	@Override
	public double doubleValue() {
		return getValue().doubleValue();
	}

	@Override
	public float floatValue() {
		return getValue().floatValue();
	}

	@Override
	public long longValue() {
		return (long) getValue();
	}

	@Override
	public int compareTo(Number o) {
		if (number != null) {
			// Long -> Double loses precision.
			if (number instanceof Long) {
				return compare(number.longValue(), o.longValue());
			// Any other number doesn't lose precision.
			} else {
				return compare(number.doubleValue(), o.doubleValue());
			}
		} else {
			// As seen above, double can accomodate all numbers except long, so we use that.
			return compare(number.doubleValue(), o.doubleValue());
		}
	}

	public int compare(double x, double y) {
		return Double.compare(x, y);
	}

	public int compare(long x, long y) {
		return Long.compare(x, y);
	}

	private static <T extends Number> BiConsumer<class_2540, ResourceBacked<T>> createSendFn(Class<T> numberClass, BiConsumer<class_2540, T> sendToPacket) {
		return (packet, rb) -> {
			if (rb.number != null) {
				packet.writeByte(0);
				sendToPacket.accept(packet, rb.number);
			} else {
				packet.writeByte(1);
				ApoliDataTypes.POWER_TYPE.send(packet, rb.powerType);
			}
		};
	}

	private static <T extends Number> Function<class_2540, ResourceBacked<T>> createReceiveFn(Class<T> numberClass, Function<class_2540, T> receiveFromPacket) {
		return packet -> {
			int type = packet.readByte();

			switch (type) {
				case 0:
					return ResourceBacked.fromNumber(receiveFromPacket.apply(packet));
				case 1:
					PowerType<?> powerType = ApoliDataTypes.POWER_TYPE.receive(packet);

					return ResourceBacked.fromPowerType(powerType, numberClass);
				default:
					throw new UnsupportedOperationException(ResourceBacked.class.getSimpleName());
			}
		};
	}

	private static <T extends Number> Function<JsonElement, ResourceBacked<T>> createReadFn(Class<T> numberClass, Function<JsonPrimitive, T> readFromJson) {
		return json -> {
			if (!(json instanceof JsonPrimitive jsonPrimitive)) throw new UnsupportedOperationException(JsonElement.class.getSimpleName());

			if (jsonPrimitive.isNumber()) {
				return ResourceBacked.fromNumber(readFromJson.apply(jsonPrimitive));
			} else {
				PowerType<?> powerType = ApoliDataTypes.POWER_TYPE.read(json);

				return ResourceBacked.fromPowerType(powerType, numberClass);
			}
		};
	}

	public static interface DataTypes {
		public static SerializableDataType<ResourceBacked<Integer>> RESOURCE_BACKED_INT = new SerializableDataType<>(
			ClassUtil.castClass(ResourceBacked.class),
			ResourceBacked.createSendFn(Integer.class, class_2540::writeInt),
			ResourceBacked.createReceiveFn(Integer.class, class_2540::readInt),
			ResourceBacked.createReadFn(Integer.class, JsonPrimitive::getAsInt)
		);

		public static SerializableDataType<ResourceBacked<Float>> RESOURCE_BACKED_FLOAT = new SerializableDataType<>(
			ClassUtil.castClass(ResourceBacked.class),
			ResourceBacked.createSendFn(Float.class, class_2540::writeFloat),
			ResourceBacked.createReceiveFn(Float.class, class_2540::readFloat),
			ResourceBacked.createReadFn(Float.class, JsonPrimitive::getAsFloat)
		);

		public static SerializableDataType<ResourceBacked<Double>> RESOURCE_BACKED_DOUBLE = new SerializableDataType<>(
			ClassUtil.castClass(ResourceBacked.class),
			ResourceBacked.createSendFn(Double.class, class_2540::writeDouble),
			ResourceBacked.createReceiveFn(Double.class, class_2540::readDouble),
			ResourceBacked.createReadFn(Double.class, JsonPrimitive::getAsDouble)
		);
	}
}
