package com.nerjal.status_hider;

import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class IpWhitelist extends FileBound {
    private final Set<String> whitelist;

    private IpWhitelist(Set<String> set, Path path) {
        super(path);
        this.whitelist = new HashSet<>(set);
    }

    public Collection<String> listPatterns() {
        return this.whitelist.stream().sorted().toList();
    }

    public boolean isWhitelisted(SocketAddress address) {
        String ip = StatusHider.cleanIp(address.toString());
        return isWhitelisted(ip);
    }

    public boolean isWhitelisted(String address) {
        synchronized (this.whitelist) {
            return this.whitelist.stream().anyMatch(address::matches);
        }
    }

    public boolean whitelist(String regex) {
        if (this.whitelist.add(regex)) {
            this.save();
            return true;
        }
        return false;
    }

    public boolean unWhitelist(String regex) {
        if (this.whitelist.remove(regex)) {
            this.save();
            return true;
        }
        return false;
    }

    @Override
    public void save() {
        this.fileLock.lock();
        try (FileWriter writer = new FileWriter(this.path.toFile())) {
            writer.write("");
            synchronized (this.whitelist) {
                int i = this.whitelist.size();
                for (String s : this.whitelist) {
                    writer.append(s);
                    if (--i != 0) writer.append('\n');
                }
            }
            writer.flush();
        } catch (IOException e) {
            //
        } finally {
            this.fileLock.unlock();
        }
    }

    public static IpWhitelist loadOrCreate(Path path) {
        if (Files.exists(path)) {
            return loadFile(path).or(() -> Optional.of(createFile(path)))
                    .map(set -> new IpWhitelist(set, path)).get();
        }
        return new IpWhitelist(createFile(path), path);
    }

    private static Optional<Set<String>> loadFile(Path path) {
        try (Stream<String> lines = Files.lines(path)) {
            return Optional.of(lines.collect(Collectors.toSet()));
        } catch (FileNotFoundException e) {
            return Optional.empty();
        } catch (IOException e) {
            StatusHider.LOGGER.error(e);
            return Optional.empty();
        }
    }

    private static Set<String> createFile(Path path) {
        try {
            Files.createFile(path);
            return new HashSet<>();
        } catch (IOException e) {
            throw new RuntimeException("Unable to create IP whitelist file "+path, e);
        }
    }
}
