package com.nerjal.status_hider;

import org.jetbrains.annotations.NotNull;

import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import net.minecraft.class_2535;

public class RequestMetrics extends FileBound {
    record Entry(int ipHash, int pingAttempts, int loginAttempts, int bannedAttempts) {
        @Override
        public @NotNull String toString() {
            return "{ipHash=" + ipHash +
                    ", pingAttempts=" + pingAttempts +
                    ", loginAttempts=" + loginAttempts +
                    ", bannedAttempts=" + bannedAttempts +
                    '}';
        }
    }

    private final Map<Integer, Entry> metrics;

    private RequestMetrics(Map<Integer, Entry> map, Path path) {
        super(path);
        this.metrics = new HashMap<>(map);
    }

    public int incrementPingAndGet(class_2535 connection) {
        return incrementPingAndGet(connection.method_10755());
    }

    public int incrementPingAndGet(SocketAddress address) {
        return incrementPingAndGet(address.toString());
    }

    public int incrementPingAndGet(String address) {
        return incrementPingAndGet(StatusHider.getHash(address));
    }

    public int incrementPingAndGet(int ipHash) {
        int r;
        synchronized (this.metrics) {
            r = this.metrics.compute(ipHash, (i, e) -> {
                if (e == null) return new Entry(i, 1, 0, 0);
                return new Entry(i, e.pingAttempts + 1, e.loginAttempts, e.bannedAttempts);
            }).pingAttempts;
        }
        this.save();
        return r;
    }

    public int incrementLoginAndGet(int ipHash) {
        int r;
        synchronized (this.metrics) {
            r = this.metrics.compute(ipHash, (i, e) -> {
                if (e == null) return new Entry(i, 0, 1, 0);
                return new Entry(i, e.pingAttempts, e.loginAttempts+1, e.bannedAttempts);
            }).loginAttempts;
        }
        this.save();
        return r;
    }

    public int incrementBannedAndGet(SocketAddress address) {
        return this.incrementBannedAndGet(StatusHider.cleanIp(address.toString()).hashCode());
    }

    public int incrementBannedAndGet(int ipHash) {
        int r;
        synchronized (this.metrics) {
            r = this.metrics.compute(ipHash, (i, e) -> {
                if (e == null) return new Entry(i, 0, 0, 1);
                return new Entry(i, e.pingAttempts, e.loginAttempts, e.bannedAttempts+1);
            }).bannedAttempts;
        }
        return r;
    }

    public int clear(int ipHash) {
        int i = 0;
        synchronized (this.metrics) {
            Entry e = this.metrics.remove(ipHash);
            if (e != null) {
                i = 1;
            }
        }
        if (i != 0) {
            this.save();
        }
        return i;
    }

    @Override
    public void save() {
        this.fileLock.lock();
        try (FileWriter writer = new FileWriter(this.path.toFile())) {
            writer.write(""); // clear file
            Set<Entry> set;
            synchronized (this.metrics) {
                set = new HashSet<>(this.metrics.values());
            }
            int i = set.size();
            for (Entry e : set) {
                writer.append(String.valueOf(e.ipHash)).append(';')
                        .append(String.valueOf(e.pingAttempts)).append(';')
                        .append(String.valueOf(e.loginAttempts)).append(';')
                        .append(String.valueOf(e.bannedAttempts));
                if (--i != 0) writer.append('\n');
            }
            writer.flush();
        } catch (IOException e) {
            //
        }
         finally {
            this.fileLock.unlock();
        }
    }

    public static RequestMetrics loadOrCreate(Path path) {
        if (Files.exists(path)) {
            return loadFile(path).or(() -> Optional.of(createFile(path)))
                    .map(map -> new RequestMetrics(map, path)).get();
        }
        return new RequestMetrics(createFile(path), path);
    }

    private static Optional<Map<Integer, Entry>> loadFile(Path path) {
        try (Stream<String> lines = Files.lines(path)) {
            Map<Integer, Entry> map = new HashMap<>();
            AtomicInteger lineCount = new AtomicInteger();
            lines.forEach((final String line) -> {
                int i = line.indexOf(';');
                if (i < 0 || i >= (line.length() - 5)) { // minimum 3 chars for numbers and 2 times ';'
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: missing or invalid ';' position.\n{}",
                            lineCount.getAndIncrement(), line);
                    return;
                }
                String current = line.substring(0, i);
                int ipHash;
                try {
                    ipHash = Integer.parseInt(current);
                } catch (NumberFormatException e) {
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: couldn't parse IP hash from {}.\n{}",
                            lineCount.getAndIncrement(), current, line
                    );
                    return;
                }
                String sub = line.substring(i+1);
                i = sub.indexOf(';');
                if (i < 0 || i >= (line.length() - 3)) { // minimum 2 chars for numbers and one ";"
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: missing or invalid ':' position.\n{}",
                            lineCount.getAndIncrement(), line
                    );
                    return;
                }
                current = sub.substring(0, i);
                int ping;
                try {
                    ping = Integer.parseInt(current);
                } catch (NumberFormatException e) {
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: couldn't parse ping attempts from {}.\n{}",
                            lineCount.getAndIncrement(), current, line
                    );
                    return;
                }
                sub = sub.substring(i+1);
                i = sub.indexOf(';');
                if (i < 0 || i >= (line.length() - 1)) { // minimum 1 character for last number
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: missing or invalid ':' position.\n{}",
                            lineCount.getAndIncrement(), line
                    );
                    return;
                }
                current = sub.substring(0, i);
                int login;
                try {
                    login = Integer.parseInt(current);
                } catch (NumberFormatException e) {
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: couldn't parse login attempts from {}.\n{}",
                            lineCount.getAndIncrement(), current, line
                    );
                    return;
                }
                sub = sub.substring(i+1);
                int banned;
                try {
                    banned = Integer.parseInt(sub);
                } catch (NumberFormatException e) {
                    StatusHider.LOGGER.warn(
                            "Unable to parse Request Metrics line {}: couldn't parse banned attempts from {}.\n{}",
                            lineCount.getAndIncrement(), current, line
                    );
                    return;
                }
                Entry e = new Entry(ipHash, ping, login, banned);
                if (map.containsKey(ipHash)) {
                    StatusHider.LOGGER.warn(
                            "Duplicate Request Metrics entries for IP hash {} at line {}: {} and {}. Keeping the former.\n{}",
                            ipHash, lineCount.getAndIncrement(), map.get(ipHash), e, line
                    );
                } else {
                    lineCount.getAndIncrement();
                    map.put(ipHash, e);
                }
            });
            return Optional.of(map);
        } catch (FileNotFoundException e) {
            return Optional.empty();
        } catch (IOException e) {
            StatusHider.LOGGER.error(e);
            return Optional.empty();
        }
    }

    private static Map<Integer, Entry> createFile(Path path) {
        try {
            Files.createFile(path);
            return new HashMap<>();
        } catch (IOException e) {
            throw new RuntimeException("Unable to create request metrics file "+path, e);
        }
    }
}
