package com.nerjal.status_hider;

import org.jetbrains.annotations.Nullable;

import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.class_3222;

public class PlayerIpCache extends FileBound {
    record Entry(String username, UUID uuid, int[] ipHashes) {}
    private static final Entry EMPTY = new Entry(null, null, null);

    private final Map<Integer, Entry> cache;
    private final Map<UUID, Entry> uuidMap;
    private final Map<String, Entry> nameMap;

    private PlayerIpCache(Map<Integer, Entry> map, Path path) {
        super(path);
        this.cache = new HashMap<>(map);
        this.uuidMap = new HashMap<>();
        this.nameMap = new HashMap<>();
        map.forEach((i, e) -> {
            uuidMap.putIfAbsent(e.uuid, e);
            nameMap.putIfAbsent(e.username, e);
        });
    }

    public boolean isIpUnknown(int ipHash) {
        return !this.cache.containsKey(ipHash);
    }

    public String getUsernameForIpHash(int ipHash) {
        return this.cache.getOrDefault(ipHash, EMPTY).username;
    }

    public void registerPlayer(int ipHash, class_3222 player) {
        boolean b;
        Entry e;
        synchronized (this.cache) {
            b = this.cache.containsKey(ipHash);
            if (!b) {
                UUID uuid = player.method_5667();
                if (!this.uuidMap.containsKey(uuid)) {
                    e = new Entry(player.method_5477().getString(), uuid, new int[]{ipHash});
                    this.cache.put(ipHash, e);
                    this.uuidMap.put(uuid, e);
                    this.nameMap.put(e.username, e);
                } else if (!StatusHider.arrayContains((e = this.uuidMap.get(uuid)).ipHashes, ipHash)) {
                    int[] arr = Arrays.copyOf(e.ipHashes, e.ipHashes.length+1);
                    arr[e.ipHashes.length] = ipHash;
                    Entry e2 = new Entry(e.username, uuid, arr);
                    this.cache.put(ipHash, e2);
                    this.uuidMap.put(uuid, e2);
                    this.nameMap.put(e.username, e2);
                    for (int i : e.ipHashes) {
                        if (e == this.cache.get(i)) this.cache.put(i, e2);
                    }
                } else {
                    this.uuidMap.put(uuid, e);
                    b = true;
                }
            }
        }
        if (!b) {
            this.save();
        }
    }

    public int registerVirtual(int ipHash, String name) {
        boolean b;
        int r;
        synchronized (this.cache) {
            b = this.cache.containsKey(ipHash);
            boolean b2 = this.nameMap.containsKey(name);
            if (!b && !b2) {
                UUID uuid;
                //noinspection StatementWithEmptyBody
                while (this.uuidMap.containsKey((uuid = UUID.randomUUID()))) {
                    // ignore
                }
                Entry e = new Entry(name, uuid, new int[]{ipHash});
                this.cache.put(ipHash, e);
                this.uuidMap.put(uuid, e);
                this.nameMap.put(name, e);
                r = 2; // new entry created
            } else if (b2) {
                Entry e = this.nameMap.get(name);
                int[] array = Arrays.copyOf(e.ipHashes, e.ipHashes.length+1);
                array[e.ipHashes.length] = ipHash;
                Entry e2 = new Entry(e.username, e.uuid, array);
                for (int i : array) this.cache.put(i, e2);
                this.uuidMap.put(e2.uuid, e2);
                this.nameMap.put(e2.username, e2);
                r = 1; // ip added to existing entry
            } else {
                r = 0; // no changes
            }
        }
        if (!b) {
            this.save();
        }
        return r;
    }

    public int forgetPlayerOrIp(String s, @Nullable class_3222 player) {
        boolean b = false;
        int r = 0;
        if (player == null) {
            int ip = StatusHider.cleanIp(s).hashCode();
            synchronized (this.cache) {
                Entry entry = this.cache.remove(ip);
                if (entry != null) {
                    r++;
                    b = true;
                    for (int i : entry.ipHashes) this.cache.remove(i);
                    this.uuidMap.remove(entry.uuid);
                    this.nameMap.remove(entry.username);
                }
                entry = this.nameMap.remove(s);
                if (entry != null) {
                    r++;
                    b = true;
                    for (int i : entry.ipHashes) this.cache.remove(i);
                    this.uuidMap.remove(entry.uuid);
                }
            }
        } else {
            synchronized (this.cache) {
                Entry entry = this.uuidMap.remove(player.method_5667());
                if (entry != null) {
                    r++;
                    b = true;
                    for (int i : entry.ipHashes) this.cache.remove(i);
                    this.nameMap.remove(entry.username);
                }
            }
        }
        if (b) {
            this.save();
        }
        return r;
    }

    @Override
    public void save() {
        this.fileLock.lock();
        try (FileWriter writer = new FileWriter(this.path.toFile())) {
            writer.write(""); // clear file
            Set<Entry> set;
            synchronized (this.cache) {
                set = new HashSet<>(this.cache.values());
            }
            int i = set.size();
            for (Entry e : set) {
                writer.append(e.username).append(';')
                        .append(e.uuid.toString()).append(';')
                        .append(String.join(",",
                                Arrays.stream(e.ipHashes).mapToObj(String::valueOf).toArray(String[]::new)));
                if (--i != 0) writer.append('\n');
            }
            writer.flush();
        } catch (IOException e) {
            //
        } finally {
            this.fileLock.unlock();
        }
    }

    public static PlayerIpCache loadOrCreate(Path path) {
        if (Files.exists(path)) {
            return loadFile(path).or(() -> Optional.of(createFile(path)))
                    .map(map -> new PlayerIpCache(map, path)).get();
        }
        return new PlayerIpCache(createFile(path), path);
    }

    private static Optional<Map<Integer, Entry>> loadFile(Path path) {
        try (Stream<String> lines = Files.lines(path)) {
            Map<Integer, Entry> map = new HashMap<>();
            AtomicInteger lineCount = new AtomicInteger();
            lines.forEach((final String line) -> {
                int i = line.indexOf(';');
                if (i < 0 || i >= (line.length() - 23)) { // minimum 18 chars for UUID, 2 times ';' and username
                    StatusHider.LOGGER.warn(
                            "Unable to parse IP cache line {}: missing or invalid ';' position.\n{}",
                            lineCount.getAndIncrement(), line);
                    return;
                }
                String username = line.substring(0, i);
                String sub = line.substring(i+1);
                i = sub.indexOf(';');
                if (i < 0 || i >= (line.length() - 19)) { // minimum 18 chars for UUID and ';'
                    StatusHider.LOGGER.warn(
                            "Unable to parse IP cache line {}: missing or invalid second ';' position.\n{}",
                            lineCount.getAndIncrement(), line);
                    return;
                }
                String uuidStr = sub.substring(0, i);
                UUID uuid;
                try {
                    uuid = UUID.fromString(uuidStr);
                } catch (IllegalArgumentException e) {
                    StatusHider.LOGGER.warn(
                            "Unable to parse IP cache line {}: non-conform UUID format {}.\n{}",
                            lineCount.getAndIncrement(), uuidStr, line, e);
                    return;
                }
                sub = sub.substring(i+1);
                String[] hashesStr = sub.split(",");
                Set<Integer> hashesSet = Arrays.stream(hashesStr).map(s -> {
                    try {
                        return Integer.parseInt(s);
                    } catch (NumberFormatException e) {
                        StatusHider.LOGGER.warn(
                                "Error parsing IP cache line {}: couldn't parse IP hash from {}.\n{}",
                                lineCount.get(), s, line);
                        return null;
                    }
                }).collect(Collectors.toSet());
                if (hashesSet.isEmpty()) {
                    StatusHider.LOGGER.warn(
                            "Ignoring IP cache line {}: empty or none parsable IP hashes.\n{}",
                            lineCount.getAndIncrement(), line);
                    return;
                }
                int[] array = new int[hashesSet.size()];
                i = 0;
                for (int j : hashesSet) {
                    array[i++] = j;
                }
                Entry e = new Entry(username, uuid, array);
                for (int j : array) {
                    map.putIfAbsent(j, e);
                }
                lineCount.incrementAndGet();
            });
            return Optional.of(map);
        } catch (FileNotFoundException e) {
            return Optional.empty();
        } catch (IOException e) {
            StatusHider.LOGGER.error(e);
            return Optional.empty();
        }
    }

    private static Map<Integer, Entry> createFile(Path path) {
        try {
            Files.createFile(path);
            return new HashMap<>();
        } catch (IOException e) {
            throw new RuntimeException("Unable to create IP cache file "+path, e);
        }
    }
}
