/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.config;

import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import net.rafalohaki.veloauth.database.DatabaseType;
import net.rafalohaki.veloauth.libs.jackson.core.JsonProcessingException;
import net.rafalohaki.veloauth.libs.jackson.databind.ObjectMapper;
import net.rafalohaki.veloauth.libs.jackson.dataformat.yaml.YAMLFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Settings {
    private static final Logger logger = LoggerFactory.getLogger(Settings.class);
    private final Path dataDirectory;
    private final Path configFile;
    private final ObjectMapper yamlMapper;
    private final PostgreSQLSettings postgreSQLSettings = new PostgreSQLSettings();
    private final PremiumSettings premiumSettings = new PremiumSettings();
    private String databaseStorageType = DatabaseType.H2.getName();
    private String databaseHostname = "localhost";
    private int databasePort = 3306;
    private String databaseName = "veloauth";
    private String databaseUser = "veloauth";
    private String databasePassword = "password";
    private String databaseConnectionUrl = null;
    private String databaseConnectionParameters = "";
    private int databaseConnectionPoolSize = 20;
    private long databaseMaxLifetimeMillis = 1800000L;
    private int cacheTtlMinutes = 60;
    private int cacheMaxSize = 10000;
    private int cacheCleanupIntervalMinutes = 5;
    private String picoLimboServerName = "lobby";
    private int picoLimboTimeoutSeconds = 300;
    private int bcryptCost = 10;
    private int bruteForceMaxAttempts = 5;
    private int bruteForceTimeoutMinutes = 5;
    private int ipLimitRegistrations = 3;
    private int minPasswordLength = 4;
    private int maxPasswordLength = 72;
    private boolean debugEnabled = false;
    private String language = "en";

    public Settings(Path dataDirectory) {
        this.dataDirectory = dataDirectory;
        this.configFile = dataDirectory.resolve("config.yml");
        this.yamlMapper = new ObjectMapper(new YAMLFactory());
        try {
            Files.createDirectories(dataDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            logger.error("Nie uda\u0142o si\u0119 utworzy\u0107 katalogu danych: {}", (Object)dataDirectory, (Object)e);
        }
    }

    public boolean load() {
        try {
            if (!Files.exists(this.configFile, new LinkOption[0])) {
                logger.info("Tworzenie domy\u015blnego pliku konfiguracji: {}", (Object)this.configFile);
                this.createDefaultConfig();
                return true;
            }
            logger.info("\u0141adowanie konfiguracji z: {}", (Object)this.configFile);
            Map config = this.yamlMapper.readValue(this.configFile.toFile(), Map.class);
            this.loadDatabaseSettings(config);
            this.loadCacheSettings(config);
            this.loadPicoLimboSettings(config);
            this.loadSecuritySettings(config);
            this.loadPremiumSettings(config);
            this.loadDebugSettings(config);
            this.loadLanguageSettings(config);
            this.processDatabaseSettings();
            this.validateSettings();
            logger.info("Konfiguracja za\u0142adowana pomy\u015blnie");
            return true;
        }
        catch (JsonProcessingException e) {
            logger.error("B\u0142\u0105d parsowania YAML w pliku konfiguracji: {}", (Object)this.configFile, (Object)e);
            return false;
        }
        catch (IOException e) {
            logger.error("B\u0142\u0105d odczytu pliku konfiguracji: {}", (Object)this.configFile, (Object)e);
            return false;
        }
    }

    private void createDefaultConfig() throws IOException {
        String defaultConfig = "# VeloAuth Configuration\n# Complete Velocity Authentication Plugin\n\n# Language configuration (supported: en, pl)\nlanguage: en # Plugin language: en = English, pl = Polski\n\n# Debug settings (enable for detailed logging)\ndebug-enabled: false # Set to true for development/debugging\n\n# Database storage configuration (supported: H2, MYSQL, POSTGRESQL, SQLITE)\ndatabase:\n  storage-type: H2 # Example: MYSQL, POSTGRESQL, SQLITE\n  hostname: localhost # Database host, e.g. db.example.com\n  port: 3306 # Default ports: MYSQL=3306, POSTGRESQL=5432\n  database: veloauth # Database/schema name\n  user: veloauth # Database user\n  password: password # Strong password recommended\n  connection-pool-size: 20 # Maximum pooled connections\n  max-lifetime-millis: 1800000 # Connection max lifetime in milliseconds (30 minutes)\n  # Optional: Full database connection URL\n  # If set, will be used instead of individual parameters\n  # Examples:\n  #   postgresql://user:pass@host:5432/database\n  #   postgresql://user:pass@host:5432/database?sslmode=disable\n  #   mysql://user:pass@host:3306/database\n  connection-url: \"\"\n  # Optional: Additional connection parameters\n  # Example: \"?autoReconnect=true&initialTimeout=1&useSSL=false&serverTimezone=UTC\"\n  connection-parameters: \"\"\n\n  # PostgreSQL-specific configuration (used when storage-type is POSTGRESQL)\n  postgresql:\n    # Enable SSL connection to PostgreSQL\n    ssl-enabled: false\n    # SSL mode: disable, allow, prefer, require, verify-ca, verify-full\n    ssl-mode: \"prefer\"\n    # Path to SSL certificate file (optional)\n    ssl-cert: \"\"\n    # Path to SSL key file (optional)\n    ssl-key: \"\"\n    # Path to SSL root certificate file (optional)\n    ssl-root-cert: \"\"\n    # SSL password for key file (optional)\n    ssl-password: \"\"\n\n# Authentication cache configuration\ncache:\n  ttl-minutes: 60 # Cache entry lifetime\n  max-size: 10000 # Maximum cached records\n  cleanup-interval-minutes: 5 # Cleanup scheduler interval\n\n# PicoLimbo integration (fallback server for unauthenticated players)\npicolimbo:\n  server-name: lobby # Registered Velocity server name\n  timeout-seconds: 300 # Kick timeout for PicoLimbo\n\n# Security settings for password hashing and brute-force protection\nsecurity:\n  bcrypt-cost: 10 # BCrypt hashing rounds (4-31)\n  bruteforce-max-attempts: 5 # Attempts before temporary block\n  bruteforce-timeout-minutes: 5 # Block duration in minutes\n  ip-limit-registrations: 3 # Account registrations per IP\n  min-password-length: 4 # Inclusive minimum password length\n  max-password-length: 72 # Inclusive maximum password length\n\n# Premium account detection configuration\npremium:\n  check-enabled: true # Enable premium account verification\n  online-mode-need-auth: false # Force auth for premium players on online-mode proxies\n  resolver:\n    mojang-enabled: true # Query Mojang API\n    ashcon-enabled: true # Query Ashcon API\n    wpme-enabled: false # Query WPME API\n    request-timeout-ms: 400 # Per-request timeout in milliseconds\n    hit-ttl-minutes: 10 # Cache TTL for positive hits\n    miss-ttl-minutes: 3 # Cache TTL for misses\n    case-sensitive: true # Preserve username case in resolver cache\n";
        Files.writeString(this.configFile, (CharSequence)defaultConfig, new OpenOption[0]);
        logger.info("Utworzono domy\u015blny plik konfiguracji");
    }

    private void loadDatabaseSettings(Map<String, Object> config) {
        Map database = (Map)config.get("database");
        if (database != null) {
            this.databaseStorageType = this.getString(database, "storage-type", this.databaseStorageType);
            this.databaseHostname = this.getString(database, "hostname", this.databaseHostname);
            this.databasePort = this.getInt(database, "port", this.databasePort);
            this.databaseName = this.getString(database, "database", this.databaseName);
            this.databaseUser = this.getString(database, "user", this.databaseUser);
            this.databasePassword = this.getString(database, "password", this.databasePassword);
            this.databaseConnectionUrl = this.getString(database, "connection-url", this.databaseConnectionUrl);
            this.databaseConnectionParameters = this.getString(database, "connection-parameters", this.databaseConnectionParameters);
            this.databaseConnectionPoolSize = this.getInt(database, "connection-pool-size", this.databaseConnectionPoolSize);
            this.databaseMaxLifetimeMillis = this.getLong(database, "max-lifetime-millis", this.databaseMaxLifetimeMillis);
            this.loadPostgreSQLSettings(database);
        }
    }

    private void loadPostgreSQLSettings(Map<String, Object> database) {
        Object postgreSQLSection = database.get("postgresql");
        if (postgreSQLSection instanceof Map) {
            Map postgreSQL = (Map)postgreSQLSection;
            this.postgreSQLSettings.setSslEnabled(this.getBoolean(postgreSQL, "ssl-enabled", this.postgreSQLSettings.isSslEnabled()));
            this.postgreSQLSettings.setSslMode(this.getString(postgreSQL, "ssl-mode", this.postgreSQLSettings.getSslMode()));
            this.postgreSQLSettings.setSslCert(this.getString(postgreSQL, "ssl-cert", this.postgreSQLSettings.getSslCert()));
            this.postgreSQLSettings.setSslKey(this.getString(postgreSQL, "ssl-key", this.postgreSQLSettings.getSslKey()));
            this.postgreSQLSettings.setSslRootCert(this.getString(postgreSQL, "ssl-root-cert", this.postgreSQLSettings.getSslRootCert()));
            this.postgreSQLSettings.setSslPassword(this.getString(postgreSQL, "ssl-password", this.postgreSQLSettings.getSslPassword()));
        }
    }

    private void loadDebugSettings(Map<String, Object> config) {
        this.debugEnabled = this.getBoolean(config, "debug-enabled", this.debugEnabled);
    }

    private void loadLanguageSettings(Map<String, Object> config) {
        this.language = this.getString(config, "language", this.language);
    }

    private void processDatabaseSettings() {
        if (this.databaseConnectionUrl != null && !this.databaseConnectionUrl.trim().isEmpty()) {
            this.parseConnectionUrl(this.databaseConnectionUrl);
        }
    }

    private void parseConnectionUrl(String connectionUrl) {
        try {
            String url = connectionUrl.trim();
            this.databaseStorageType = this.detectDatabaseType(url);
            if (this.databaseStorageType == null) {
                return;
            }
            String remaining = url.substring(url.indexOf("://") + 3);
            this.parseConnectionCredentials(remaining);
        }
        catch (StringIndexOutOfBoundsException e) {
            logger.error("Nieprawid\u0142owy format URL po\u0142\u0105czenia z baz\u0105 danych: {}", (Object)connectionUrl, (Object)e);
        }
        catch (IllegalArgumentException e) {
            logger.error("Nieprawid\u0142owe parametry URL po\u0142\u0105czenia: {}", (Object)connectionUrl, (Object)e);
        }
    }

    private String detectDatabaseType(String url) {
        DatabaseType dbType = DatabaseType.fromUrl(url);
        return dbType != null ? dbType.getName() : null;
    }

    private void parseConnectionCredentials(String remaining) {
        String[] parts = remaining.split("@");
        if (parts.length == 2) {
            String authPart = parts[0];
            String hostPart = parts[1];
            this.parseAuthPart(authPart);
            this.parseHostPart(hostPart);
            logger.info("Parsed connection URL: {}@{}:{}/{}", this.databaseUser, this.databaseHostname, this.databasePort, this.databaseName);
        }
    }

    private void parseAuthPart(String authPart) {
        String[] authSplit = authPart.split(":");
        if (authSplit.length >= 1) {
            this.databaseUser = URLDecoder.decode(authSplit[0], StandardCharsets.UTF_8);
        }
        if (authSplit.length >= 2) {
            this.databasePassword = URLDecoder.decode(authSplit[1], StandardCharsets.UTF_8);
        }
    }

    private void parseHostPart(String hostPart) {
        String[] hpSplit;
        String[] hostSplit = hostPart.split("/");
        String hostAndPort = hostSplit[0];
        if (hostSplit.length >= 2) {
            this.databaseName = hostSplit[1];
        }
        if ((hpSplit = hostAndPort.split(":")).length >= 1) {
            this.databaseHostname = hpSplit[0];
        }
        if (hpSplit.length >= 2) {
            try {
                this.databasePort = Integer.parseInt(hpSplit[1]);
            }
            catch (NumberFormatException e) {
                logger.warn("Invalid port in connection URL: {}", (Object)hpSplit[1]);
            }
        }
    }

    private void loadCacheSettings(Map<String, Object> config) {
        Map cache = (Map)config.get("cache");
        if (cache != null) {
            this.cacheTtlMinutes = this.getInt(cache, "ttl-minutes", this.cacheTtlMinutes);
            this.cacheMaxSize = this.getInt(cache, "max-size", this.cacheMaxSize);
            this.cacheCleanupIntervalMinutes = this.getInt(cache, "cleanup-interval-minutes", this.cacheCleanupIntervalMinutes);
        }
    }

    private void loadPicoLimboSettings(Map<String, Object> config) {
        Map picolimbo = (Map)config.get("picolimbo");
        if (picolimbo != null) {
            this.picoLimboServerName = this.getString(picolimbo, "server-name", this.picoLimboServerName);
            this.picoLimboTimeoutSeconds = this.getInt(picolimbo, "timeout-seconds", this.picoLimboTimeoutSeconds);
        }
    }

    private void loadSecuritySettings(Map<String, Object> config) {
        Map security = (Map)config.get("security");
        if (security != null) {
            this.bcryptCost = this.getInt(security, "bcrypt-cost", this.bcryptCost);
            this.bruteForceMaxAttempts = this.getInt(security, "bruteforce-max-attempts", this.bruteForceMaxAttempts);
            this.bruteForceTimeoutMinutes = this.getInt(security, "bruteforce-timeout-minutes", this.bruteForceTimeoutMinutes);
            this.ipLimitRegistrations = this.getInt(security, "ip-limit-registrations", this.ipLimitRegistrations);
            this.minPasswordLength = this.getInt(security, "min-password-length", this.minPasswordLength);
            this.maxPasswordLength = this.getInt(security, "max-password-length", this.maxPasswordLength);
        }
    }

    private void loadPremiumSettings(Map<String, Object> config) {
        Object premiumSection = config.get("premium");
        if (premiumSection instanceof Map) {
            Object resolverSection;
            Map premium = (Map)premiumSection;
            this.premiumSettings.setCheckEnabled(this.getBoolean(premium, "check-enabled", this.premiumSettings.isCheckEnabled()));
            this.premiumSettings.setOnlineModeNeedAuth(this.getBoolean(premium, "online-mode-need-auth", this.premiumSettings.isOnlineModeNeedAuth()));
            if (premium.containsKey("premium-uuid-resolver")) {
                logger.warn("Wykryto legacy klucz premium.premium-uuid-resolver \u2013 ignoruj\u0119. Skonfiguruj premium.resolver.*");
            }
            if ((resolverSection = premium.get("resolver")) instanceof Map) {
                Map resolver = (Map)resolverSection;
                this.applyResolverSettings(resolver);
            }
        }
        if (config.containsKey("premium-resolver")) {
            logger.warn("Wykryto legacy sekcj\u0119 premium-resolver \u2013 ignoruj\u0119. Skonfiguruj premium.resolver.*");
        }
    }

    private void applyResolverSettings(Map<String, Object> resolver) {
        PremiumResolverSettings target = this.premiumSettings.getResolver();
        target.setMojangEnabled(this.getBoolean(resolver, "mojang-enabled", target.isMojangEnabled()));
        target.setAshconEnabled(this.getBoolean(resolver, "ashcon-enabled", target.isAshconEnabled()));
        target.setWpmeEnabled(this.getBoolean(resolver, "wpme-enabled", target.isWpmeEnabled()));
        target.setRequestTimeoutMs(this.getInt(resolver, "request-timeout-ms", target.getRequestTimeoutMs()));
        target.setHitTtlMinutes(this.getInt(resolver, "hit-ttl-minutes", target.getHitTtlMinutes()));
        target.setMissTtlMinutes(this.getInt(resolver, "miss-ttl-minutes", target.getMissTtlMinutes()));
        target.setCaseSensitive(this.getBoolean(resolver, "case-sensitive", target.isCaseSensitive()));
    }

    private void validateSettings() {
        this.validateDatabaseType();
        this.validateConnectionSettings();
        this.validateCacheSettings();
        this.validateSecuritySettings();
        this.validateConnectionPoolSettings();
        this.validatePicoLimboSettings();
        this.validateServerSettings();
        this.validateLanguageSettings();
    }

    private void validateDatabaseType() {
        DatabaseType dbType = DatabaseType.fromName(this.databaseStorageType);
        if (dbType == null) {
            throw new IllegalArgumentException("Nieobs\u0142ugiwany typ bazy danych: " + this.databaseStorageType);
        }
    }

    private void validateConnectionSettings() {
        if (this.databasePort <= 0 || this.databasePort > 65535) {
            throw new IllegalArgumentException("Port bazy danych musi by\u0107 w zakresie 1-65535");
        }
    }

    private void validateCacheSettings() {
        if (this.cacheTtlMinutes < 0) {
            throw new IllegalArgumentException("Cache TTL nie mo\u017ce by\u0107 ujemny");
        }
        if (this.cacheMaxSize <= 0) {
            throw new IllegalArgumentException("Cache max size musi by\u0107 > 0");
        }
        if (this.cacheCleanupIntervalMinutes <= 0) {
            throw new IllegalArgumentException("Cache cleanup interval musi by\u0107 > 0");
        }
    }

    private void validateSecuritySettings() {
        if (this.bcryptCost < 4 || this.bcryptCost > 31) {
            throw new IllegalArgumentException("BCrypt cost musi by\u0107 w zakresie 4-31");
        }
        if (this.bruteForceMaxAttempts <= 0) {
            throw new IllegalArgumentException("Brute force max attempts musi by\u0107 > 0");
        }
        if (this.bruteForceTimeoutMinutes <= 0) {
            throw new IllegalArgumentException("Brute force timeout musi by\u0107 > 0");
        }
        if (this.minPasswordLength <= 0) {
            throw new IllegalArgumentException("Min password length musi by\u0107 > 0");
        }
        if (this.maxPasswordLength <= this.minPasswordLength) {
            throw new IllegalArgumentException("Max password length musi by\u0107 > min password length");
        }
        if (this.maxPasswordLength > 72) {
            logger.warn("Max password length > 72 (BCrypt limit). Ustawianie na 72.");
            this.adjustMaxPasswordLength();
        }
    }

    private void adjustMaxPasswordLength() {
        this.maxPasswordLength = 72;
    }

    private void validateConnectionPoolSettings() {
        if (this.databaseConnectionPoolSize <= 0) {
            throw new IllegalArgumentException("Connection pool size musi by\u0107 > 0");
        }
    }

    private void validatePicoLimboSettings() {
        if (this.picoLimboTimeoutSeconds <= 0) {
            throw new IllegalArgumentException("PicoLimbo timeout musi by\u0107 > 0");
        }
    }

    private void validateServerSettings() {
        if (this.ipLimitRegistrations <= 0) {
            throw new IllegalArgumentException("IP limit registrations musi by\u0107 > 0");
        }
        if (this.picoLimboServerName == null || this.picoLimboServerName.trim().isEmpty()) {
            throw new IllegalArgumentException("PicoLimbo server name nie mo\u017ce by\u0107 pusty");
        }
        PremiumResolverSettings resolver = this.premiumSettings.getResolver();
        if (!(resolver.isMojangEnabled() || resolver.isAshconEnabled() || resolver.isWpmeEnabled())) {
            throw new IllegalArgumentException("Premium resolver: co najmniej jedno \u017ar\u00f3d\u0142o (mojang/ashcon/wpme) musi by\u0107 w\u0142\u0105czone");
        }
        if (resolver.getRequestTimeoutMs() <= 0) {
            throw new IllegalArgumentException("Premium resolver: request-timeout-ms musi by\u0107 > 0");
        }
        if (resolver.getHitTtlMinutes() < 0 || resolver.getMissTtlMinutes() < 0) {
            throw new IllegalArgumentException("Premium resolver: TTL w minutach nie mog\u0105 by\u0107 ujemne");
        }
        logger.debug("Walidacja konfiguracji zako\u0144czona pomy\u015blnie");
    }

    private void validateLanguageSettings() {
        if (this.language == null || this.language.trim().isEmpty()) {
            logger.warn("Language setting is empty, using default 'en'");
            this.language = "en";
            return;
        }
        String normalizedLang = this.language.toLowerCase().trim();
        if (!"en".equals(normalizedLang) && !"pl".equals(normalizedLang)) {
            logger.warn("Unsupported language '{}', using default 'en'. Supported languages: en, pl", (Object)this.language);
            this.language = "en";
        } else {
            this.language = normalizedLang;
        }
    }

    private String getString(Map<String, Object> map, String key, String defaultValue) {
        Object value = map.get(key);
        return value != null ? value.toString() : defaultValue;
    }

    private int getInt(Map<String, Object> map, String key, int defaultValue) {
        Object value = map.get(key);
        if (value instanceof Number) {
            Number number = (Number)value;
            return number.intValue();
        }
        return defaultValue;
    }

    private long getLong(Map<String, Object> map, String key, long defaultValue) {
        Object value = map.get(key);
        if (value instanceof Number) {
            Number number = (Number)value;
            return number.longValue();
        }
        return defaultValue;
    }

    private boolean getBoolean(Map<String, Object> map, String key, boolean defaultValue) {
        Object value = map.get(key);
        if (value instanceof Boolean) {
            Boolean bool = (Boolean)value;
            return bool;
        }
        return defaultValue;
    }

    public String getDatabaseStorageType() {
        return this.databaseStorageType;
    }

    public String getDatabaseHostname() {
        return this.databaseHostname;
    }

    public int getDatabasePort() {
        return this.databasePort;
    }

    public String getDatabaseName() {
        return this.databaseName;
    }

    public String getDatabaseUser() {
        return this.databaseUser;
    }

    public String getDatabasePassword() {
        return this.databasePassword;
    }

    public String getDatabaseConnectionUrl() {
        return this.databaseConnectionUrl;
    }

    public String getDatabaseConnectionParameters() {
        return this.databaseConnectionParameters;
    }

    public int getDatabaseConnectionPoolSize() {
        return this.databaseConnectionPoolSize;
    }

    public long getDatabaseMaxLifetimeMillis() {
        return this.databaseMaxLifetimeMillis;
    }

    public PostgreSQLSettings getPostgreSQLSettings() {
        return this.postgreSQLSettings;
    }

    public int getCacheTtlMinutes() {
        return this.cacheTtlMinutes;
    }

    public int getCacheMaxSize() {
        return this.cacheMaxSize;
    }

    public int getCacheCleanupIntervalMinutes() {
        return this.cacheCleanupIntervalMinutes;
    }

    public String getPicoLimboServerName() {
        return this.picoLimboServerName;
    }

    public int getPicoLimboTimeoutSeconds() {
        return this.picoLimboTimeoutSeconds;
    }

    public int getBcryptCost() {
        return this.bcryptCost;
    }

    public int getBruteForceMaxAttempts() {
        return this.bruteForceMaxAttempts;
    }

    public int getBruteForceTimeoutMinutes() {
        return this.bruteForceTimeoutMinutes;
    }

    public int getIpLimitRegistrations() {
        return this.ipLimitRegistrations;
    }

    public int getMinPasswordLength() {
        return this.minPasswordLength;
    }

    public int getMaxPasswordLength() {
        return this.maxPasswordLength;
    }

    public boolean isPremiumCheckEnabled() {
        return this.premiumSettings.isCheckEnabled();
    }

    public boolean isOnlineModeNeedAuth() {
        return this.premiumSettings.isOnlineModeNeedAuth();
    }

    public PremiumResolverSettings getPremiumResolverSettings() {
        return this.premiumSettings.getResolver();
    }

    public PremiumSettings getPremiumSettings() {
        return this.premiumSettings;
    }

    public boolean isDebugEnabled() {
        return this.debugEnabled;
    }

    public Path getDataDirectory() {
        return this.dataDirectory;
    }

    public Path getConfigFile() {
        return this.configFile;
    }

    public String getLanguage() {
        return this.language;
    }

    public static class PostgreSQLSettings {
        private boolean sslEnabled = true;
        private String sslMode = "require";
        private String sslCert = "";
        private String sslKey = "";
        private String sslRootCert = "";
        private String sslPassword = "";

        public boolean isSslEnabled() {
            return this.sslEnabled;
        }

        void setSslEnabled(boolean value) {
            this.sslEnabled = value;
        }

        public String getSslMode() {
            return this.sslMode;
        }

        void setSslMode(String value) {
            this.sslMode = value;
        }

        public String getSslCert() {
            return this.sslCert;
        }

        void setSslCert(String value) {
            this.sslCert = value;
        }

        public String getSslKey() {
            return this.sslKey;
        }

        void setSslKey(String value) {
            this.sslKey = value;
        }

        public String getSslRootCert() {
            return this.sslRootCert;
        }

        void setSslRootCert(String value) {
            this.sslRootCert = value;
        }

        public String getSslPassword() {
            return this.sslPassword;
        }

        void setSslPassword(String value) {
            this.sslPassword = value;
        }
    }

    public static class PremiumSettings {
        private final PremiumResolverSettings resolver = new PremiumResolverSettings();
        private boolean checkEnabled = true;
        private boolean onlineModeNeedAuth = false;

        public boolean isCheckEnabled() {
            return this.checkEnabled;
        }

        void setCheckEnabled(boolean value) {
            this.checkEnabled = value;
        }

        public boolean isOnlineModeNeedAuth() {
            return this.onlineModeNeedAuth;
        }

        void setOnlineModeNeedAuth(boolean value) {
            this.onlineModeNeedAuth = value;
        }

        public PremiumResolverSettings getResolver() {
            return this.resolver;
        }
    }

    public static class PremiumResolverSettings {
        private boolean mojangEnabled = true;
        private boolean ashconEnabled = true;
        private boolean wpmeEnabled = false;
        private int requestTimeoutMs = 400;
        private int hitTtlMinutes = 10;
        private int missTtlMinutes = 3;
        private boolean caseSensitive = true;

        public boolean isMojangEnabled() {
            return this.mojangEnabled;
        }

        void setMojangEnabled(boolean value) {
            this.mojangEnabled = value;
        }

        public boolean isAshconEnabled() {
            return this.ashconEnabled;
        }

        void setAshconEnabled(boolean value) {
            this.ashconEnabled = value;
        }

        public boolean isWpmeEnabled() {
            return this.wpmeEnabled;
        }

        void setWpmeEnabled(boolean value) {
            this.wpmeEnabled = value;
        }

        public int getRequestTimeoutMs() {
            return this.requestTimeoutMs;
        }

        void setRequestTimeoutMs(int value) {
            this.requestTimeoutMs = value;
        }

        public int getHitTtlMinutes() {
            return this.hitTtlMinutes;
        }

        void setHitTtlMinutes(int value) {
            this.hitTtlMinutes = value;
        }

        public int getMissTtlMinutes() {
            return this.missTtlMinutes;
        }

        void setMissTtlMinutes(int value) {
            this.missTtlMinutes = value;
        }

        public boolean isCaseSensitive() {
            return this.caseSensitive;
        }

        void setCaseSensitive(boolean value) {
            this.caseSensitive = value;
        }
    }
}

