/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.database;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.util.Objects;
import javax.sql.DataSource;
import net.rafalohaki.veloauth.config.Settings;
import net.rafalohaki.veloauth.database.DatabaseType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DatabaseConfig {
    private static final Logger logger = LoggerFactory.getLogger(DatabaseConfig.class);
    private final String storageType;
    private final String hostname;
    private final int port;
    private final String database;
    private final String user;
    private final String password;
    private final int connectionPoolSize;
    private final DataSource dataSource;
    private final String jdbcUrl;

    public DatabaseConfig(String storageType, String hostname, int port, String database, String user, String password, int connectionPoolSize) {
        if (storageType == null || storageType.isEmpty()) {
            throw new IllegalArgumentException("StorageType nie mo\u017ce by\u0107 pusty");
        }
        if (database == null || database.isEmpty()) {
            throw new IllegalArgumentException("Database nie mo\u017ce by\u0107 pusty");
        }
        if (connectionPoolSize <= 0) {
            throw new IllegalArgumentException("ConnectionPoolSize musi by\u0107 > 0");
        }
        DatabaseType dbType = DatabaseType.fromName(storageType);
        if (dbType == null) {
            throw new IllegalArgumentException("Nieobs\u0142ugiwany typ bazy danych: " + storageType);
        }
        if (dbType.isRemoteDatabase()) {
            if (hostname == null || hostname.isEmpty()) {
                throw new IllegalArgumentException("Hostname nie mo\u017ce by\u0107 pusty dla " + storageType);
            }
            if (port <= 0 || port > 65535) {
                throw new IllegalArgumentException("Port musi by\u0107 w zakresie 1-65535");
            }
        }
        this.storageType = dbType.getName();
        this.hostname = hostname;
        this.port = port;
        this.database = database;
        this.user = user;
        this.password = password;
        this.connectionPoolSize = connectionPoolSize;
        this.dataSource = null;
        this.jdbcUrl = DatabaseConfig.buildJdbcUrl(dbType, hostname, port, database, null, null);
    }

    public DatabaseConfig(String storageType, DataSource dataSource, int connectionPoolSize, String jdbcUrl) {
        if (storageType == null || storageType.isEmpty()) {
            throw new IllegalArgumentException("StorageType nie mo\u017ce by\u0107 pusty");
        }
        if (dataSource == null) {
            throw new IllegalArgumentException("DataSource nie mo\u017ce by\u0107 null");
        }
        if (jdbcUrl == null || jdbcUrl.isEmpty()) {
            throw new IllegalArgumentException("jdbcUrl nie mo\u017ce by\u0107 pusty");
        }
        this.storageType = storageType.toUpperCase();
        this.dataSource = dataSource;
        this.connectionPoolSize = connectionPoolSize;
        this.jdbcUrl = jdbcUrl;
        this.hostname = null;
        this.port = 0;
        this.database = null;
        this.user = null;
        this.password = null;
    }

    public static DatabaseConfig forLocalDatabase(String storageType, String database) {
        return new DatabaseConfig(storageType, null, 0, database, null, null, 1);
    }

    public static DatabaseConfig forRemoteWithHikari(String storageType, String hostname, int port, String database, String user, String password, int connectionPoolSize, int maxLifetime, String connectionParameters, Settings.PostgreSQLSettings postgreSQLSettings, boolean debugEnabled) {
        DatabaseType dbType = DatabaseType.fromName(storageType);
        if (dbType == null) {
            throw new IllegalArgumentException("Nieobs\u0142ugiwany typ bazy danych: " + storageType);
        }
        String jdbcUrl = DatabaseConfig.buildJdbcUrl(dbType, hostname, port, database, connectionParameters, postgreSQLSettings);
        String driverClass = DatabaseConfig.resolveDriverClass(dbType);
        if (debugEnabled) {
            logger.debug("[VeloAuth DEBUG] JDBC URL: {}", (Object)jdbcUrl);
            logger.debug("[VeloAuth DEBUG] Database Type: {}", (Object)dbType);
            logger.debug("[VeloAuth DEBUG] Hostname: {}:{}", (Object)hostname, (Object)port);
            logger.debug("[VeloAuth DEBUG] SSL Settings: {}", (Object)(postgreSQLSettings != null ? "enabled" : "using defaults"));
        }
        HikariConfig hikariConfig = new HikariConfig();
        DatabaseConfig.configureBasicHikariSettings(hikariConfig, jdbcUrl, user, password, connectionPoolSize, maxLifetime);
        DatabaseConfig.configureDatabaseOptimizations(hikariConfig, dbType, postgreSQLSettings);
        hikariConfig.setDriverClassName(driverClass);
        try {
            Class.forName(driverClass);
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("Nie znaleziono sterownika JDBC: " + driverClass, e);
        }
        HikariDataSource dataSource = new HikariDataSource(hikariConfig);
        return new DatabaseConfig(storageType, dataSource, connectionPoolSize, jdbcUrl);
    }

    private static void configureBasicHikariSettings(HikariConfig hikariConfig, String jdbcUrl, String user, String password, int connectionPoolSize, int maxLifetime) {
        hikariConfig.setJdbcUrl(jdbcUrl);
        hikariConfig.setUsername(user);
        hikariConfig.setPassword(password);
        hikariConfig.setMaximumPoolSize(connectionPoolSize);
        hikariConfig.setMinimumIdle(Math.max(5, connectionPoolSize / 4));
        hikariConfig.setMaxLifetime(maxLifetime);
        hikariConfig.setConnectionTimeout(30000L);
        hikariConfig.setIdleTimeout(600000L);
        hikariConfig.setAutoCommit(true);
        hikariConfig.setPoolName("VeloAuth-HikariCP");
        hikariConfig.addDataSourceProperty("socketTimeout", "30");
        hikariConfig.addDataSourceProperty("connectTimeout", "15");
        hikariConfig.setValidationTimeout(10000L);
        hikariConfig.setLeakDetectionThreshold(60000L);
    }

    private static void configureDatabaseOptimizations(HikariConfig hikariConfig, DatabaseType dbType, Settings.PostgreSQLSettings postgreSQLSettings) {
        if (dbType == DatabaseType.MYSQL) {
            hikariConfig.addDataSourceProperty("cachePrepStmts", "true");
            hikariConfig.addDataSourceProperty("prepStmtCacheSize", "250");
            hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
            hikariConfig.addDataSourceProperty("useServerPrepStmts", "true");
        } else if (dbType == DatabaseType.POSTGRESQL) {
            hikariConfig.addDataSourceProperty("prepareThreshold", "1");
            hikariConfig.addDataSourceProperty("preparedStatementCacheQueries", "256");
            hikariConfig.addDataSourceProperty("preparedStatementCacheSizeMiB", "5");
            DatabaseConfig.configurePostgreSQLSsl(hikariConfig, postgreSQLSettings);
        }
    }

    private static void configurePostgreSQLSsl(HikariConfig hikariConfig, Settings.PostgreSQLSettings postgreSQLSettings) {
        if (postgreSQLSettings != null && postgreSQLSettings.isSslEnabled()) {
            hikariConfig.addDataSourceProperty("ssl", "true");
            hikariConfig.addDataSourceProperty("sslmode", postgreSQLSettings.getSslMode());
            if (postgreSQLSettings.getSslCert() != null && !postgreSQLSettings.getSslCert().isEmpty()) {
                hikariConfig.addDataSourceProperty("sslcert", postgreSQLSettings.getSslCert());
            }
            if (postgreSQLSettings.getSslKey() != null && !postgreSQLSettings.getSslKey().isEmpty()) {
                hikariConfig.addDataSourceProperty("sslkey", postgreSQLSettings.getSslKey());
            }
            if (postgreSQLSettings.getSslRootCert() != null && !postgreSQLSettings.getSslRootCert().isEmpty()) {
                hikariConfig.addDataSourceProperty("sslrootcert", postgreSQLSettings.getSslRootCert());
            }
            if (postgreSQLSettings.getSslPassword() != null && !postgreSQLSettings.getSslPassword().isEmpty()) {
                hikariConfig.addDataSourceProperty("sslpassword", postgreSQLSettings.getSslPassword());
            }
        }
    }

    private static String buildJdbcUrl(DatabaseType dbType, String hostname, int port, String database, String connectionParameters, Settings.PostgreSQLSettings postgreSQLSettings) {
        Object params = "";
        if (connectionParameters != null && !connectionParameters.trim().isEmpty()) {
            if (!connectionParameters.startsWith("?") && !connectionParameters.startsWith("&")) {
                params = (String)params + "?";
            }
            params = (String)params + connectionParameters;
        }
        return switch (dbType) {
            case DatabaseType.MYSQL -> {
                String baseUrl = String.format("jdbc:mysql://%s:%d/%s", hostname, port, database);
                if (((String)params).isEmpty()) {
                    params = "?useSSL=false&serverTimezone=UTC&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048";
                } else if (!((String)params).contains("useSSL")) {
                    params = (String)params + "&useSSL=false";
                }
                yield baseUrl + (String)params;
            }
            case DatabaseType.POSTGRESQL -> {
                String baseUrl = String.format("jdbc:postgresql://%s:%d/%s", hostname, port, database);
                String sslParams = DatabaseConfig.buildPostgreSQLSslParams(postgreSQLSettings);
                params = DatabaseConfig.mergeSslParams((String)params, sslParams);
                yield baseUrl + (String)params;
            }
            case DatabaseType.H2 -> String.format("jdbc:h2:file:./data/%s;MODE=MySQL;DATABASE_TO_LOWER=TRUE", database);
            case DatabaseType.SQLITE -> String.format("jdbc:sqlite:./data/%s.db", database);
            default -> throw new IllegalArgumentException("Invalid storage type: " + String.valueOf((Object)dbType));
        };
    }

    private static String resolveDriverClass(DatabaseType dbType) {
        return dbType.getDriverClass();
    }

    public static DatabaseConfig forRemoteDatabase(String storageType, String hostname, int port, String database, String user, String password, int connectionPoolSize) {
        return new DatabaseConfig(storageType, hostname, port, database, user, password, connectionPoolSize);
    }

    private static String buildPostgreSQLSslParams(Settings.PostgreSQLSettings postgreSQLSettings) {
        if (postgreSQLSettings == null) {
            return "ssl=true&sslmode=require";
        }
        if (!postgreSQLSettings.isSslEnabled()) {
            return "sslmode=disable";
        }
        Object sslParams = "ssl=true";
        String sslMode = postgreSQLSettings.getSslMode();
        sslParams = sslMode == null || sslMode.isEmpty() ? (String)sslParams + "&sslmode=require" : (String)sslParams + "&sslmode=" + sslMode;
        if (postgreSQLSettings.getSslCert() != null && !postgreSQLSettings.getSslCert().isEmpty()) {
            sslParams = (String)sslParams + "&sslcert=" + postgreSQLSettings.getSslCert();
        }
        if (postgreSQLSettings.getSslKey() != null && !postgreSQLSettings.getSslKey().isEmpty()) {
            sslParams = (String)sslParams + "&sslkey=" + postgreSQLSettings.getSslKey();
        }
        if (postgreSQLSettings.getSslRootCert() != null && !postgreSQLSettings.getSslRootCert().isEmpty()) {
            sslParams = (String)sslParams + "&sslrootcert=" + postgreSQLSettings.getSslRootCert();
        }
        if (postgreSQLSettings.getSslPassword() != null && !postgreSQLSettings.getSslPassword().isEmpty()) {
            sslParams = (String)sslParams + "&sslpassword=" + postgreSQLSettings.getSslPassword();
        }
        return sslParams;
    }

    private static String mergeSslParams(String existingParams, String sslParams) {
        if (existingParams == null || ((String)existingParams).isEmpty()) {
            existingParams = "?" + sslParams;
        } else {
            CharSequence[] paramPairs = ((String)existingParams).substring(1).split("&");
            boolean sslReplaced = false;
            for (int i = 0; i < paramPairs.length; ++i) {
                String[] pair = paramPairs[i].split("=", 2);
                if (pair.length != 2 || !"ssl".equals(pair[0])) continue;
                paramPairs[i] = sslParams;
                sslReplaced = true;
                break;
            }
            existingParams = sslReplaced ? "?" + String.join((CharSequence)"&", paramPairs) : (String)existingParams + "&" + sslParams;
        }
        if (!((String)existingParams).contains("prepareThreshold")) {
            existingParams = (String)existingParams + "&prepareThreshold=0";
        }
        return existingParams;
    }

    public String getStorageType() {
        return this.storageType;
    }

    public String getHostname() {
        return this.hostname;
    }

    public int getPort() {
        return this.port;
    }

    public String getDatabase() {
        return this.database;
    }

    public String getUser() {
        return this.user;
    }

    public String getPassword() {
        return this.password;
    }

    public int getConnectionPoolSize() {
        return this.connectionPoolSize;
    }

    public DataSource getDataSource() {
        return this.dataSource;
    }

    public String getJdbcUrl() {
        return this.jdbcUrl;
    }

    public boolean hasDataSource() {
        return this.dataSource != null;
    }

    public boolean isLocalDatabase() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null && dbType.isLocalDatabase();
    }

    public boolean isRemoteDatabase() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null && dbType.isRemoteDatabase();
    }

    public int getDefaultPort() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null ? dbType.getDefaultPort() : 0;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        DatabaseConfig that = (DatabaseConfig)obj;
        return this.port == that.port && this.connectionPoolSize == that.connectionPoolSize && Objects.equals(this.storageType, that.storageType) && Objects.equals(this.hostname, that.hostname) && Objects.equals(this.database, that.database) && Objects.equals(this.user, that.user) && Objects.equals(this.password, that.password) && Objects.equals(this.jdbcUrl, that.jdbcUrl);
    }

    public int hashCode() {
        return Objects.hash(this.storageType, this.hostname, this.port, this.database, this.user, this.password, this.connectionPoolSize, this.jdbcUrl);
    }

    public String toString() {
        return "DatabaseConfig{storageType='" + this.storageType + "', hostname='" + this.hostname + "', port=" + this.port + ", database='" + this.database + "', user='" + this.user + "', password='***', connectionPoolSize=" + this.connectionPoolSize + ", jdbcUrl='" + this.jdbcUrl + "', isLocal=" + this.isLocalDatabase() + "}";
    }
}

