/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.database;

import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import net.rafalohaki.veloauth.database.DatabaseConfig;
import net.rafalohaki.veloauth.database.DatabaseType;
import net.rafalohaki.veloauth.database.JdbcAuthDao;
import net.rafalohaki.veloauth.database.PremiumUuidDao;
import net.rafalohaki.veloauth.i18n.Messages;
import net.rafalohaki.veloauth.libs.ormlite.dao.Dao;
import net.rafalohaki.veloauth.libs.ormlite.dao.DaoManager;
import net.rafalohaki.veloauth.libs.ormlite.jdbc.DataSourceConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.jdbc.JdbcConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.misc.TransactionManager;
import net.rafalohaki.veloauth.libs.ormlite.support.ConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.support.DatabaseConnection;
import net.rafalohaki.veloauth.libs.ormlite.table.TableUtils;
import net.rafalohaki.veloauth.model.PremiumUuid;
import net.rafalohaki.veloauth.model.RegisteredPlayer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public class DatabaseManager {
    private static final Logger logger = LoggerFactory.getLogger(DatabaseManager.class);
    private static final Marker DB_MARKER = MarkerFactory.getMarker("DATABASE");
    private static final Marker CACHE_MARKER = MarkerFactory.getMarker("CACHE");
    private static final String DATABASE_NOT_CONNECTED = "Database not connected";
    private static final String DATABASE_NOT_CONNECTED_PREMIUM_CHECK = "Database not connected - cannot check premium status for {}";
    private final ConcurrentHashMap<String, RegisteredPlayer> playerCache;
    private final ReentrantLock databaseLock;
    private final DatabaseConfig config;
    private final Messages messages;
    private final ExecutorService dbExecutor;
    private final ScheduledExecutorService healthCheckExecutor;
    private ConnectionSource connectionSource;
    private Dao<RegisteredPlayer, String> playerDao;
    private PremiumUuidDao premiumUuidDao;
    private JdbcAuthDao jdbcAuthDao;
    private volatile boolean connected;
    private volatile long lastHealthCheckTime;
    private volatile boolean lastHealthCheckPassed;

    public DatabaseManager(DatabaseConfig config, Messages messages) {
        if (config == null) {
            throw new IllegalArgumentException("Config cannot be null");
        }
        this.config = config;
        this.messages = messages;
        this.playerCache = new ConcurrentHashMap();
        this.databaseLock = new ReentrantLock();
        this.connected = false;
        this.dbExecutor = Executors.newVirtualThreadPerTaskExecutor();
        this.healthCheckExecutor = Executors.newSingleThreadScheduledExecutor();
        this.jdbcAuthDao = new JdbcAuthDao(config);
        this.lastHealthCheckTime = 0L;
        this.lastHealthCheckPassed = false;
        logger.info(DB_MARKER, messages.get("database.manager.created", new Object[0]), (Object)config.getStorageType());
    }

    public CompletableFuture<Boolean> initialize() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                this.databaseLock.lock();
                try {
                    Object jdbcUrl;
                    if (this.connected) {
                        logger.warn(DB_MARKER, "Baza danych ju\u017c jest po\u0142\u0105czona");
                        Boolean bl = true;
                        return bl;
                    }
                    if (this.config.hasDataSource()) {
                        logger.info(DB_MARKER, this.messages.get("database.manager.hikari_init", new Object[0]));
                        this.connectionSource = new DataSourceConnectionSource(this.config.getDataSource(), this.config.getJdbcUrl());
                        logger.info(DB_MARKER, this.messages.get("database.manager.hikari_ready", new Object[0]), (Object)this.config.getStorageType());
                    } else {
                        jdbcUrl = this.config.getJdbcUrl();
                        logger.info(DB_MARKER, "Connecting to database (standard JDBC): {}", jdbcUrl);
                        this.connectionSource = new JdbcConnectionSource((String)jdbcUrl, this.config.getUser(), this.config.getPassword());
                        logger.info(DB_MARKER, this.messages.get("database.manager.standard_jdbc", new Object[0]), (Object)this.config.getStorageType());
                    }
                    this.playerDao = DaoManager.createDao(this.connectionSource, RegisteredPlayer.class);
                    this.premiumUuidDao = new PremiumUuidDao(this.connectionSource);
                    this.jdbcAuthDao = new JdbcAuthDao(this.config);
                    this.createTablesIfNotExists();
                    this.connected = true;
                    logger.info(DB_MARKER, this.messages.get("database.manager.connected", new Object[0]), (Object)this.config.getStorageType());
                    this.startHealthChecks();
                    jdbcUrl = true;
                    return jdbcUrl;
                }
                finally {
                    this.databaseLock.unlock();
                }
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas inicjalizacji bazy danych", e);
                return false;
            }
        }, this.dbExecutor);
    }

    public <T> CompletableFuture<T> executeInTransaction(Callable<T> operation) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (this.connectionSource == null) {
                    throw new RuntimeException(DATABASE_NOT_CONNECTED);
                }
                TransactionManager transactionManager = new TransactionManager(this.connectionSource);
                return transactionManager.callInTransaction(operation);
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d SQL podczas transakcji", e);
                throw new RuntimeException("SQL transaction failed", e);
            }
            catch (IllegalStateException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d stanu w transakcji DB", e);
                throw new RuntimeException("Transaction state failed", e);
            }
            catch (RuntimeException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d wykonania w transakcji DB", e);
                throw new RuntimeException("Transaction execution failed", e);
            }
        }, this.dbExecutor);
    }

    private void startHealthChecks() {
        this.healthCheckExecutor.scheduleAtFixedRate(() -> {
            try {
                this.performHealthCheck();
            }
            catch (Exception e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas health check bazy danych", e);
            }
        }, 30L, 30L, TimeUnit.SECONDS);
        logger.info(DB_MARKER, this.messages.get("database.manager.health_checks_started", new Object[0]));
    }

    private void performHealthCheck() {
        try {
            if (!this.connected) {
                logger.debug(DB_MARKER, "Health check pomini\u0119ty - baza danych nie jest po\u0142\u0105czona");
                return;
            }
            boolean healthy = this.jdbcAuthDao.healthCheck();
            this.lastHealthCheckTime = System.currentTimeMillis();
            this.lastHealthCheckPassed = healthy;
            if (!healthy) {
                logger.warn(DB_MARKER, "\u274c Database health check FAILED - connection may be unstable");
            } else {
                logger.debug(DB_MARKER, "\u2705 Database health check PASSED");
            }
        }
        catch (Exception e) {
            this.lastHealthCheckTime = System.currentTimeMillis();
            this.lastHealthCheckPassed = false;
            logger.error(DB_MARKER, "\u274c Database health check FAILED with exception: {}", (Object)e.getMessage());
        }
    }

    public boolean isHealthy() {
        return this.connected;
    }

    public long getLastHealthCheckTime() {
        return this.lastHealthCheckTime;
    }

    public boolean wasLastHealthCheckPassed() {
        return this.lastHealthCheckPassed;
    }

    public void shutdown() {
        try {
            if (this.healthCheckExecutor != null && !this.healthCheckExecutor.isShutdown()) {
                this.healthCheckExecutor.shutdown();
                try {
                    if (!this.healthCheckExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                        this.healthCheckExecutor.shutdownNow();
                    }
                }
                catch (InterruptedException e) {
                    this.healthCheckExecutor.shutdownNow();
                    Thread.currentThread().interrupt();
                }
                logger.info(DB_MARKER, "Health checks zatrzymane");
            }
            this.databaseLock.lock();
            try {
                if (this.connectionSource != null) {
                    this.connectionSource.close();
                    this.connectionSource = null;
                    logger.info(DB_MARKER, this.messages.get("database.manager.connection_closed", new Object[0]));
                }
                this.connected = false;
                this.playerCache.clear();
                logger.debug(CACHE_MARKER, "Cache graczy wyczyszczony");
            }
            finally {
                this.databaseLock.unlock();
            }
        }
        catch (Exception e) {
            logger.error(DB_MARKER, "B\u0142\u0105d podczas zamykania bazy danych", e);
        }
        finally {
            this.dbExecutor.shutdown();
        }
    }

    public CompletableFuture<DbResult<RegisteredPlayer>> findPlayerByNickname(String nickname) {
        if (nickname == null || nickname.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(null));
        }
        String normalizedNickname = nickname.toLowerCase();
        return CompletableFuture.supplyAsync(() -> {
            RegisteredPlayer cached = this.playerCache.get(normalizedNickname);
            if (cached != null) {
                if (!cached.getLowercaseNickname().equals(normalizedNickname)) {
                    this.playerCache.remove(normalizedNickname);
                    logger.warn(CACHE_MARKER, "Cache corruption detected for {} - removing invalid entry", (Object)normalizedNickname);
                } else {
                    logger.debug(CACHE_MARKER, "Cache HIT dla gracza: {}", (Object)normalizedNickname);
                    return DbResult.success(cached);
                }
            }
            if (!this.connected || !this.isHealthy()) {
                logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
                return DbResult.databaseError(DATABASE_NOT_CONNECTED);
            }
            try {
                RegisteredPlayer player = this.jdbcAuthDao.findPlayerByLowercaseNickname(normalizedNickname);
                if (player != null) {
                    if (player.getLowercaseNickname().equals(normalizedNickname)) {
                        this.playerCache.put(normalizedNickname, player);
                        logger.debug(CACHE_MARKER, "Cache MISS -> DB HIT dla gracza: {}", (Object)normalizedNickname);
                    } else {
                        logger.warn(CACHE_MARKER, "Database inconsistency for {} - expected {}, found {}", normalizedNickname, normalizedNickname, player.getLowercaseNickname());
                    }
                } else {
                    logger.debug(DB_MARKER, "Gracz nie znaleziony: {}", (Object)normalizedNickname);
                }
                return DbResult.success(player);
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas wyszukiwania gracza: {}", (Object)normalizedNickname, (Object)e);
                return DbResult.databaseError(this.messages.get("database.error", new Object[0]) + ": " + e.getMessage());
            }
        }, this.dbExecutor);
    }

    public CompletableFuture<DbResult<Boolean>> savePlayer(RegisteredPlayer player) {
        if (player == null) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            if (!this.connected) {
                logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
                return DbResult.databaseError(DATABASE_NOT_CONNECTED);
            }
            try {
                boolean success = this.jdbcAuthDao.upsertPlayer(player);
                if (success) {
                    this.playerCache.put(player.getLowercaseNickname(), player);
                    logger.debug(DB_MARKER, "Zapisano gracza (upsert): {}", (Object)player.getNickname());
                }
                return DbResult.success(success);
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas zapisywania gracza: {}", (Object)player.getNickname(), (Object)e);
                return DbResult.databaseError(this.messages.get("database.error", new Object[0]) + ": " + e.getMessage());
            }
        }, this.dbExecutor);
    }

    public CompletableFuture<DbResult<Boolean>> deletePlayer(String lowercaseNickname) {
        if (lowercaseNickname == null || lowercaseNickname.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            if (!this.connected) {
                logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
                return DbResult.databaseError(DATABASE_NOT_CONNECTED);
            }
            try {
                boolean deleted = this.jdbcAuthDao.deletePlayer(lowercaseNickname);
                this.playerCache.remove(lowercaseNickname);
                if (deleted) {
                    logger.debug(DB_MARKER, "Usuni\u0119to gracza: {}", (Object)lowercaseNickname);
                    return DbResult.success(true);
                }
                logger.debug(DB_MARKER, "Gracz nie znaleziony do usuni\u0119cia: {}", (Object)lowercaseNickname);
                return DbResult.success(false);
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas usuwania gracza: {}", (Object)lowercaseNickname, (Object)e);
                return DbResult.databaseError(this.messages.get("database.error", new Object[0]) + ": " + e.getMessage());
            }
        }, this.dbExecutor);
    }

    public CompletableFuture<DbResult<Boolean>> isPremium(String username) {
        if (username == null || username.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            if (!this.connected) {
                logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED_PREMIUM_CHECK, (Object)username);
                return DbResult.databaseError(DATABASE_NOT_CONNECTED);
            }
            try {
                boolean premium = this.premiumUuidDao.findByNickname(username).isPresent();
                logger.debug(DB_MARKER, "Premium status z PREMIUM_UUIDS dla {}: {}", (Object)username, (Object)premium);
                return DbResult.success(premium);
            }
            catch (RuntimeException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d wykonania podczas sprawdzania premium status dla gracza: {}", (Object)username, (Object)e);
                return DbResult.databaseError(this.messages.get("database.error", new Object[0]) + ": " + e.getMessage());
            }
        }, this.dbExecutor);
    }

    public CompletableFuture<List<RegisteredPlayer>> getAllPlayers() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.connected || this.playerDao == null) {
                    logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
                    return List.of();
                }
                List<RegisteredPlayer> players = this.playerDao.queryForAll();
                logger.debug(DB_MARKER, "Pobrano {} graczy z bazy danych", (Object)players.size());
                return players;
            }
            catch (SQLException e) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas pobierania wszystkich graczy", e);
                return List.of();
            }
        }, this.dbExecutor);
    }

    public void clearCache() {
        this.playerCache.clear();
        logger.debug(CACHE_MARKER, "Cache graczy wyczyszczony");
    }

    public void removeCachedPlayer(String lowercaseNickname) {
        if (lowercaseNickname != null) {
            this.playerCache.remove(lowercaseNickname);
            logger.debug("Usuni\u0119to z cache gracza: {}", (Object)lowercaseNickname);
        }
    }

    public int getCacheSize() {
        return this.playerCache.size();
    }

    public boolean isConnected() {
        return this.connected;
    }

    public DatabaseConfig getConfig() {
        return this.config;
    }

    public PremiumUuidDao getPremiumUuidDao() {
        return this.premiumUuidDao;
    }

    private void createTablesIfNotExists() throws SQLException {
        logger.info(this.messages.get("database.manager.creating_tables", new Object[0]));
        TableUtils.createTableIfNotExists(this.connectionSource, RegisteredPlayer.class);
        TableUtils.createTableIfNotExists(this.connectionSource, PremiumUuid.class);
        this.createIndexesIfNotExists();
        logger.info(this.messages.get("database.manager.tables_created", new Object[0]));
    }

    private void createIndexesIfNotExists() {
        try {
            DatabaseType dbType = DatabaseType.fromName(this.config.getStorageType());
            if (dbType == DatabaseType.POSTGRESQL) {
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_ip ON \"AUTH\" (\"IP\")");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_uuid ON \"AUTH\" (\"UUID\")");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_logindate ON \"AUTH\" (\"LOGINDATE\")");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_regdate ON \"AUTH\" (\"REGDATE\")");
            } else if (dbType == DatabaseType.MYSQL) {
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_ip ON `AUTH` (`IP`)");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_uuid ON `AUTH` (`UUID`)");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_logindate ON `AUTH` (`LOGINDATE`)");
                this.executeUpdate("CREATE INDEX IF NOT EXISTS idx_auth_regdate ON `AUTH` (`REGDATE`)");
            }
        }
        catch (SQLException e) {
            logger.warn(this.messages.get("database.manager.index_error", new Object[0]), (Object)e.getMessage());
        }
    }

    private void executeUpdate(String sql) throws SQLException {
        if (this.connectionSource != null) {
            DatabaseConnection connection = null;
            try {
                connection = this.connectionSource.getReadWriteConnection(null);
                connection.executeStatement(sql, 0);
            }
            finally {
                if (connection != null) {
                    this.connectionSource.releaseConnection(connection);
                }
            }
        }
    }

    public static final class DbResult<T> {
        private final T value;
        private final boolean isDatabaseError;
        private final String errorMessage;

        private DbResult(T value, boolean isDatabaseError, String errorMessage) {
            this.value = value;
            this.isDatabaseError = isDatabaseError;
            this.errorMessage = errorMessage;
        }

        public static <T> DbResult<T> success(T value) {
            return new DbResult<T>(value, false, null);
        }

        public static <T> DbResult<T> databaseError(String errorMessage) {
            return new DbResult<Object>(null, true, errorMessage);
        }

        public T getValue() {
            return this.value;
        }

        public boolean isDatabaseError() {
            return this.isDatabaseError;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public boolean isSuccess() {
            return !this.isDatabaseError;
        }
    }
}

