/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.util;

import java.util.concurrent.CompletableFuture;
import net.rafalohaki.veloauth.config.Settings;
import net.rafalohaki.veloauth.database.DatabaseManager;
import net.rafalohaki.veloauth.i18n.Messages;
import net.rafalohaki.veloauth.libs.bcrypt.bcrypt.BCrypt;
import net.rafalohaki.veloauth.model.RegisteredPlayer;
import org.slf4j.Logger;
import org.slf4j.Marker;

public final class AuthenticationHelper {
    private AuthenticationHelper() {
    }

    public static String hashPassword(String password, Settings settings) {
        return BCrypt.with(BCrypt.Version.VERSION_2Y).hashToString(settings.getBcryptCost(), password.toCharArray());
    }

    public static boolean verifyPassword(String password, String hash) {
        BCrypt.Result result = BCrypt.verifyer().verify(password.toCharArray(), hash);
        return result.verified;
    }

    public static RegisteredPlayer createRegisteredPlayer(String username, String hashedPassword, String playerIp, String playerUuid) {
        return new RegisteredPlayer(username, hashedPassword, playerIp, playerUuid);
    }

    public static CompletableFuture<RegisteredPlayer> performRegistration(AuthenticationContext context) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                DatabaseManager.DbResult<RegisteredPlayer> existingResult = context.databaseManager().findPlayerByNickname(context.username()).join();
                if (existingResult.isDatabaseError()) {
                    context.logger().error(context.dbMarker(), "Database error during registration check for {}: {}", (Object)context.username(), (Object)existingResult.getErrorMessage());
                    return null;
                }
                RegisteredPlayer existingPlayer = existingResult.getValue();
                if (existingPlayer != null) {
                    context.logger().debug(context.dbMarker(), context.messages().get("player.already_exists", new Object[0]), (Object)context.username());
                    return null;
                }
                String hashedPassword = AuthenticationHelper.hashPassword(context.password(), context.settings());
                RegisteredPlayer newPlayer = AuthenticationHelper.createRegisteredPlayer(context.username(), hashedPassword, context.playerIp(), context.playerUuid());
                DatabaseManager.DbResult<Boolean> saveResult = context.databaseManager().savePlayer(newPlayer).join();
                if (saveResult.isDatabaseError()) {
                    context.logger().error(context.dbMarker(), "Database error during registration save for {}: {}", (Object)context.username(), (Object)saveResult.getErrorMessage());
                    return null;
                }
                boolean saved = saveResult.getValue();
                if (!saved) {
                    context.logger().error(context.dbMarker(), context.messages().get("player.save.failed", new Object[0]), (Object)context.username());
                    return null;
                }
                context.logger().info(context.dbMarker(), context.messages().get("player.registered.success", new Object[0]), (Object)context.username());
                return newPlayer;
            }
            catch (Exception e) {
                context.logger().error(context.dbMarker(), context.messages().get("player.registration.error", new Object[0]), (Object)context.username(), (Object)e);
                return null;
            }
        });
    }

    public static CompletableFuture<RegisteredPlayer> performLogin(DatabaseManager databaseManager, String username, String password, Logger logger, Marker dbMarker, Messages messages) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                DatabaseManager.DbResult<RegisteredPlayer> playerResult = databaseManager.findPlayerByNickname(username).join();
                if (playerResult.isDatabaseError()) {
                    logger.error(dbMarker, "Database error during login lookup for {}: {}", (Object)username, (Object)playerResult.getErrorMessage());
                    return null;
                }
                RegisteredPlayer registeredPlayer = playerResult.getValue();
                if (registeredPlayer == null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(dbMarker, messages.get("player.not.found", new Object[0]), (Object)username);
                    }
                    return null;
                }
                if (!AuthenticationHelper.verifyPassword(password, registeredPlayer.getHash())) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(dbMarker, messages.get("player.password.invalid", new Object[0]), (Object)username);
                    }
                    return null;
                }
                logger.info(dbMarker, messages.get("player.verified.success", new Object[0]), (Object)username);
                return registeredPlayer;
            }
            catch (Exception e) {
                logger.error(dbMarker, messages.get("player.login.error", new Object[0]), (Object)username, (Object)e);
                return null;
            }
        });
    }

    public static CompletableFuture<Boolean> performPasswordChange(PasswordChangeContext context) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                DatabaseManager.DbResult<RegisteredPlayer> playerResult = context.databaseManager().findPlayerByNickname(context.username()).join();
                if (playerResult.isDatabaseError()) {
                    context.logger().error(context.dbMarker(), "Database error during password change lookup for {}: {}", (Object)context.username(), (Object)playerResult.getErrorMessage());
                    return false;
                }
                RegisteredPlayer registeredPlayer = playerResult.getValue();
                if (registeredPlayer == null) {
                    context.logger().debug(context.dbMarker(), context.messages().get("player.not.found", new Object[0]), (Object)context.username());
                    return false;
                }
                if (!AuthenticationHelper.verifyPassword(context.oldPassword(), registeredPlayer.getHash())) {
                    context.logger().debug(context.dbMarker(), context.messages().get("player.old_password.invalid", new Object[0]), (Object)context.username());
                    return false;
                }
                String newHashedPassword = AuthenticationHelper.hashPassword(context.newPassword(), context.settings());
                registeredPlayer.setHash(newHashedPassword);
                DatabaseManager.DbResult<Boolean> saveResult = context.databaseManager().savePlayer(registeredPlayer).join();
                if (saveResult.isDatabaseError()) {
                    context.logger().error(context.dbMarker(), "Database error during password change save for {}: {}", (Object)context.username(), (Object)saveResult.getErrorMessage());
                    return false;
                }
                boolean saved = saveResult.getValue();
                if (saved) {
                    context.logger().info(context.dbMarker(), context.messages().get("player.password.changed.success", new Object[0]), (Object)context.username());
                    return true;
                }
                context.logger().error(context.dbMarker(), context.messages().get("player.password.save.failed", new Object[0]), (Object)context.username());
                return false;
            }
            catch (Exception e) {
                context.logger().error(context.dbMarker(), context.messages().get("player.password.change.error", new Object[0]), (Object)context.username(), (Object)e);
                return false;
            }
        });
    }

    private static RegisteredPlayer validatePlayerForDeletion(AccountDeletionContext context) {
        DatabaseManager.DbResult<RegisteredPlayer> playerResult = context.databaseManager().findPlayerByNickname(context.username()).join();
        if (playerResult.isDatabaseError()) {
            context.logger().error(context.dbMarker(), "Database error during account deletion lookup for {}: {}", (Object)context.username(), (Object)playerResult.getErrorMessage());
            return null;
        }
        RegisteredPlayer registeredPlayer = playerResult.getValue();
        if (registeredPlayer == null) {
            if (context.logger().isDebugEnabled()) {
                context.logger().debug(context.dbMarker(), context.messages().get("player.not.found", new Object[0]), (Object)context.username());
            }
            return null;
        }
        if (!AuthenticationHelper.verifyPassword(context.password(), registeredPlayer.getHash())) {
            if (context.logger().isDebugEnabled()) {
                context.logger().debug(context.dbMarker(), context.messages().get("player.password.invalid.deletion", new Object[0]), (Object)context.username());
            }
            return null;
        }
        return registeredPlayer;
    }

    private static boolean executeDeletion(AccountDeletionContext context, String lowercaseNick) {
        DatabaseManager.DbResult<Boolean> deleteResult = context.databaseManager().deletePlayer(lowercaseNick).join();
        if (deleteResult.isDatabaseError()) {
            context.logger().error(context.dbMarker(), "Database error during account deletion for {}: {}", (Object)context.username(), (Object)deleteResult.getErrorMessage());
            return false;
        }
        boolean deleted = deleteResult.getValue();
        if (deleted) {
            context.logger().info(context.dbMarker(), context.messages().get("player.account.deleted.success", new Object[0]), (Object)context.username());
            return true;
        }
        context.logger().error(context.dbMarker(), context.messages().get("player.account.delete.failed", new Object[0]), (Object)context.username());
        return false;
    }

    public static CompletableFuture<Boolean> performAccountDeletion(AccountDeletionContext context) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                RegisteredPlayer registeredPlayer = AuthenticationHelper.validatePlayerForDeletion(context);
                if (registeredPlayer == null) {
                    return false;
                }
                return AuthenticationHelper.executeDeletion(context, context.username());
            }
            catch (Exception e) {
                context.logger().error(context.dbMarker(), context.messages().get("player.account.deletion.error", new Object[0]), (Object)context.username(), (Object)e);
                return false;
            }
        });
    }

    public record AuthenticationContext(DatabaseManager databaseManager, String username, String password, String playerIp, String playerUuid, Settings settings, Logger logger, Marker dbMarker, Messages messages) {
    }

    public record PasswordChangeContext(DatabaseManager databaseManager, String username, String oldPassword, String newPassword, Settings settings, Logger logger, Marker dbMarker, Messages messages) {
    }

    public record AccountDeletionContext(DatabaseManager databaseManager, String username, String password, Logger logger, Marker dbMarker, Messages messages) {
    }
}

