/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.cache;

import java.net.InetAddress;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.rafalohaki.veloauth.config.Settings;
import net.rafalohaki.veloauth.i18n.Messages;
import net.rafalohaki.veloauth.model.CachedAuthUser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public class AuthCache {
    private static final Logger logger = LoggerFactory.getLogger(AuthCache.class);
    private static final Marker SECURITY_MARKER = MarkerFactory.getMarker("SECURITY");
    private final ConcurrentHashMap<UUID, CachedAuthUser> authorizedPlayers;
    private final ConcurrentHashMap<InetAddress, BruteForceEntry> bruteForceAttempts;
    private final ConcurrentHashMap<String, PremiumCacheEntry> premiumCache;
    private final ConcurrentHashMap<UUID, ActiveSession> activeSessions;
    private final ReentrantLock cacheLock;
    private final int ttlMinutes;
    private final int maxSize;
    private final int maxSessions;
    private final int maxPremiumCache;
    private final int maxLoginAttempts;
    private final int bruteForceTimeoutMinutes;
    private final int premiumTtlHours;
    private final double premiumRefreshThreshold;
    private final ScheduledExecutorService cleanupScheduler;
    private final ScheduledExecutorService metricsScheduler;
    private final AtomicLong cacheHits = new AtomicLong(0L);
    private final AtomicLong cacheMisses = new AtomicLong(0L);
    private final Settings settings;
    private final Messages messages;

    public AuthCache(int ttlMinutes, int maxSize, int maxSessions, int maxPremiumCache, int maxLoginAttempts, int bruteForceTimeoutMinutes, int cleanupIntervalMinutes, Settings settings, Messages messages) {
        String error = AuthCache.validateParams(ttlMinutes, maxSize, maxSessions, maxPremiumCache, maxLoginAttempts, bruteForceTimeoutMinutes, messages);
        if (error != null) {
            throw new IllegalArgumentException(error);
        }
        this.ttlMinutes = ttlMinutes;
        this.maxSize = maxSize;
        this.maxSessions = maxSessions;
        this.maxPremiumCache = maxPremiumCache;
        this.maxLoginAttempts = maxLoginAttempts;
        this.bruteForceTimeoutMinutes = bruteForceTimeoutMinutes;
        this.premiumTtlHours = settings.getPremiumTtlHours();
        this.premiumRefreshThreshold = settings.getPremiumRefreshThreshold();
        this.settings = settings;
        this.messages = messages;
        this.authorizedPlayers = new ConcurrentHashMap();
        this.bruteForceAttempts = new ConcurrentHashMap();
        this.premiumCache = new ConcurrentHashMap();
        this.activeSessions = new ConcurrentHashMap();
        this.cacheLock = new ReentrantLock();
        this.cleanupScheduler = Executors.newScheduledThreadPool(1, r -> {
            Thread t2 = new Thread(r, "AuthCache-Cleanup");
            t2.setDaemon(true);
            return t2;
        });
        this.metricsScheduler = Executors.newScheduledThreadPool(1, r -> {
            Thread t2 = new Thread(r, "AuthCache-Metrics");
            t2.setDaemon(true);
            return t2;
        });
        if (cleanupIntervalMinutes > 0) {
            this.cleanupScheduler.scheduleAtFixedRate(this::cleanupExpiredEntries, cleanupIntervalMinutes, cleanupIntervalMinutes, TimeUnit.MINUTES);
        }
        this.metricsScheduler.scheduleAtFixedRate(this::logCacheMetrics, 5L, 5L, TimeUnit.MINUTES);
        if (logger.isInfoEnabled()) {
            logger.info(messages.get("cache.auth.created", new Object[0]), ttlMinutes, maxSize, maxLoginAttempts, bruteForceTimeoutMinutes);
        }
    }

    private static String validateParams(int ttlMinutes, int maxSize, int maxSessions, int maxPremiumCache, int maxLoginAttempts, int bruteForceTimeoutMinutes, Messages messages) {
        ParamCheck[] checks;
        for (ParamCheck c : checks = new ParamCheck[]{new ParamCheck(ttlMinutes < 0, messages.get("validation.ttl.negative", new Object[0])), new ParamCheck(maxSize <= 0, messages.get("validation.maxsize.gt_zero", new Object[0])), new ParamCheck(maxSessions <= 0, messages.get("validation.maxsessions.gt_zero", new Object[0])), new ParamCheck(maxPremiumCache <= 0, messages.get("validation.maxpremiumcache.gt_zero", new Object[0])), new ParamCheck(maxLoginAttempts <= 0, messages.get("validation.maxloginattempts.gt_zero", new Object[0])), new ParamCheck(bruteForceTimeoutMinutes <= 0, messages.get("validation.bruteforcetimeout.gt_zero", new Object[0]))}) {
            if (!c.invalid) continue;
            return c.message;
        }
        return null;
    }

    public void addAuthorizedPlayer(UUID uuid, CachedAuthUser user) {
        if (uuid == null || user == null) {
            throw new IllegalArgumentException("UUID i user nie mog\u0105 by\u0107 null");
        }
        if (this.authorizedPlayers.size() >= this.maxSize && !this.authorizedPlayers.containsKey(uuid)) {
            this.evictOldestAuthorizedEntryAtomic();
        }
        this.authorizedPlayers.put(uuid, user);
        if (logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.auth.added", new Object[0]), (Object)user.getNickname(), (Object)uuid);
        }
    }

    @Nullable
    public CachedAuthUser getAuthorizedPlayer(@Nullable UUID uuid) {
        if (uuid == null) {
            return null;
        }
        CachedAuthUser user = this.authorizedPlayers.get(uuid);
        if (user == null) {
            this.cacheMisses.incrementAndGet();
            double rate = (double)this.cacheHits.get() / (double)Math.max(1L, this.cacheHits.get() + this.cacheMisses.get()) * 100.0;
            String rateStr = String.format(Locale.US, "%.1f", rate);
            if (logger.isDebugEnabled()) {
                logger.debug(this.messages.get("cache.debug.uuid.miss", new Object[0]), (Object)uuid, (Object)rateStr);
            }
            return null;
        }
        if (!user.isValid(this.ttlMinutes)) {
            this.removeAuthorizedPlayer(uuid);
            this.cacheMisses.incrementAndGet();
            double rate = (double)this.cacheHits.get() / (double)Math.max(1L, this.cacheHits.get() + this.cacheMisses.get()) * 100.0;
            String rateStr = String.format(Locale.US, "%.1f", rate);
            if (logger.isDebugEnabled()) {
                logger.debug(this.messages.get("cache.debug.uuid.expired", new Object[0]), (Object)uuid, (Object)rateStr);
            }
            return null;
        }
        this.cacheHits.incrementAndGet();
        double rate = (double)this.cacheHits.get() / (double)Math.max(1L, this.cacheHits.get() + this.cacheMisses.get()) * 100.0;
        String rateStr = String.format(Locale.US, "%.1f", rate);
        if (logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.hit.rate", new Object[0]), (Object)rateStr);
        }
        return user;
    }

    @Nonnull
    public Optional<CachedAuthUser> findAuthorizedPlayer(@Nullable UUID uuid) {
        return Objects.requireNonNull(Optional.ofNullable(this.getAuthorizedPlayer(uuid)), "Optional cannot be null");
    }

    public void removeAuthorizedPlayer(UUID uuid) {
        CachedAuthUser removed;
        if (uuid != null && (removed = this.authorizedPlayers.remove(uuid)) != null && logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.player.removed", new Object[0]), (Object)removed.getNickname(), (Object)uuid);
        }
    }

    public boolean isPlayerAuthorized(UUID uuid, String currentIp) {
        CachedAuthUser user = this.getAuthorizedPlayer(uuid);
        if (user == null) {
            return false;
        }
        return user.matchesIp(currentIp);
    }

    public void invalidatePlayerData(UUID playerUuid) {
        if (playerUuid == null) {
            return;
        }
        CachedAuthUser user = this.authorizedPlayers.get(playerUuid);
        if (user != null) {
            this.authorizedPlayers.remove(playerUuid);
            if (logger.isDebugEnabled()) {
                logger.debug("Invalidated cached data for player UUID: {} (nickname: {})", (Object)playerUuid, (Object)user.getNickname());
            }
        }
    }

    public void removePremiumPlayer(String nickname) {
        if (nickname == null || nickname.isEmpty()) {
            return;
        }
        PremiumCacheEntry removed = this.premiumCache.remove(nickname.toLowerCase());
        if (removed != null && logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.premium.removed", new Object[0]), (Object)nickname, (Object)removed.isPremium());
        }
    }

    public void addPremiumPlayer(String nickname, UUID premiumUuid) {
        if (nickname == null || nickname.isEmpty()) {
            return;
        }
        String key = nickname.toLowerCase();
        if (this.premiumCache.size() >= this.maxPremiumCache && !this.premiumCache.containsKey(key)) {
            this.evictOldestPremiumEntryAtomic();
        }
        long ttl = TimeUnit.HOURS.toMillis(this.premiumTtlHours);
        PremiumCacheEntry entry = new PremiumCacheEntry(premiumUuid != null, premiumUuid, ttl, this.premiumRefreshThreshold);
        this.premiumCache.put(key, entry);
        if (logger.isDebugEnabled()) {
            logger.debug("{} | nickname: {}, premium entry: {}, TTL: {}h, threshold: {}", this.messages.get("cache.debug.premium.added", new Object[0]), nickname, premiumUuid != null, this.premiumTtlHours, this.premiumRefreshThreshold);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean registerFailedLogin(InetAddress address) {
        if (address == null) {
            return false;
        }
        this.cacheLock.lock();
        try {
            boolean blocked;
            BruteForceEntry entry = this.bruteForceAttempts.computeIfAbsent(address, k -> new BruteForceEntry());
            if (entry.isExpired(this.bruteForceTimeoutMinutes)) {
                entry.reset();
            }
            entry.incrementAttempts();
            boolean bl = blocked = entry.getAttempts() >= this.maxLoginAttempts;
            if (blocked) {
                if (logger.isWarnEnabled()) {
                    logger.warn(this.messages.get("cache.warn.ip.blocked", new Object[0]), (Object)address.getHostAddress(), (Object)entry.getAttempts());
                }
            } else if (logger.isDebugEnabled()) {
                logger.debug(this.messages.get("cache.debug.failed.login", new Object[0]), address.getHostAddress(), entry.getAttempts(), this.maxLoginAttempts);
            }
            boolean bl2 = blocked;
            this.cacheLock.unlock();
            return bl2;
        }
        catch (Throwable throwable) {
            try {
                this.cacheLock.unlock();
                throw throwable;
            }
            catch (IllegalStateException e) {
                logger.error(this.messages.get("cache.error.state.register_failed", new Object[0]) + String.valueOf(address), e);
                return false;
            }
            catch (IllegalArgumentException e) {
                logger.error(this.messages.get("cache.error.args.register_failed", new Object[0]) + String.valueOf(address), e);
                return false;
            }
        }
    }

    public boolean isBlocked(InetAddress address) {
        if (address == null) {
            return false;
        }
        BruteForceEntry entry = this.bruteForceAttempts.get(address);
        if (entry == null) {
            return false;
        }
        if (entry.isExpired(this.bruteForceTimeoutMinutes)) {
            this.bruteForceAttempts.remove(address);
            return false;
        }
        return entry.getAttempts() >= this.maxLoginAttempts;
    }

    public void resetLoginAttempts(InetAddress address) {
        BruteForceEntry removed;
        if (address != null && (removed = this.bruteForceAttempts.remove(address)) != null && logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.reset.attempts", new Object[0]), (Object)address.getHostAddress());
        }
    }

    @Nullable
    public PremiumCacheEntry getPremiumStatus(@Nullable String nickname) {
        if (nickname == null || nickname.isEmpty()) {
            return null;
        }
        String key = nickname.toLowerCase();
        PremiumCacheEntry entry = this.premiumCache.get(key);
        if (entry == null) {
            return null;
        }
        if (entry.isExpired()) {
            this.premiumCache.remove(key);
            if (logger.isDebugEnabled()) {
                logger.debug("Premium cache entry expired for {} (age: {}ms, TTL: {}ms)", nickname, entry.getAgeMillis(), entry.getTtlMillis());
            }
            return null;
        }
        return entry;
    }

    @Nonnull
    public Optional<PremiumCacheEntry> findPremiumStatus(@Nullable String nickname) {
        return Objects.requireNonNull(Optional.ofNullable(this.getPremiumStatus(nickname)), "Optional cannot be null");
    }

    public void clearAll() {
        try {
            this.cacheLock.lock();
            try {
                this.authorizedPlayers.clear();
                this.bruteForceAttempts.clear();
                this.premiumCache.clear();
                this.activeSessions.clear();
                if (logger.isInfoEnabled()) {
                    logger.info(this.messages.get("cache.all_cleared", new Object[0]));
                }
            }
            finally {
                this.cacheLock.unlock();
            }
        }
        catch (IllegalStateException e) {
            logger.error(this.messages.get("cache.error.state.clear", new Object[0]), e);
        }
    }

    public void startSession(UUID uuid, String nickname, String ip) {
        if (uuid == null || nickname == null) {
            return;
        }
        if (this.activeSessions.size() >= this.maxSessions && !this.activeSessions.containsKey(uuid)) {
            this.evictOldestSessionAtomic();
        }
        ActiveSession session = new ActiveSession(uuid, nickname, ip);
        this.activeSessions.put(uuid, session);
        if (logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.session.started", new Object[0]), nickname, uuid, ip);
        }
    }

    public void endSession(UUID uuid) {
        if (uuid == null) {
            return;
        }
        ActiveSession removed = this.activeSessions.remove(uuid);
        if (removed != null && logger.isDebugEnabled()) {
            logger.debug(this.messages.get("cache.debug.session.ended", new Object[0]), (Object)removed.getNickname(), (Object)uuid);
        }
    }

    public boolean hasActiveSession(UUID uuid, String nickname, String currentIp) {
        if (this.invalidSessionParams(uuid, nickname, currentIp)) {
            return false;
        }
        ActiveSession session = this.activeSessions.get(uuid);
        if (session == null) {
            return false;
        }
        if (this.isNicknameMismatch(session, nickname, uuid)) {
            return false;
        }
        if (this.isIpMismatch(session, currentIp, uuid)) {
            return false;
        }
        session.updateActivity();
        return true;
    }

    private boolean invalidSessionParams(UUID uuid, String nickname, String currentIp) {
        return uuid == null || nickname == null || currentIp == null;
    }

    private boolean isNicknameMismatch(ActiveSession session, String nickname, UUID uuid) {
        if (session.getNickname().equalsIgnoreCase(nickname)) {
            return false;
        }
        if (logger.isWarnEnabled()) {
            logger.warn(SECURITY_MARKER, this.messages.get("security.session.hijack", new Object[0]), uuid, session.getNickname(), nickname);
        }
        this.activeSessions.remove(uuid);
        return true;
    }

    private boolean isIpMismatch(ActiveSession session, String currentIp, UUID uuid) {
        if (session.getIp().equals(currentIp)) {
            return false;
        }
        if (logger.isWarnEnabled()) {
            logger.warn(SECURITY_MARKER, this.messages.get("security.session.ip.mismatch", new Object[0]), uuid, session.getIp(), currentIp);
        }
        this.activeSessions.remove(uuid);
        return true;
    }

    private void logCacheMetrics() {
        try {
            if (!this.settings.isDebugEnabled()) {
                return;
            }
            CacheStats stats = this.getStats();
            double hitRate = stats.getHitRate();
            long totalRequests = stats.getTotalRequests();
            logger.debug("=== CACHE METRICS ===");
            logger.debug("Authorized Players: {}/{} ({}% full)", stats.authorizedPlayersCount(), stats.maxSize(), (double)stats.authorizedPlayersCount() / (double)stats.maxSize() * 100.0);
            logger.debug("Brute Force Entries: {}", (Object)stats.bruteForceEntriesCount());
            logger.debug("Premium Cache Entries: {}", (Object)stats.premiumCacheCount());
            String hitRateStr = String.format(Locale.US, "%.2f", hitRate);
            logger.debug("Cache Performance: {} hits, {} misses, {}% hit rate", stats.cacheHits(), stats.cacheMisses(), hitRateStr);
            logger.debug("Total Requests: {}", (Object)totalRequests);
            if (hitRate < 80.0 && totalRequests > 100L) {
                logger.warn("Low cache hit rate ({}%) - consider increasing TTL or cache size", (Object)hitRateStr);
            }
            if ((double)stats.authorizedPlayersCount() >= (double)stats.maxSize() * 0.9) {
                logger.warn("Cache approaching capacity ({}/{} entries) - consider increasing maxSize", (Object)stats.authorizedPlayersCount(), (Object)stats.maxSize());
            }
            logger.debug("====================");
        }
        catch (Exception e) {
            logger.error("Error logging cache metrics", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanupExpiredEntries() {
        try {
            this.cacheLock.lock();
            try {
                int removedAuth = this.cleanupCache(this.authorizedPlayers, entry -> !((CachedAuthUser)entry.getValue()).isValid(this.ttlMinutes));
                int removedBrute = this.cleanupCache(this.bruteForceAttempts, entry -> ((BruteForceEntry)entry.getValue()).isExpired(this.bruteForceTimeoutMinutes));
                int removedPremium = this.cleanupCache(this.premiumCache, entry -> ((PremiumCacheEntry)entry.getValue()).isExpired());
                int removedSessions = this.cleanupCache(this.activeSessions, entry -> !((ActiveSession)entry.getValue()).isActive(60L));
                if (removedAuth > 0 || removedBrute > 0 || removedPremium > 0 || removedSessions > 0) {
                    logger.debug("Cleanup: usuni\u0119to {} auth, {} brute force, {} premium, {} sessions", removedAuth, removedBrute, removedPremium, removedSessions);
                }
            }
            finally {
                this.cacheLock.unlock();
            }
        }
        catch (Exception e) {
            logger.error("B\u0142\u0105d podczas czyszczenia cache", e);
        }
    }

    private <K, V> int cleanupCache(Map<K, V> cache, Predicate<Map.Entry<K, V>> shouldRemove) {
        int removed = 0;
        Iterator<Map.Entry<K, V>> iterator = cache.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<K, V> entry = iterator.next();
            if (!shouldRemove.test(entry)) continue;
            iterator.remove();
            ++removed;
        }
        return removed;
    }

    private void evictOldestAuthorizedEntryAtomic() {
        Map.Entry oldest = this.authorizedPlayers.entrySet().stream().min(Comparator.comparingLong(e -> ((CachedAuthUser)e.getValue()).getCacheTime())).orElse(null);
        if (oldest != null) {
            this.authorizedPlayers.remove(oldest.getKey(), oldest.getValue());
        }
    }

    private void evictOldestSessionAtomic() {
        Map.Entry oldest = this.activeSessions.entrySet().stream().min(Comparator.comparingLong(e -> ((ActiveSession)e.getValue()).getSessionStartTime())).orElse(null);
        if (oldest != null) {
            this.activeSessions.remove(oldest.getKey(), oldest.getValue());
        }
    }

    private void evictOldestPremiumEntryAtomic() {
        Map.Entry oldest = this.premiumCache.entrySet().stream().min(Comparator.comparingLong(e -> ((PremiumCacheEntry)e.getValue()).getTimestamp())).orElse(null);
        if (oldest != null) {
            String evictedKey = (String)oldest.getKey();
            PremiumCacheEntry evictedEntry = (PremiumCacheEntry)oldest.getValue();
            this.premiumCache.remove(evictedKey, evictedEntry);
            if (logger.isDebugEnabled()) {
                logger.debug("Premium cache LRU eviction: {} (age: {}ms, was premium: {})", evictedKey, evictedEntry.getAgeMillis(), evictedEntry.isPremium());
            }
        }
    }

    public void shutdown() {
        block7: {
            try {
                if (this.cacheHits.get() + this.cacheMisses.get() > 0L) {
                    double rate = (double)this.cacheHits.get() / (double)(this.cacheHits.get() + this.cacheMisses.get()) * 100.0;
                    String rateStr = String.format(Locale.US, "%.1f", rate);
                    if (logger.isInfoEnabled()) {
                        logger.info(this.messages.get("cache.stats_final", new Object[0]), this.cacheHits.get(), this.cacheMisses.get(), rateStr);
                    }
                }
                this.cleanupScheduler.shutdown();
                if (!this.cleanupScheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.cleanupScheduler.shutdownNow();
                }
                this.metricsScheduler.shutdown();
                if (!this.metricsScheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.metricsScheduler.shutdownNow();
                }
                this.clearAll();
                if (logger.isInfoEnabled()) {
                    logger.info(this.messages.get("cache.shutdown", new Object[0]));
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                this.cleanupScheduler.shutdownNow();
                this.metricsScheduler.shutdownNow();
                if (!logger.isWarnEnabled()) break block7;
                logger.warn(this.messages.get("cache.interrupted_shutdown", new Object[0]));
            }
        }
    }

    public CacheStats getStats() {
        return new CacheStats(this.authorizedPlayers.size(), this.bruteForceAttempts.size(), this.premiumCache.size(), this.cacheHits.get(), this.cacheMisses.get(), this.maxSize, this.ttlMinutes);
    }

    private static final class ParamCheck {
        final boolean invalid;
        final String message;

        ParamCheck(boolean invalid, String message) {
            this.invalid = invalid;
            this.message = message;
        }
    }

    public static class PremiumCacheEntry {
        private final boolean isPremium;
        private final UUID premiumUuid;
        private final long timestamp;
        private final long ttlMillis;
        private final double refreshThreshold;

        public PremiumCacheEntry(boolean isPremium, UUID premiumUuid, long ttlMillis, double refreshThreshold) {
            this.isPremium = isPremium;
            this.premiumUuid = premiumUuid;
            this.timestamp = System.currentTimeMillis();
            this.ttlMillis = ttlMillis;
            this.refreshThreshold = refreshThreshold;
        }

        public boolean isPremium() {
            return this.isPremium;
        }

        public UUID getPremiumUuid() {
            return this.premiumUuid;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public long getTtlMillis() {
            return this.ttlMillis;
        }

        public boolean isExpired() {
            return System.currentTimeMillis() - this.timestamp > this.ttlMillis;
        }

        public boolean isStale() {
            return (double)(System.currentTimeMillis() - this.timestamp) > (double)this.ttlMillis * this.refreshThreshold;
        }

        public long getAgeMillis() {
            return System.currentTimeMillis() - this.timestamp;
        }
    }

    private static class BruteForceEntry {
        private int attempts = 0;
        private long firstAttemptTime = System.currentTimeMillis();

        private BruteForceEntry() {
        }

        public void incrementAttempts() {
            ++this.attempts;
        }

        public int getAttempts() {
            return this.attempts;
        }

        public void reset() {
            this.attempts = 0;
            this.firstAttemptTime = System.currentTimeMillis();
        }

        public boolean isExpired(int timeoutMinutes) {
            long timeoutMillis = (long)timeoutMinutes * 60L * 1000L;
            return System.currentTimeMillis() - this.firstAttemptTime > timeoutMillis;
        }
    }

    public static class ActiveSession {
        private final UUID uuid;
        private final String nickname;
        private final String ip;
        private final long sessionStartTime;
        private volatile long lastActivityTime;

        public ActiveSession(UUID uuid, String nickname, String ip) {
            this.uuid = uuid;
            this.nickname = nickname;
            this.ip = ip;
            this.sessionStartTime = System.currentTimeMillis();
            this.lastActivityTime = System.currentTimeMillis();
        }

        public boolean isActive(long timeoutMinutes) {
            long timeoutMillis = timeoutMinutes * 60L * 1000L;
            return System.currentTimeMillis() - this.lastActivityTime < timeoutMillis;
        }

        public void updateActivity() {
            this.lastActivityTime = System.currentTimeMillis();
        }

        public UUID getUuid() {
            return this.uuid;
        }

        public String getNickname() {
            return this.nickname;
        }

        public String getIp() {
            return this.ip;
        }

        public long getSessionStartTime() {
            return this.sessionStartTime;
        }

        public long getLastActivityTime() {
            return this.lastActivityTime;
        }
    }

    public record CacheStats(int authorizedPlayersCount, int bruteForceEntriesCount, int premiumCacheCount, long cacheHits, long cacheMisses, int maxSize, int ttlMinutes) {
        public double getHitRate() {
            long total = this.cacheHits + this.cacheMisses;
            return total == 0L ? 0.0 : (double)this.cacheHits / (double)total * 100.0;
        }

        public long getTotalRequests() {
            return this.cacheHits + this.cacheMisses;
        }
    }
}

