/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.database;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.util.Objects;
import javax.sql.DataSource;
import net.rafalohaki.veloauth.config.Settings;
import net.rafalohaki.veloauth.database.DatabaseType;
import net.rafalohaki.veloauth.database.HikariConfigParams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DatabaseConfig {
    private static final Logger logger = LoggerFactory.getLogger(DatabaseConfig.class);
    private final String storageType;
    private final String hostname;
    private final int port;
    private final String database;
    private final String user;
    private final String password;
    private final int connectionPoolSize;
    private final DataSource dataSource;
    private final String jdbcUrl;

    public DatabaseConfig(String storageType, String hostname, int port, String database, String user, String password, int connectionPoolSize) {
        this.validateBasicParameters(storageType, database, connectionPoolSize);
        DatabaseType dbType = this.validateDatabaseType(storageType);
        this.validateRemoteDatabaseParameters(dbType, hostname, port, storageType);
        this.storageType = dbType.getName();
        this.hostname = hostname;
        this.port = port;
        this.database = database;
        this.user = user;
        this.password = password;
        this.connectionPoolSize = connectionPoolSize;
        this.dataSource = null;
        this.jdbcUrl = DatabaseConfig.buildJdbcUrl(dbType, hostname, port, database, null, null);
    }

    private void validateBasicParameters(String storageType, String database, int connectionPoolSize) {
        if (storageType == null || storageType.isEmpty()) {
            throw new IllegalArgumentException("StorageType nie mo\u017ce by\u0107 pusty");
        }
        if (database == null || database.isEmpty()) {
            throw new IllegalArgumentException("Database nie mo\u017ce by\u0107 pusty");
        }
        if (connectionPoolSize <= 0) {
            throw new IllegalArgumentException("ConnectionPoolSize musi by\u0107 > 0");
        }
    }

    private DatabaseType validateDatabaseType(String storageType) {
        DatabaseType dbType = DatabaseType.fromName(storageType);
        if (dbType == null) {
            throw new IllegalArgumentException("Nieobs\u0142ugiwany typ bazy danych: " + storageType);
        }
        return dbType;
    }

    private void validateRemoteDatabaseParameters(DatabaseType dbType, String hostname, int port, String storageType) {
        if (dbType.isRemoteDatabase()) {
            if (hostname == null || hostname.isEmpty()) {
                throw new IllegalArgumentException("Hostname nie mo\u017ce by\u0107 pusty dla " + storageType);
            }
            if (port <= 0 || port > 65535) {
                throw new IllegalArgumentException("Port musi by\u0107 w zakresie 1-65535");
            }
        }
    }

    public DatabaseConfig(String storageType, DataSource dataSource, int connectionPoolSize, String jdbcUrl) {
        if (storageType == null || storageType.isEmpty()) {
            throw new IllegalArgumentException("StorageType nie mo\u017ce by\u0107 pusty");
        }
        if (dataSource == null) {
            throw new IllegalArgumentException("DataSource nie mo\u017ce by\u0107 null");
        }
        if (jdbcUrl == null || jdbcUrl.isEmpty()) {
            throw new IllegalArgumentException("jdbcUrl nie mo\u017ce by\u0107 pusty");
        }
        this.storageType = storageType.toUpperCase();
        this.dataSource = dataSource;
        this.connectionPoolSize = connectionPoolSize;
        this.jdbcUrl = jdbcUrl;
        this.hostname = null;
        this.port = 0;
        this.database = null;
        this.user = null;
        this.password = null;
    }

    public static DatabaseConfig forLocalDatabase(String storageType, String database) {
        return new DatabaseConfig(storageType, null, 0, database, null, null, 1);
    }

    public static DatabaseConfig forRemoteWithHikari(HikariConfigParams params) {
        DatabaseType dbType = DatabaseType.fromName(params.getStorageType());
        if (dbType == null) {
            throw new IllegalArgumentException("Nieobs\u0142ugiwany typ bazy danych: " + params.getStorageType());
        }
        String jdbcUrl = DatabaseConfig.buildJdbcUrl(dbType, params.getHostname(), params.getPort(), params.getDatabase(), params.getConnectionParameters(), params.getPostgreSQLSettings());
        String driverClass = DatabaseConfig.resolveDriverClass(dbType);
        if (params.isDebugEnabled()) {
            logger.debug("[VeloAuth DEBUG] JDBC URL: {}", (Object)jdbcUrl);
            logger.debug("[VeloAuth DEBUG] Database Type: {}", (Object)dbType);
            logger.debug("[VeloAuth DEBUG] Hostname: {}:{}", (Object)params.getHostname(), (Object)params.getPort());
            logger.debug("[VeloAuth DEBUG] SSL Settings: {}", (Object)(params.getPostgreSQLSettings() != null ? "enabled" : "using defaults"));
        }
        HikariConfig hikariConfig = new HikariConfig();
        DatabaseConfig.configureBasicHikariSettings(hikariConfig, jdbcUrl, params.getUser(), params.getPassword(), params.getConnectionPoolSize(), params.getMaxLifetime());
        DatabaseConfig.configureDatabaseOptimizations(hikariConfig, dbType, params.getPostgreSQLSettings());
        hikariConfig.setDriverClassName(driverClass);
        try {
            Class.forName(driverClass);
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("Nie znaleziono sterownika JDBC: " + driverClass, e);
        }
        HikariDataSource dataSource = new HikariDataSource(hikariConfig);
        return new DatabaseConfig(params.getStorageType(), dataSource, params.getConnectionPoolSize(), jdbcUrl);
    }

    private static void configureBasicHikariSettings(HikariConfig hikariConfig, String jdbcUrl, String user, String password, int connectionPoolSize, int maxLifetime) {
        hikariConfig.setJdbcUrl(jdbcUrl);
        hikariConfig.setUsername(user);
        hikariConfig.setPassword(password);
        hikariConfig.setMaximumPoolSize(connectionPoolSize);
        hikariConfig.setMinimumIdle(5);
        hikariConfig.setMaxLifetime(maxLifetime);
        hikariConfig.setConnectionTimeout(5000L);
        hikariConfig.setIdleTimeout(300000L);
        hikariConfig.setAutoCommit(true);
        hikariConfig.setPoolName("VeloAuth-HikariCP");
        hikariConfig.addDataSourceProperty("loginTimeout", "3");
        hikariConfig.addDataSourceProperty("socketTimeout", "3000");
        hikariConfig.setValidationTimeout(5000L);
        hikariConfig.setLeakDetectionThreshold(10000L);
    }

    private static void configureDatabaseOptimizations(HikariConfig hikariConfig, DatabaseType dbType, Settings.PostgreSQLSettings postgreSQLSettings) {
        if (dbType == DatabaseType.MYSQL) {
            hikariConfig.addDataSourceProperty("cachePrepStmts", "true");
            hikariConfig.addDataSourceProperty("prepStmtCacheSize", "250");
            hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
            hikariConfig.addDataSourceProperty("useServerPrepStmts", "true");
        } else if (dbType == DatabaseType.POSTGRESQL) {
            hikariConfig.addDataSourceProperty("prepareThreshold", "1");
            hikariConfig.addDataSourceProperty("preparedStatementCacheQueries", "256");
            hikariConfig.addDataSourceProperty("preparedStatementCacheSizeMiB", "5");
            DatabaseConfig.configurePostgreSQLSsl(hikariConfig, postgreSQLSettings);
        }
    }

    private static void configurePostgreSQLSsl(HikariConfig hikariConfig, Settings.PostgreSQLSettings postgreSQLSettings) {
        if (postgreSQLSettings != null && postgreSQLSettings.isSslEnabled()) {
            hikariConfig.addDataSourceProperty("ssl", "true");
            hikariConfig.addDataSourceProperty("sslmode", postgreSQLSettings.getSslMode());
            DatabaseConfig.addPropertyIfNotEmpty(hikariConfig, "sslcert", postgreSQLSettings.getSslCert());
            DatabaseConfig.addPropertyIfNotEmpty(hikariConfig, "sslkey", postgreSQLSettings.getSslKey());
            DatabaseConfig.addPropertyIfNotEmpty(hikariConfig, "sslrootcert", postgreSQLSettings.getSslRootCert());
            DatabaseConfig.addPropertyIfNotEmpty(hikariConfig, "sslpassword", postgreSQLSettings.getSslPassword());
        }
    }

    private static void addPropertyIfNotEmpty(HikariConfig hikariConfig, String propertyName, String value) {
        if (value != null && !value.isEmpty()) {
            hikariConfig.addDataSourceProperty(propertyName, value);
        }
    }

    private static String buildJdbcUrl(DatabaseType dbType, String hostname, int port, String database, String connectionParameters, Settings.PostgreSQLSettings postgreSQLSettings) {
        String params = DatabaseConfig.processConnectionParameters(connectionParameters);
        return switch (dbType) {
            case DatabaseType.MYSQL -> DatabaseConfig.buildMySqlUrl(hostname, port, database, params);
            case DatabaseType.POSTGRESQL -> DatabaseConfig.buildPostgreSqlUrl(hostname, port, database, params, postgreSQLSettings);
            case DatabaseType.H2 -> DatabaseConfig.buildH2Url(database);
            case DatabaseType.SQLITE -> DatabaseConfig.buildSqliteUrl(database);
            default -> throw new IllegalArgumentException("Invalid storage type: " + String.valueOf((Object)dbType));
        };
    }

    private static String processConnectionParameters(String connectionParameters) {
        if (connectionParameters == null || connectionParameters.trim().isEmpty()) {
            return "";
        }
        if (!connectionParameters.startsWith("?") && !connectionParameters.startsWith("&")) {
            return "?" + connectionParameters;
        }
        return connectionParameters;
    }

    private static String buildMySqlUrl(String hostname, int port, String database, String params) {
        String baseUrl = String.format("jdbc:mysql://%s:%d/%s", hostname, port, database);
        String mysqlParams = DatabaseConfig.addMySqlDefaultParams(params);
        return baseUrl + mysqlParams;
    }

    private static String addMySqlDefaultParams(String params) {
        if (params.isEmpty()) {
            return "?useSSL=false&serverTimezone=UTC&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048";
        }
        if (!params.contains("useSSL")) {
            return params + "&useSSL=false";
        }
        return params;
    }

    private static String buildPostgreSqlUrl(String hostname, int port, String database, String params, Settings.PostgreSQLSettings postgreSQLSettings) {
        String baseUrl = String.format("jdbc:postgresql://%s:%d/%s", hostname, port, database);
        String sslParams = DatabaseConfig.buildPostgreSQLSslParams(postgreSQLSettings);
        String mergedParams = DatabaseConfig.mergeSslParams(params, sslParams);
        return baseUrl + mergedParams;
    }

    private static String buildH2Url(String database) {
        return String.format("jdbc:h2:file:./data/%s;MODE=MySQL;DATABASE_TO_LOWER=TRUE", database);
    }

    private static String buildSqliteUrl(String database) {
        return String.format("jdbc:sqlite:./data/%s.db", database);
    }

    private static String resolveDriverClass(DatabaseType dbType) {
        return dbType.getDriverClass();
    }

    public static DatabaseConfig forRemoteDatabase(String storageType, String hostname, int port, String database, String user, String password, int connectionPoolSize) {
        return new DatabaseConfig(storageType, hostname, port, database, user, password, connectionPoolSize);
    }

    private static String buildPostgreSQLSslParams(Settings.PostgreSQLSettings postgreSQLSettings) {
        if (postgreSQLSettings == null) {
            return "ssl=true&sslmode=require";
        }
        if (!postgreSQLSettings.isSslEnabled()) {
            return "sslmode=disable";
        }
        return DatabaseConfig.buildEnabledSslParams(postgreSQLSettings);
    }

    private static String buildEnabledSslParams(Settings.PostgreSQLSettings postgreSQLSettings) {
        String sslParams = "ssl=true";
        sslParams = DatabaseConfig.addSslMode(sslParams, postgreSQLSettings.getSslMode());
        sslParams = DatabaseConfig.addSslCert(sslParams, postgreSQLSettings.getSslCert());
        sslParams = DatabaseConfig.addSslKey(sslParams, postgreSQLSettings.getSslKey());
        sslParams = DatabaseConfig.addSslRootCert(sslParams, postgreSQLSettings.getSslRootCert());
        return sslParams;
    }

    private static String addSslMode(String sslParams, String sslMode) {
        if (sslMode == null || sslMode.isEmpty()) {
            return sslParams + "&sslmode=require";
        }
        return sslParams + "&sslmode=" + sslMode;
    }

    private static String addSslCert(String sslParams, String sslCert) {
        if (sslCert != null && !sslCert.isEmpty()) {
            return sslParams + "&sslcert=" + sslCert;
        }
        return sslParams;
    }

    private static String addSslKey(String sslParams, String sslKey) {
        if (sslKey != null && !sslKey.isEmpty()) {
            return sslParams + "&sslkey=" + sslKey;
        }
        return sslParams;
    }

    private static String addSslRootCert(String sslParams, String sslRootCert) {
        if (sslRootCert != null && !sslRootCert.isEmpty()) {
            return sslParams + "&sslrootcert=" + sslRootCert;
        }
        return sslParams;
    }

    private static String mergeSslParams(String existingParams, String sslParams) {
        Object params = existingParams;
        if (params == null || ((String)params).isEmpty()) {
            params = "?" + sslParams;
        } else {
            CharSequence[] paramPairs = existingParams.substring(1).split("&");
            boolean sslReplaced = false;
            for (int i = 0; i < paramPairs.length; ++i) {
                String[] pair = paramPairs[i].split("=", 2);
                if (pair.length != 2 || !"ssl".equals(pair[0])) continue;
                paramPairs[i] = sslParams;
                sslReplaced = true;
                break;
            }
            params = sslReplaced ? "?" + String.join((CharSequence)"&", paramPairs) : (String)params + "&" + sslParams;
        }
        if (!((String)params).contains("prepareThreshold")) {
            params = (String)params + "&prepareThreshold=0";
        }
        return params;
    }

    public String getStorageType() {
        return this.storageType;
    }

    public String getHostname() {
        return this.hostname;
    }

    public int getPort() {
        return this.port;
    }

    public String getDatabase() {
        return this.database;
    }

    public String getUser() {
        return this.user;
    }

    public String getPassword() {
        return this.password;
    }

    public int getConnectionPoolSize() {
        return this.connectionPoolSize;
    }

    public DataSource getDataSource() {
        return this.dataSource;
    }

    public String getJdbcUrl() {
        return this.jdbcUrl;
    }

    public boolean hasDataSource() {
        return this.dataSource != null;
    }

    public boolean isLocalDatabase() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null && dbType.isLocalDatabase();
    }

    public boolean isRemoteDatabase() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null && dbType.isRemoteDatabase();
    }

    public int getDefaultPort() {
        DatabaseType dbType = DatabaseType.fromName(this.storageType);
        return dbType != null ? dbType.getDefaultPort() : 0;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        DatabaseConfig that = (DatabaseConfig)obj;
        return this.port == that.port && this.connectionPoolSize == that.connectionPoolSize && Objects.equals(this.storageType, that.storageType) && Objects.equals(this.hostname, that.hostname) && Objects.equals(this.database, that.database) && Objects.equals(this.user, that.user) && Objects.equals(this.password, that.password) && Objects.equals(this.jdbcUrl, that.jdbcUrl);
    }

    public int hashCode() {
        return Objects.hash(this.storageType, this.hostname, this.port, this.database, this.user, this.password, this.connectionPoolSize, this.jdbcUrl);
    }

    public String toString() {
        return "DatabaseConfig{storageType='" + this.storageType + "', hostname='" + this.hostname + "', port=" + this.port + ", database='" + this.database + "', user='" + this.user + "', password='***', connectionPoolSize=" + this.connectionPoolSize + ", jdbcUrl='" + this.jdbcUrl + "', isLocal=" + this.isLocalDatabase() + "}";
    }
}

