/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.database;

import java.lang.ref.WeakReference;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.rafalohaki.veloauth.cache.AuthCache;
import net.rafalohaki.veloauth.database.DataAccessException;
import net.rafalohaki.veloauth.database.DatabaseConfig;
import net.rafalohaki.veloauth.database.DatabaseType;
import net.rafalohaki.veloauth.database.JdbcAuthDao;
import net.rafalohaki.veloauth.database.PremiumUuidDao;
import net.rafalohaki.veloauth.i18n.Messages;
import net.rafalohaki.veloauth.libs.ormlite.dao.Dao;
import net.rafalohaki.veloauth.libs.ormlite.dao.DaoManager;
import net.rafalohaki.veloauth.libs.ormlite.jdbc.DataSourceConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.jdbc.JdbcConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.misc.TransactionManager;
import net.rafalohaki.veloauth.libs.ormlite.support.ConnectionSource;
import net.rafalohaki.veloauth.libs.ormlite.support.DatabaseConnection;
import net.rafalohaki.veloauth.libs.ormlite.table.TableUtils;
import net.rafalohaki.veloauth.model.PremiumUuid;
import net.rafalohaki.veloauth.model.RegisteredPlayer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public class DatabaseManager {
    private static final Logger logger = LoggerFactory.getLogger(DatabaseManager.class);
    private static final String DATABASE_ERROR = "database.error";
    private static final String ALTER_TABLE = "ALTER TABLE ";
    private static final String ADD_COLUMN = " ADD COLUMN ";
    private static final String AUTH_TABLE = "AUTH";
    private static final String WHERE_CLAUSE = " WHERE ";
    private static final Marker DB_MARKER = MarkerFactory.getMarker("DATABASE");
    private static final Marker CACHE_MARKER = MarkerFactory.getMarker("CACHE");
    private static final String DATABASE_NOT_CONNECTED = "Database not connected";
    private static final String DATABASE_NOT_CONNECTED_PREMIUM_CHECK = "Database not connected - cannot check premium status for {}";
    private final ConcurrentHashMap<String, RegisteredPlayer> playerCache;
    private final ReentrantLock databaseLock;
    private final DatabaseConfig config;
    private final Messages messages;
    private final ExecutorService dbExecutor;
    private final ScheduledExecutorService healthCheckExecutor;
    private ConnectionSource connectionSource;
    private Dao<RegisteredPlayer, String> playerDao;
    private PremiumUuidDao premiumUuidDao;
    private JdbcAuthDao jdbcAuthDao;
    private volatile boolean connected;
    private volatile long lastHealthCheckTime;
    private volatile boolean lastHealthCheckPassed;
    private WeakReference<AuthCache> authCacheRef;

    public DatabaseManager(DatabaseConfig config, Messages messages) {
        if (config == null) {
            throw new IllegalArgumentException("Config cannot be null");
        }
        this.config = config;
        this.messages = messages;
        this.playerCache = new ConcurrentHashMap();
        this.databaseLock = new ReentrantLock();
        this.connected = false;
        this.dbExecutor = Executors.newVirtualThreadPerTaskExecutor();
        this.healthCheckExecutor = Executors.newSingleThreadScheduledExecutor();
        this.jdbcAuthDao = new JdbcAuthDao(config);
        this.lastHealthCheckTime = 0L;
        this.lastHealthCheckPassed = false;
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, messages.get("database.manager.created", new Object[0]), (Object)config.getStorageType());
        }
    }

    public CompletableFuture<Boolean> initialize() {
        return CompletableFuture.supplyAsync(() -> {
            this.databaseLock.lock();
            try {
                Boolean bl = this.performDatabaseInitialization();
                this.databaseLock.unlock();
                return bl;
            }
            catch (Throwable throwable) {
                try {
                    this.databaseLock.unlock();
                    throw throwable;
                }
                catch (SQLException e) {
                    if (logger.isErrorEnabled()) {
                        logger.error(DB_MARKER, "B\u0142\u0105d podczas inicjalizacji bazy danych", e);
                    }
                    return false;
                }
            }
        }, this.dbExecutor);
    }

    private boolean performDatabaseInitialization() throws SQLException {
        if (this.isAlreadyConnected()) {
            return true;
        }
        this.initializeConnection();
        this.initializeDaos();
        this.createTablesIfNotExists();
        this.markAsConnected();
        this.startHealthChecks();
        return true;
    }

    private boolean isAlreadyConnected() {
        if (this.connected) {
            if (logger.isWarnEnabled()) {
                logger.warn(DB_MARKER, "Baza danych ju\u017c jest po\u0142\u0105czona");
            }
            return true;
        }
        return false;
    }

    private void initializeConnection() throws SQLException {
        if (this.config.hasDataSource()) {
            this.initializeHikariConnection();
        } else {
            this.initializeStandardJdbcConnection();
        }
    }

    private void initializeHikariConnection() throws SQLException {
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, this.messages.get("database.manager.hikari_init", new Object[0]));
        }
        this.connectionSource = new DataSourceConnectionSource(this.config.getDataSource(), this.config.getJdbcUrl());
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, this.messages.get("database.manager.hikari_ready", new Object[0]), (Object)this.config.getStorageType());
        }
    }

    private void initializeStandardJdbcConnection() throws SQLException {
        String jdbcUrl = this.config.getJdbcUrl();
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, "Connecting to database (standard JDBC): {}", (Object)jdbcUrl);
        }
        this.connectionSource = new JdbcConnectionSource(jdbcUrl, this.config.getUser(), this.config.getPassword());
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, this.messages.get("database.manager.standard_jdbc", new Object[0]), (Object)this.config.getStorageType());
        }
    }

    private void initializeDaos() throws SQLException {
        this.playerDao = DaoManager.createDao(this.connectionSource, RegisteredPlayer.class);
        this.premiumUuidDao = new PremiumUuidDao(this.connectionSource);
        this.jdbcAuthDao = new JdbcAuthDao(this.config);
    }

    private void markAsConnected() {
        this.connected = true;
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, this.messages.get("database.manager.connected", new Object[0]), (Object)this.config.getStorageType());
        }
    }

    public <T> CompletableFuture<T> executeInTransaction(Callable<T> operation) {
        return CompletableFuture.supplyAsync(() -> {
            this.validateConnectionSource();
            return this.executeTransaction(operation);
        }, this.dbExecutor);
    }

    private void validateConnectionSource() {
        if (this.connectionSource == null) {
            throw new IllegalStateException(DATABASE_NOT_CONNECTED);
        }
    }

    private <T> T executeTransaction(Callable<T> operation) {
        try {
            TransactionManager transactionManager = new TransactionManager(this.connectionSource);
            return transactionManager.callInTransaction(operation);
        }
        catch (SQLException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d SQL podczas transakcji", e);
            }
            throw new DataAccessException("SQL transaction failed", e);
        }
        catch (IllegalStateException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d stanu w transakcji DB", e);
            }
            throw new DataAccessException("Transaction state failed", e);
        }
        catch (RuntimeException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d wykonania w transakcji DB", e);
            }
            throw new DataAccessException("Transaction execution failed", e);
        }
    }

    private void startHealthChecks() {
        this.healthCheckExecutor.scheduleAtFixedRate(() -> {
            block2: {
                try {
                    this.performHealthCheck();
                }
                catch (RuntimeException e) {
                    if (!logger.isErrorEnabled()) break block2;
                    logger.error(DB_MARKER, "B\u0142\u0105d podczas health check bazy danych", e);
                }
            }
        }, 30L, 30L, TimeUnit.SECONDS);
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, this.messages.get("database.manager.health_checks_started", new Object[0]));
        }
    }

    private void performHealthCheck() {
        block8: {
            try {
                if (!this.connected) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(DB_MARKER, "Health check pomini\u0119ty - baza danych nie jest po\u0142\u0105czona");
                    }
                    return;
                }
                boolean healthy = this.jdbcAuthDao.healthCheck();
                this.lastHealthCheckTime = System.currentTimeMillis();
                this.lastHealthCheckPassed = healthy;
                if (!healthy) {
                    if (logger.isWarnEnabled()) {
                        logger.warn(DB_MARKER, "\u274c Database health check FAILED - connection may be unstable");
                    }
                } else if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "\u2705 Database health check PASSED");
                }
            }
            catch (RuntimeException e) {
                this.lastHealthCheckTime = System.currentTimeMillis();
                this.lastHealthCheckPassed = false;
                if (!logger.isErrorEnabled()) break block8;
                logger.error(DB_MARKER, "\u274c Database health check FAILED with exception: {}", (Object)e.getMessage());
            }
        }
    }

    public boolean isHealthy() {
        return this.connected;
    }

    public long getLastHealthCheckTime() {
        return this.lastHealthCheckTime;
    }

    public boolean wasLastHealthCheckPassed() {
        return this.lastHealthCheckPassed;
    }

    public void shutdown() {
        try {
            this.stopHealthChecks();
            this.databaseLock.lock();
            try {
                this.closeConnectionResources();
            }
            finally {
                this.databaseLock.unlock();
            }
        }
        catch (RuntimeException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas zamykania bazy danych", e);
            }
        }
        finally {
            this.dbExecutor.shutdown();
        }
    }

    private void stopHealthChecks() {
        if (this.healthCheckExecutor != null && !this.healthCheckExecutor.isShutdown()) {
            this.healthCheckExecutor.shutdown();
            try {
                if (!this.healthCheckExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.healthCheckExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.healthCheckExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
            if (logger.isInfoEnabled()) {
                logger.info(DB_MARKER, "Health checks zatrzymane");
            }
        }
    }

    private void closeConnectionResources() {
        if (this.connectionSource != null) {
            block5: {
                try {
                    this.connectionSource.close();
                }
                catch (Exception e) {
                    if (!logger.isErrorEnabled()) break block5;
                    logger.error(DB_MARKER, "Error closing database connection", e);
                }
            }
            this.connectionSource = null;
            if (logger.isInfoEnabled()) {
                logger.info(DB_MARKER, this.messages.get("database.manager.connection_closed", new Object[0]));
            }
        }
        this.connected = false;
        this.playerCache.clear();
        if (logger.isDebugEnabled()) {
            logger.debug(CACHE_MARKER, "Cache graczy wyczyszczony");
        }
    }

    public CompletableFuture<DbResult<RegisteredPlayer>> findPlayerByNickname(String nickname) {
        if (nickname == null || nickname.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(null));
        }
        String normalizedNickname = nickname.toLowerCase();
        return CompletableFuture.supplyAsync(() -> {
            DbResult<RegisteredPlayer> cacheResult = this.checkCacheSafe(normalizedNickname);
            if (cacheResult.isDatabaseError()) {
                return cacheResult;
            }
            RegisteredPlayer cachedPlayer = cacheResult.getValue();
            if (cachedPlayer != null) {
                return DbResult.success(cachedPlayer);
            }
            DbResult<Void> connectionResult = this.validateDatabaseConnection();
            if (connectionResult.isDatabaseError()) {
                return DbResult.databaseError(connectionResult.getErrorMessage());
            }
            return this.queryPlayerFromDatabase(normalizedNickname);
        }, this.dbExecutor);
    }

    private boolean isCacheCorrupted(RegisteredPlayer cached, String normalizedNickname) {
        return !cached.getLowercaseNickname().equals(normalizedNickname);
    }

    private void handleCacheCorruption(String normalizedNickname) {
        this.playerCache.remove(normalizedNickname);
        if (logger.isWarnEnabled()) {
            logger.warn(CACHE_MARKER, "Cache corruption detected for {} - removing invalid entry", (Object)normalizedNickname);
        }
    }

    private void logCacheHit(String normalizedNickname) {
        if (logger.isDebugEnabled()) {
            logger.debug(CACHE_MARKER, "Cache HIT dla gracza: {}", (Object)normalizedNickname);
        }
    }

    private DbResult<RegisteredPlayer> queryPlayerFromDatabase(String normalizedNickname) {
        try {
            RegisteredPlayer player = this.jdbcAuthDao.findPlayerByLowercaseNickname(normalizedNickname);
            if (player != null) {
                this.handleDatabaseResult(player, normalizedNickname);
            } else {
                this.logPlayerNotFound(normalizedNickname);
            }
            return DbResult.success(player);
        }
        catch (SQLException e) {
            return this.handleDatabaseError(normalizedNickname, e);
        }
    }

    private void handleDatabaseResult(RegisteredPlayer player, String normalizedNickname) {
        if (player.getLowercaseNickname().equals(normalizedNickname)) {
            this.playerCache.put(normalizedNickname, player);
            if (logger.isDebugEnabled()) {
                logger.debug(CACHE_MARKER, "Cache MISS -> DB HIT dla gracza: {}", (Object)normalizedNickname);
            }
        } else if (logger.isWarnEnabled()) {
            logger.warn(CACHE_MARKER, "Database inconsistency for {} - expected {}, found {}", normalizedNickname, normalizedNickname, player.getLowercaseNickname());
        }
    }

    private void logPlayerNotFound(String normalizedNickname) {
        if (logger.isDebugEnabled()) {
            logger.debug(DB_MARKER, "Gracz nie znaleziony: {}", (Object)normalizedNickname);
        }
    }

    private DbResult<RegisteredPlayer> handleDatabaseError(String normalizedNickname, SQLException e) {
        if (logger.isErrorEnabled()) {
            logger.error(DB_MARKER, "B\u0142\u0105d podczas wyszukiwania gracza: {}", (Object)normalizedNickname, (Object)e);
        }
        return DbResult.databaseError(this.messages.get(DATABASE_ERROR, new Object[0]) + ": " + e.getMessage());
    }

    public CompletableFuture<DbResult<Boolean>> savePlayer(RegisteredPlayer player) {
        if (player == null) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            DbResult<Void> connectionResult = this.validateDatabaseConnection();
            if (connectionResult.isDatabaseError()) {
                return DbResult.databaseError(connectionResult.getErrorMessage());
            }
            return this.executePlayerSave(player);
        }, this.dbExecutor);
    }

    private DbResult<Void> validateDatabaseConnection() {
        if (!this.connected) {
            if (logger.isWarnEnabled()) {
                logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
            }
            return DbResult.databaseError(DATABASE_NOT_CONNECTED);
        }
        return DbResult.success(null);
    }

    private DbResult<Boolean> executePlayerSave(RegisteredPlayer player) {
        try {
            boolean success = this.jdbcAuthDao.upsertPlayer(player);
            if (success) {
                this.playerCache.put(player.getLowercaseNickname(), player);
                if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "Zapisano gracza (upsert): {}", (Object)player.getNickname());
                }
                this.notifyAuthCacheOfUpdate(player);
            }
            return DbResult.success(success);
        }
        catch (SQLException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas zapisywania gracza: {}", (Object)player.getNickname(), (Object)e);
            }
            return DbResult.databaseError(this.messages.get(DATABASE_ERROR, new Object[0]) + ": " + e.getMessage());
        }
    }

    private void notifyAuthCacheOfUpdate(RegisteredPlayer player) {
        block8: {
            if (this.authCacheRef == null) {
                return;
            }
            try {
                AuthCache authCache = (AuthCache)this.authCacheRef.get();
                if (authCache == null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(DB_MARKER, "AuthCache reference is null (GC collected) - skipping cache invalidation");
                    }
                    return;
                }
                UUID playerUuid = UUID.fromString(player.getUuid());
                authCache.invalidatePlayerData(playerUuid);
                if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "Notified AuthCache of update for player: {} (UUID: {})", (Object)player.getNickname(), (Object)playerUuid);
                }
            }
            catch (IllegalArgumentException e) {
                if (logger.isWarnEnabled()) {
                    logger.warn(DB_MARKER, "Failed to parse UUID for cache invalidation: {} - {}", (Object)player.getUuid(), (Object)e.getMessage());
                }
            }
            catch (Exception e) {
                if (!logger.isErrorEnabled()) break block8;
                logger.error(DB_MARKER, "Error notifying AuthCache of player update: {}", (Object)player.getNickname(), (Object)e);
            }
        }
    }

    public CompletableFuture<DbResult<Boolean>> deletePlayer(String lowercaseNickname) {
        if (lowercaseNickname == null || lowercaseNickname.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            DbResult<Void> connectionResult = this.validateDatabaseConnection();
            if (connectionResult.isDatabaseError()) {
                return DbResult.databaseError(connectionResult.getErrorMessage());
            }
            return this.executePlayerDelete(lowercaseNickname);
        }, this.dbExecutor);
    }

    private DbResult<Boolean> executePlayerDelete(String lowercaseNickname) {
        try {
            boolean deleted = this.jdbcAuthDao.deletePlayer(lowercaseNickname);
            this.playerCache.remove(lowercaseNickname);
            if (deleted) {
                if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "Usuni\u0119to gracza: {}", (Object)lowercaseNickname);
                }
                return DbResult.success(true);
            }
            if (logger.isDebugEnabled()) {
                logger.debug(DB_MARKER, "Gracz nie znaleziony do usuni\u0119cia: {}", (Object)lowercaseNickname);
            }
            return DbResult.success(false);
        }
        catch (SQLException e) {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "B\u0142\u0105d podczas usuwania gracza: {}", (Object)lowercaseNickname, (Object)e);
            }
            return DbResult.databaseError(this.messages.get(DATABASE_ERROR, new Object[0]) + ": " + e.getMessage());
        }
    }

    public CompletableFuture<DbResult<Boolean>> isPremium(String username) {
        if (username == null || username.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(false));
        }
        return CompletableFuture.supplyAsync(() -> {
            if (!this.connected) {
                if (logger.isWarnEnabled()) {
                    logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED_PREMIUM_CHECK, (Object)username);
                }
                return DbResult.databaseError(DATABASE_NOT_CONNECTED);
            }
            try {
                boolean premium = this.premiumUuidDao.findByNickname(username).isPresent();
                if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "Premium status z PREMIUM_UUIDS dla {}: {}", (Object)username, (Object)premium);
                }
                return DbResult.success(premium);
            }
            catch (RuntimeException e) {
                if (logger.isErrorEnabled()) {
                    logger.error(DB_MARKER, "B\u0142\u0105d wykonania podczas sprawdzania premium status dla gracza: {}", (Object)username, (Object)e);
                }
                return DbResult.databaseError(this.messages.get(DATABASE_ERROR, new Object[0]) + ": " + e.getMessage());
            }
        }, this.dbExecutor);
    }

    public CompletableFuture<List<RegisteredPlayer>> getAllPlayers() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                if (!this.connected || this.playerDao == null) {
                    if (logger.isWarnEnabled()) {
                        logger.warn(DB_MARKER, DATABASE_NOT_CONNECTED);
                    }
                    return List.of();
                }
                List<RegisteredPlayer> players = this.playerDao.queryForAll();
                if (logger.isDebugEnabled()) {
                    logger.debug(DB_MARKER, "Pobrano {} graczy z bazy danych", (Object)players.size());
                }
                return players;
            }
            catch (SQLException e) {
                if (logger.isErrorEnabled()) {
                    logger.error(DB_MARKER, "B\u0142\u0105d podczas pobierania wszystkich graczy", e);
                }
                return List.of();
            }
        }, this.dbExecutor);
    }

    public void clearCache() {
        this.playerCache.clear();
        if (logger.isDebugEnabled()) {
            logger.debug(CACHE_MARKER, "Cache graczy wyczyszczony");
        }
    }

    public void removeCachedPlayer(String lowercaseNickname) {
        if (lowercaseNickname != null) {
            this.playerCache.remove(lowercaseNickname);
            if (logger.isDebugEnabled()) {
                logger.debug("Usuni\u0119to z cache gracza: {}", (Object)lowercaseNickname);
            }
        }
    }

    public int getCacheSize() {
        return this.playerCache.size();
    }

    public boolean isConnected() {
        return this.isHealthy();
    }

    public CompletableFuture<Integer> getTotalNonPremiumAccounts() {
        return CompletableFuture.supplyAsync(() -> {
            if (!this.connected) {
                return 0;
            }
            try {
                boolean postgres = DatabaseType.POSTGRESQL.getName().equalsIgnoreCase(this.config.getStorageType());
                String auth = postgres ? "\"AUTH\"" : AUTH_TABLE;
                String hash = postgres ? "\"HASH\"" : "HASH";
                String sql = "SELECT COUNT(*) FROM " + auth + WHERE_CLAUSE + hash + " IS NOT NULL";
                return this.executeCountQuery(sql);
            }
            catch (SQLException e) {
                if (logger.isErrorEnabled()) {
                    logger.error(DB_MARKER, "Error counting non-premium accounts", e);
                }
                return 0;
            }
        }, this.dbExecutor);
    }

    /*
     * Exception decompiling
     */
    private int executeCountQuery(String sql) throws SQLException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    public CompletableFuture<Integer> getTotalRegisteredAccounts() {
        return ((CompletableFuture)this.getAllPlayers().thenApply(Collection::size)).exceptionally(e -> {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "Error getting total registered accounts", (Throwable)e);
            }
            return 0;
        });
    }

    public CompletableFuture<Integer> getTotalPremiumAccounts() {
        return ((CompletableFuture)this.getAllPlayers().thenApply(players -> (int)players.stream().filter(player -> player.getPremiumUuid() != null || player.getHash() == null).count())).exceptionally(e -> {
            if (logger.isErrorEnabled()) {
                logger.error(DB_MARKER, "Error getting total premium accounts", (Throwable)e);
            }
            return 0;
        });
    }

    public DatabaseConfig getConfig() {
        return this.config;
    }

    public PremiumUuidDao getPremiumUuidDao() {
        return this.premiumUuidDao;
    }

    public void setAuthCacheReference(AuthCache authCache) {
        if (authCache != null) {
            this.authCacheRef = new WeakReference<AuthCache>(authCache);
            if (logger.isDebugEnabled()) {
                logger.debug(DB_MARKER, "AuthCache reference set for cache invalidation coordination");
            }
        }
    }

    private void createTablesIfNotExists() throws SQLException {
        if (logger.isInfoEnabled()) {
            logger.info(this.messages.get("database.manager.creating_tables", new Object[0]));
        }
        TableUtils.createTableIfNotExists(this.connectionSource, RegisteredPlayer.class);
        TableUtils.createTableIfNotExists(this.connectionSource, PremiumUuid.class);
        this.migrateAuthTableForLimboauth();
        this.createIndexesIfNotExists();
        if (logger.isInfoEnabled()) {
            logger.info(this.messages.get("database.manager.tables_created", new Object[0]));
        }
    }

    private void migrateAuthTableForLimboauth() throws SQLException {
        DatabaseConnection dbConnection = this.connectionSource.getReadWriteConnection(null);
        try {
            this.performColumnMigration(dbConnection);
        }
        finally {
            this.connectionSource.releaseConnection(dbConnection);
        }
    }

    private void performColumnMigration(DatabaseConnection dbConnection) throws SQLException {
        Connection connection = dbConnection.getUnderlyingConnection();
        ColumnMigrationResult migrationResult = this.checkExistingColumns(connection);
        DatabaseType dbType = DatabaseType.fromName(this.config.getStorageType());
        String quote = dbType == DatabaseType.POSTGRESQL ? "\"" : "`";
        this.addMissingColumns(connection, migrationResult, quote);
        this.logMigrationComplete(migrationResult);
    }

    private ColumnMigrationResult checkExistingColumns(Connection connection) throws SQLException {
        boolean hasPremiumUuid = this.columnExists(connection, AUTH_TABLE, "PREMIUMUUID");
        boolean hasTotpToken = this.columnExists(connection, AUTH_TABLE, "TOTPTOKEN");
        boolean hasIssuedTime = this.columnExists(connection, AUTH_TABLE, "ISSUEDTIME");
        boolean hasConflictMode = this.columnExists(connection, AUTH_TABLE, "CONFLICT_MODE");
        boolean hasConflictTimestamp = this.columnExists(connection, AUTH_TABLE, "CONFLICT_TIMESTAMP");
        boolean hasOriginalNickname = this.columnExists(connection, AUTH_TABLE, "ORIGINAL_NICKNAME");
        return new ColumnMigrationResult(hasPremiumUuid, hasTotpToken, hasIssuedTime, hasConflictMode, hasConflictTimestamp, hasOriginalNickname);
    }

    private void addMissingColumns(Connection connection, ColumnMigrationResult result, String quote) throws SQLException {
        if (!result.hasPremiumUuid) {
            this.addColumn(connection, quote, "PREMIUMUUID", "VARCHAR(36)", "Dodano kolumn\u0119 PREMIUMUUID do tabeli AUTH");
        }
        if (!result.hasTotpToken) {
            this.addColumn(connection, quote, "TOTPTOKEN", "VARCHAR(32)", "Dodano kolumn\u0119 TOTPTOKEN do tabeli AUTH");
        }
        if (!result.hasIssuedTime) {
            this.addColumn(connection, quote, "ISSUEDTIME", "BIGINT DEFAULT 0", "Dodano kolumn\u0119 ISSUEDTIME do tabeli AUTH");
        }
        if (!result.hasConflictMode) {
            this.addColumn(connection, quote, "CONFLICT_MODE", "BOOLEAN DEFAULT FALSE", "Dodano kolumn\u0119 CONFLICT_MODE do tabeli AUTH");
        }
        if (!result.hasConflictTimestamp) {
            this.addColumn(connection, quote, "CONFLICT_TIMESTAMP", "BIGINT DEFAULT 0", "Dodano kolumn\u0119 CONFLICT_TIMESTAMP do tabeli AUTH");
        }
        if (!result.hasOriginalNickname) {
            this.addColumn(connection, quote, "ORIGINAL_NICKNAME", "VARCHAR(16)", "Dodano kolumn\u0119 ORIGINAL_NICKNAME do tabeli AUTH");
        }
    }

    private void addColumn(Connection connection, String quote, String columnName, String columnDefinition, String logMessage) throws SQLException {
        String sql = ALTER_TABLE + quote + AUTH_TABLE + quote + ADD_COLUMN + quote + columnName + quote + " " + columnDefinition;
        this.executeAlterTable(connection, sql);
        if (logger.isInfoEnabled()) {
            logger.info(DB_MARKER, logMessage);
        }
    }

    private void logMigrationComplete(ColumnMigrationResult result) {
        if (!(!logger.isInfoEnabled() || result.hasPremiumUuid && result.hasTotpToken && result.hasIssuedTime)) {
            logger.info(DB_MARKER, "Migracja schematu AUTH dla limboauth zako\u0144czona");
        }
    }

    private void executeAlterTable(Connection connection, String sql) throws SQLException {
        try (Statement stmt = connection.createStatement();){
            stmt.execute(sql);
        }
    }

    private boolean columnExists(Connection connection, String tableName, String columnName) throws SQLException {
        DatabaseMetaData metaData = connection.getMetaData();
        try (ResultSet columns = metaData.getColumns(null, null, tableName, columnName);){
            boolean bl = columns.next();
            return bl;
        }
    }

    private void createIndexesIfNotExists() {
        DatabaseType dbType = DatabaseType.fromName(this.config.getStorageType());
        if (dbType == DatabaseType.POSTGRESQL) {
            this.createPostgreSqlIndexes();
        } else if (dbType == DatabaseType.MYSQL) {
            this.createMySqlIndexes();
        }
    }

    private void createPostgreSqlIndexes() {
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_ip ON \"AUTH\" (\"IP\")");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_uuid ON \"AUTH\" (\"UUID\")");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_logindate ON \"AUTH\" (\"LOGINDATE\")");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_regdate ON \"AUTH\" (\"REGDATE\")");
    }

    private void createMySqlIndexes() {
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_ip ON `AUTH` (`IP`)");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_uuid ON `AUTH` (`UUID`)");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_logindate ON `AUTH` (`LOGINDATE`)");
        this.createIndexSafely("CREATE INDEX IF NOT EXISTS idx_auth_regdate ON `AUTH` (`REGDATE`)");
    }

    private void createIndexSafely(String sql) {
        block2: {
            try {
                this.executeUpdate(sql);
            }
            catch (SQLException e) {
                if (!logger.isDebugEnabled()) break block2;
                logger.debug("Index creation failed (may already exist): {}", (Object)e.getMessage());
            }
        }
    }

    private void executeUpdate(String sql) throws SQLException {
        if (this.connectionSource != null) {
            DatabaseConnection connection = null;
            try {
                connection = this.connectionSource.getReadWriteConnection(null);
                connection.executeStatement(sql, 0);
            }
            finally {
                if (connection != null) {
                    this.connectionSource.releaseConnection(connection);
                }
            }
        }
    }

    public boolean isPlayerPremiumRuntime(RegisteredPlayer player) {
        if (player == null) {
            return false;
        }
        String hash = player.getHash();
        return hash == null || hash.isEmpty() || hash.isBlank();
    }

    public CompletableFuture<DbResult<RegisteredPlayer>> findPlayerWithRuntimeDetection(String nickname) {
        if (nickname == null || nickname.isEmpty()) {
            return CompletableFuture.completedFuture(DbResult.success(null));
        }
        String normalizedNickname = nickname.toLowerCase();
        return CompletableFuture.supplyAsync(() -> {
            DbResult<RegisteredPlayer> cacheResult = this.checkCacheSafe(normalizedNickname);
            if (cacheResult.isDatabaseError()) {
                return cacheResult;
            }
            RegisteredPlayer cachedPlayer = cacheResult.getValue();
            if (cachedPlayer != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug(CACHE_MARKER, "Runtime detection - cache HIT: {}", (Object)normalizedNickname);
                }
                return DbResult.success(cachedPlayer);
            }
            if (logger.isDebugEnabled()) {
                logger.debug(CACHE_MARKER, "Runtime detection - cache MISS: {}", (Object)normalizedNickname);
            }
            return this.queryPlayerWithRuntimeDetection(normalizedNickname, nickname);
        }, this.dbExecutor);
    }

    private DbResult<RegisteredPlayer> checkCacheSafe(String normalizedNickname) {
        RegisteredPlayer cached = this.playerCache.get(normalizedNickname);
        if (cached == null) {
            return DbResult.success(null);
        }
        if (this.isCacheCorrupted(cached, normalizedNickname)) {
            this.handleCacheCorruption(normalizedNickname);
            return DbResult.success(null);
        }
        this.logCacheHit(normalizedNickname);
        return DbResult.success(cached);
    }

    private DbResult<RegisteredPlayer> queryPlayerWithRuntimeDetection(String normalizedNickname, String originalNickname) {
        try {
            RegisteredPlayer player = this.jdbcAuthDao.findPlayerByLowercaseNickname(normalizedNickname);
            if (player == null) {
                return DbResult.success(null);
            }
            boolean isPremium = this.isPlayerPremiumRuntime(player);
            this.playerCache.put(normalizedNickname, player);
            this.logRuntimeDetection(originalNickname, isPremium, player.getHash());
            return DbResult.success(player);
        }
        catch (SQLException e) {
            logger.error("Database error during runtime detection for player: {}", (Object)originalNickname, (Object)e);
            return DbResult.databaseError("Database error: " + e.getMessage());
        }
    }

    private void logRuntimeDetection(String nickname, boolean isPremium, String hash) {
        if (logger.isDebugEnabled()) {
            logger.debug("[RUNTIME DETECTION] {} -> {} (hash empty: {})", nickname, isPremium ? "PREMIUM" : "OFFLINE", hash == null || hash.isEmpty());
        }
    }

    public CompletableFuture<List<RegisteredPlayer>> findPlayersInConflictMode() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.jdbcAuthDao.findAllPlayersInConflictMode();
            }
            catch (SQLException e) {
                logger.error("Database error while finding players in conflict mode", e);
                return List.of();
            }
        }, this.dbExecutor);
    }

    public static final class DbResult<T> {
        private final T value;
        private final boolean isDatabaseError;
        private final String errorMessage;

        private DbResult(T value, boolean isDatabaseError, String errorMessage) {
            this.value = value;
            this.isDatabaseError = isDatabaseError;
            this.errorMessage = errorMessage;
        }

        public static <T> DbResult<T> success(T value) {
            return new DbResult<T>(value, false, null);
        }

        public static <T> DbResult<T> databaseError(String errorMessage) {
            return new DbResult<Object>(null, true, errorMessage);
        }

        @Nullable
        public T getValue() {
            return this.value;
        }

        @Nonnull
        public Optional<T> getValueOptional() {
            return Objects.requireNonNull(Optional.ofNullable(this.value), "Optional cannot be null");
        }

        public boolean isDatabaseError() {
            return this.isDatabaseError;
        }

        @Nullable
        public String getErrorMessage() {
            return this.errorMessage;
        }

        public boolean isSuccess() {
            return !this.isDatabaseError;
        }
    }

    private record ColumnMigrationResult(boolean hasPremiumUuid, boolean hasTotpToken, boolean hasIssuedTime, boolean hasConflictMode, boolean hasConflictTimestamp, boolean hasOriginalNickname) {
    }
}

