/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.i18n;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.concurrent.ConcurrentHashMap;
import net.rafalohaki.veloauth.i18n.LanguageFileManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Messages {
    private static final Logger logger = LoggerFactory.getLogger(Messages.class);
    private static final Map<String, Properties> messageCache = new ConcurrentHashMap<String, Properties>();
    private String currentLanguage = "en";
    private final LanguageFileManager languageFileManager;
    private ResourceBundle bundle;
    private final boolean useExternalFiles;

    public Messages(Path dataDirectory, String language) throws IOException {
        this.languageFileManager = new LanguageFileManager(dataDirectory);
        this.currentLanguage = language != null ? language.toLowerCase(Locale.ROOT) : "en";
        this.useExternalFiles = true;
        try {
            this.languageFileManager.initializeLanguageFiles();
            this.reload();
        }
        catch (IOException e) {
            throw new IOException("Language initialization failed: " + e.getMessage(), e);
        }
    }

    public Messages() {
        this.languageFileManager = null;
        this.useExternalFiles = false;
        this.currentLanguage = "en";
    }

    public void setLanguage(String language) {
        if (language == null || language.trim().isEmpty()) {
            logger.warn("Invalid language provided, using default 'en'");
            this.currentLanguage = "en";
            return;
        }
        if (!this.isLanguageSupported(language)) {
            logger.warn("Unsupported language '{}', falling back to 'en'", (Object)language);
            this.currentLanguage = "en";
            return;
        }
        this.currentLanguage = language.toLowerCase(Locale.ROOT);
        logger.info("Language set to: {}", (Object)this.currentLanguage);
        if (this.useExternalFiles) {
            try {
                this.reload();
            }
            catch (IOException e) {
                logger.error("Failed to reload language files", e);
            }
        } else {
            this.loadMessages(this.currentLanguage);
        }
    }

    public void reload() throws IOException {
        if (!this.useExternalFiles) {
            logger.warn("Cannot reload - not using external files");
            return;
        }
        this.bundle = this.languageFileManager.loadLanguageBundle(this.currentLanguage);
        logger.info("Loaded language: {}", (Object)this.currentLanguage);
    }

    public String getCurrentLanguage() {
        return this.currentLanguage;
    }

    public String get(String key, Object ... args) {
        if (this.useExternalFiles) {
            return this.getFromBundle(key, args);
        }
        return this.get(this.currentLanguage, key, args);
    }

    public String get(String language, String key, Object ... args) {
        Properties messages = this.loadMessages(language);
        String message = messages.getProperty(key);
        if (message == null) {
            logger.debug("Message key '{}' not found for language '{}', falling back to English", (Object)key, (Object)language);
            if (!"en".equals(language)) {
                Properties englishMessages = this.loadMessages("en");
                message = englishMessages.getProperty(key);
            }
            if (message == null) {
                logger.warn("Message key '{}' not found in any language file", (Object)key);
                message = key;
            }
        }
        if (args.length > 0) {
            try {
                return MessageFormat.format(message, args);
            }
            catch (IllegalArgumentException e) {
                if (logger.isWarnEnabled()) {
                    logger.warn("Failed to format message '{}': {}", (Object)key, (Object)e.getMessage());
                }
                return message;
            }
        }
        return message;
    }

    private String getFromBundle(String key, Object ... args) {
        try {
            String message = this.bundle.getString(key);
            return this.formatMessageSafely(message, key, args);
        }
        catch (MissingResourceException e) {
            logger.warn("Missing translation key: {}", (Object)key);
            return "Missing: " + key;
        }
    }

    private String formatMessageSafely(String message, String key, Object ... args) {
        if (args.length == 0) {
            return message;
        }
        try {
            return MessageFormat.format(message, args);
        }
        catch (IllegalArgumentException e) {
            logger.warn("Failed to format message '{}': {}", (Object)key, (Object)e.getMessage());
            return message;
        }
    }

    public boolean isLanguageSupported(String language) {
        if (language == null) {
            return false;
        }
        String lang = language.toLowerCase(Locale.ROOT);
        if (this.useExternalFiles && this.languageFileManager != null) {
            try {
                this.languageFileManager.loadLanguageBundle(lang);
                return true;
            }
            catch (IOException e) {
                return false;
            }
        }
        return "en".equals(lang) || "pl".equals(lang);
    }

    public String[] getSupportedLanguages() {
        return new String[]{"en", "pl"};
    }

    public void clearCache() {
        messageCache.clear();
        logger.info("Message cache cleared");
    }

    private Properties loadMessages(String language) {
        return messageCache.computeIfAbsent(language, this::loadMessagesFromFile);
    }

    private Properties loadMessagesFromFile(String language) {
        Properties properties = new Properties();
        String fileName = "lang/messages_" + language + ".properties";
        try (InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(fileName);){
            if (inputStream != null) {
                properties.load(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
                logger.debug("Loaded {} messages from file: {}", (Object)properties.size(), (Object)fileName);
            } else {
                logger.error("Could not find language file: {}", (Object)fileName);
            }
        }
        catch (Exception e) {
            logger.error("Error loading language file: {}", (Object)fileName, (Object)e);
        }
        return properties;
    }

    public String getLanguageDisplayName(String languageCode) {
        return switch (languageCode.toLowerCase(Locale.ROOT)) {
            case "en" -> "English";
            case "pl" -> "Polish";
            default -> languageCode;
        };
    }

    public String getLanguageNativeName(String languageCode) {
        return switch (languageCode.toLowerCase(Locale.ROOT)) {
            case "en" -> "English";
            case "pl" -> "Polski";
            default -> languageCode;
        };
    }
}

