/*
 * Decompiled with CFR 0.152.
 */
package net.rafalohaki.veloauth.premium;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import net.rafalohaki.veloauth.config.Settings;
import net.rafalohaki.veloauth.database.PremiumUuidDao;
import net.rafalohaki.veloauth.model.PremiumUuid;
import net.rafalohaki.veloauth.premium.ConfigurablePremiumResolver;
import net.rafalohaki.veloauth.premium.PremiumResolution;
import net.rafalohaki.veloauth.premium.PremiumResolver;
import net.rafalohaki.veloauth.premium.ResolverConfig;
import org.slf4j.Logger;

public class PremiumResolverService {
    private static final Pattern VALID_USERNAME = Pattern.compile("^\\w{3,16}$");
    private static final String RESOLVER_SERVICE = "resolver-service";
    private final Logger logger;
    private final PremiumUuidDao dao;
    private final List<PremiumResolver> resolvers;
    private final ConcurrentHashMap<String, CachedEntry> cache = new ConcurrentHashMap();
    private final long premiumTtlMillis;
    private final long missTtlMillis;
    private final int maxCacheSize;
    private final ReentrantLock cacheSizeLock = new ReentrantLock();

    public PremiumResolverService(Logger logger, Settings settings, PremiumUuidDao premiumUuidDao) {
        this.logger = Objects.requireNonNull(logger, "logger");
        Settings.PremiumResolverSettings rs = Objects.requireNonNull(settings, "settings").getPremiumResolverSettings();
        this.dao = Objects.requireNonNull(premiumUuidDao, "premiumUuidDao");
        logger.info("[PremiumResolver] Config - Mojang: {}, Ashcon: {}, Wpme: {}", rs.isMojangEnabled(), rs.isAshconEnabled(), rs.isWpmeEnabled());
        int timeoutMs = Math.max(100, rs.getRequestTimeoutMs());
        ArrayList<ConfigurablePremiumResolver> resolverList = new ArrayList<ConfigurablePremiumResolver>();
        resolverList.add(new ConfigurablePremiumResolver(logger, rs.isMojangEnabled(), timeoutMs, ResolverConfig.MOJANG));
        resolverList.add(new ConfigurablePremiumResolver(logger, rs.isAshconEnabled(), timeoutMs, ResolverConfig.ASHCON));
        resolverList.add(new ConfigurablePremiumResolver(logger, rs.isWpmeEnabled(), timeoutMs, ResolverConfig.WPME));
        this.resolvers = Collections.unmodifiableList(resolverList);
        this.premiumTtlMillis = Math.max(0L, (long)rs.getHitTtlMinutes()) * 60000L;
        this.missTtlMillis = Math.max(0L, (long)rs.getMissTtlMinutes()) * 60000L;
        this.maxCacheSize = 10000;
    }

    private PremiumResolution resolveFromApi(String trimmed, String cacheKey) {
        PremiumResolution offlineCandidate = this.tryApiResolvers(trimmed);
        if (offlineCandidate != null) {
            this.cacheResult(cacheKey, offlineCandidate);
            return offlineCandidate;
        }
        this.logger.warn("[PremiumResolver] No premium resolvers enabled - defaulting offline");
        PremiumResolution disabled = PremiumResolution.offline(trimmed, RESOLVER_SERVICE, "no resolvers enabled");
        this.cacheResult(cacheKey, disabled);
        return disabled;
    }

    private PremiumResolution tryApiResolvers(String trimmed) {
        PremiumResolution offlineCandidate = null;
        boolean anyEnabled = false;
        for (PremiumResolver resolver : this.resolvers) {
            if (!resolver.enabled()) continue;
            anyEnabled = true;
            PremiumResolution result = this.processResolverResult(resolver, trimmed);
            if (result != null) {
                return result;
            }
            PremiumResolution rawResolution = resolver.resolve(trimmed);
            PremiumResolution resolution = this.normalizeResolution(resolver, rawResolution, trimmed);
            if (!resolution.isOffline() || offlineCandidate != null) continue;
            offlineCandidate = resolution;
        }
        return anyEnabled ? offlineCandidate : null;
    }

    private PremiumResolution processResolverResult(PremiumResolver resolver, String trimmed) {
        PremiumResolution rawResolution = resolver.resolve(trimmed);
        PremiumResolution resolution = this.normalizeResolution(resolver, rawResolution, trimmed);
        if (resolution.isPremium()) {
            boolean saved;
            if (resolution.uuid() != null && (saved = this.dao.saveOrUpdate(resolution.uuid(), trimmed))) {
                this.logger.debug("[PremiumResolver] zapisano do DB cache: {} -> {}", (Object)trimmed, (Object)resolution.uuid());
            }
            return resolution;
        }
        return null;
    }

    public PremiumResolution resolve(String username) {
        if (username == null || username.isBlank()) {
            return PremiumResolution.offline(username, RESOLVER_SERVICE, "empty username");
        }
        String trimmed = username.trim();
        if (!VALID_USERNAME.matcher(trimmed).matches()) {
            return PremiumResolution.offline(trimmed, RESOLVER_SERVICE, "invalid characters");
        }
        String cacheKey = trimmed.toLowerCase(Locale.ROOT);
        PremiumResolution cached = this.getFromCache(cacheKey);
        if (cached != null) {
            this.logger.debug("[PremiumResolver] memory cache hit {} -> {}", (Object)trimmed, (Object)cached.status());
            return cached;
        }
        Optional<PremiumUuid> dbResult = this.dao.findByNickname(trimmed);
        if (dbResult.isPresent()) {
            PremiumUuid premiumUuid = dbResult.get();
            PremiumResolution result = PremiumResolution.premium(premiumUuid.getUuid(), trimmed, "resolver-service-db-cache");
            this.cacheResult(cacheKey, result);
            this.logger.info("[PremiumResolver] database cache hit {} -> {} (UUID: {})", new Object[]{trimmed, result.status(), premiumUuid.getUuid()});
            return result;
        }
        return this.resolveFromApi(trimmed, cacheKey);
    }

    private PremiumResolution normalizeResolution(PremiumResolver resolver, PremiumResolution resolution, String requestName) {
        String source = resolver.id();
        if (resolution == null) {
            return PremiumResolution.unknown(source, "null resolution");
        }
        source = resolution.source() != null ? resolution.source() : source;
        String canonical = resolution.canonicalUsername() != null ? resolution.canonicalUsername() : requestName;
        switch (resolution.status()) {
            case PREMIUM: {
                if (resolution.uuid() == null) {
                    return PremiumResolution.unknown(source, "missing uuid");
                }
                if (!canonical.equalsIgnoreCase(requestName)) {
                    this.logger.debug("[PremiumResolver] username mismatch {} vs {} from {}", canonical, requestName, source);
                    return PremiumResolution.offline(requestName, source, "username mismatch with canonical name");
                }
                return PremiumResolution.premium(resolution.uuid(), canonical, source);
            }
            case OFFLINE: {
                return PremiumResolution.offline(requestName, source, resolution.message());
            }
        }
        return PremiumResolution.unknown(source, resolution.message());
    }

    private PremiumResolution getFromCache(String key) {
        CachedEntry entry = this.cache.get(key);
        if (entry == null) {
            return null;
        }
        if (entry.isExpired(this.premiumTtlMillis, this.missTtlMillis)) {
            this.cache.remove(key);
            return null;
        }
        return entry.resolution();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cacheResult(String key, PremiumResolution resolution) {
        long ttl;
        long l = ttl = resolution.isPremium() ? this.premiumTtlMillis : this.missTtlMillis;
        if (ttl <= 0L) {
            this.cache.remove(key);
            return;
        }
        this.cacheSizeLock.lock();
        try {
            if (this.cache.size() >= this.maxCacheSize) {
                int entriesToRemove = Math.max(1, this.maxCacheSize / 10);
                this.cache.entrySet().stream().sorted((e1, e2) -> Long.compare(((CachedEntry)e1.getValue()).timestamp(), ((CachedEntry)e2.getValue()).timestamp())).limit(entriesToRemove).forEach(entry -> this.cache.remove(entry.getKey()));
                this.logger.debug("[PremiumResolver] Cache eviction: removed {} entries, new size: {}", (Object)entriesToRemove, (Object)this.cache.size());
            }
            this.cache.put(key, new CachedEntry(resolution, System.currentTimeMillis()));
        }
        finally {
            this.cacheSizeLock.unlock();
        }
    }

    private record CachedEntry(PremiumResolution resolution, long timestamp) {
        boolean isExpired(long premiumTtlMillis, long missTtlMillis) {
            long ttl;
            long l = ttl = this.resolution.isPremium() ? premiumTtlMillis : missTtlMillis;
            if (ttl <= 0L) {
                return true;
            }
            return System.currentTimeMillis() - this.timestamp > ttl;
        }
    }
}

