/*
 * Decompiled with CFR 0.152.
 */
package com.candyrush.managers;

import com.candyrush.CandyRushPlugin;
import com.candyrush.models.PlayerData;
import com.candyrush.models.TeamColor;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;

public class PlayerManager {
    private final CandyRushPlugin plugin;
    private final Map<UUID, PlayerData> playerDataCache;

    public PlayerManager(CandyRushPlugin plugin) {
        this.plugin = plugin;
        this.playerDataCache = new ConcurrentHashMap<UUID, PlayerData>();
    }

    public void initialize() {
        this.plugin.getLogger().info("PlayerManager initialized");
    }

    public Optional<PlayerData> getPlayerData(UUID uuid) {
        PlayerData cached = this.playerDataCache.get(uuid);
        if (cached != null) {
            return Optional.of(cached);
        }
        try {
            Optional<PlayerData> loaded = this.plugin.getPlayerDataStorage().loadPlayer(uuid);
            loaded.ifPresent(data -> this.playerDataCache.put(uuid, (PlayerData)data));
            return loaded;
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to load player data for " + String.valueOf(uuid), e);
            return Optional.empty();
        }
    }

    public PlayerData getOrCreatePlayerData(Player player) {
        return this.getPlayerData(player.getUniqueId()).orElseGet(() -> this.createNewPlayerData(player));
    }

    private PlayerData createNewPlayerData(Player player) {
        PlayerData data = new PlayerData(player.getUniqueId(), player.getName());
        this.playerDataCache.put(player.getUniqueId(), data);
        try {
            this.plugin.getPlayerDataStorage().savePlayer(data);
            this.plugin.getLogger().info("Created new player data for: " + player.getName());
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to save new player data", e);
        }
        return data;
    }

    public void savePlayerData(PlayerData data) {
        try {
            this.plugin.getPlayerDataStorage().savePlayer(data);
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to save player data for " + String.valueOf(data.getUuid()), e);
        }
    }

    public void handlePlayerJoin(Player player) {
        PlayerData data = this.getOrCreatePlayerData(player);
        data.setName(player.getName());
        data.updateLastSeen();
        this.savePlayerData(data);
        this.plugin.getLogger().fine("Player joined: " + player.getName() + " (" + String.valueOf(player.getUniqueId()) + ")");
    }

    public void handlePlayerQuit(Player player) {
        PlayerData data = this.playerDataCache.get(player.getUniqueId());
        if (data != null) {
            data.updateLastSeen();
            this.savePlayerData(data);
        }
        if (!this.plugin.getGameManager().isGameRunning()) {
            this.playerDataCache.remove(player.getUniqueId());
        }
        this.plugin.getLogger().fine("Player quit: " + player.getName());
    }

    public void addPoints(UUID uuid, int points) {
        this.getPlayerData(uuid).ifPresent(data -> {
            data.addPoints(points);
            this.savePlayerData((PlayerData)data);
        });
    }

    public void incrementKills(UUID uuid) {
        this.getPlayerData(uuid).ifPresent(data -> {
            data.incrementKills();
            this.savePlayerData((PlayerData)data);
        });
    }

    public void incrementDeaths(UUID uuid) {
        this.getPlayerData(uuid).ifPresent(data -> {
            data.incrementDeaths();
            this.savePlayerData((PlayerData)data);
        });
    }

    public boolean setMurderer(UUID uuid, int durationSeconds) {
        boolean[] isFirstTime = new boolean[]{false};
        this.getPlayerData(uuid).ifPresent(data -> {
            long now = System.currentTimeMillis() / 1000L;
            long maxUntil = now + 3600L;
            if (!data.isMurdererActive()) {
                isFirstTime[0] = true;
                data.setMurderer(true);
                long until = now + (long)durationSeconds;
                data.setMurdererUntil(Math.min(until, maxUntil));
                this.plugin.getLogger().info("Player " + String.valueOf(uuid) + " became murderer for the first time (" + durationSeconds + " seconds)");
            } else {
                long currentUntil = data.getMurdererUntil();
                long newUntil = currentUntil + (long)durationSeconds;
                if (newUntil > maxUntil) {
                    newUntil = maxUntil;
                    this.plugin.getLogger().info("Player " + String.valueOf(uuid) + " murderer time capped at 60 minutes");
                } else {
                    long addedMinutes = durationSeconds / 60;
                    long totalMinutes = (newUntil - now) / 60L;
                    this.plugin.getLogger().info("Player " + String.valueOf(uuid) + " murderer time extended by " + addedMinutes + " minutes (total: " + totalMinutes + " minutes)");
                }
                data.setMurdererUntil(newUntil);
            }
            this.savePlayerData((PlayerData)data);
        });
        return isFirstTime[0];
    }

    public void clearMurderer(UUID uuid) {
        this.getPlayerData(uuid).ifPresent(data -> {
            data.clearMurderer();
            this.savePlayerData((PlayerData)data);
            Player player = Bukkit.getPlayer((UUID)uuid);
            if (player != null) {
                player.setDisplayName("\u00a7f" + player.getName());
                player.setPlayerListName("\u00a7f" + player.getName());
            }
        });
    }

    public void updatePlayerNameColor(Player player) {
        if (this.isMurderer(player.getUniqueId())) {
            player.setDisplayName("\u00a7c" + player.getName());
            player.setPlayerListName("\u00a7c" + player.getName());
        } else {
            player.setDisplayName("\u00a7f" + player.getName());
            player.setPlayerListName("\u00a7f" + player.getName());
        }
    }

    public boolean isMurderer(UUID uuid) {
        return this.getPlayerData(uuid).map(PlayerData::isMurdererActive).orElse(false);
    }

    public void clearAllTeamAssignments() {
        try {
            this.plugin.getPlayerDataStorage().clearAllTeamAssignments();
            for (PlayerData data : this.playerDataCache.values()) {
                data.setTeamColor(null);
            }
            this.plugin.getLogger().info("Cleared all team assignments");
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to clear team assignments", e);
        }
    }

    public void resetAllPoints() {
        try {
            this.plugin.getPlayerDataStorage().resetAllPoints();
            for (PlayerData data : this.playerDataCache.values()) {
                data.setPoints(0);
            }
            this.plugin.getLogger().info("Reset all player points");
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to reset points", e);
        }
    }

    public List<PlayerData> getTopPlayers(int limit) {
        try {
            return this.plugin.getPlayerDataStorage().getTopPlayers(limit);
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to get top players", e);
            return Collections.emptyList();
        }
    }

    public List<PlayerData> getTeamPlayers(TeamColor teamColor) {
        try {
            return this.plugin.getPlayerDataStorage().loadPlayersByTeam(teamColor);
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to get team players", e);
            return Collections.emptyList();
        }
    }

    public void saveAll() {
        int count = 0;
        for (PlayerData data : this.playerDataCache.values()) {
            try {
                this.plugin.getPlayerDataStorage().savePlayer(data);
                ++count;
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to save player data", e);
            }
        }
        this.plugin.getLogger().info("Saved " + count + " player data entries");
    }

    public void clearCache() {
        this.saveAll();
        this.playerDataCache.clear();
        this.plugin.getLogger().info("Player data cache cleared");
    }

    public void shutdown() {
        this.saveAll();
        this.playerDataCache.clear();
        this.plugin.getLogger().info("PlayerManager shutdown complete");
    }

    public int getCacheSize() {
        return this.playerDataCache.size();
    }
}

