/*
 * Decompiled with CFR 0.152.
 */
package com.candyrush.managers;

import com.candyrush.CandyRushPlugin;
import com.candyrush.models.ChestType;
import java.lang.invoke.CallSite;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.bukkit.Bukkit;
import org.bukkit.Chunk;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockState;
import org.bukkit.block.Container;
import org.bukkit.inventory.Inventory;
import org.bukkit.inventory.InventoryHolder;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitTask;

public class TreasureChestManager {
    private final CandyRushPlugin plugin;
    private final Map<Location, ChestData> activeChests;
    private final Set<Location> pendingRespawn;
    private BukkitTask respawnTask;
    private Integer currentRoundId;

    public TreasureChestManager(CandyRushPlugin plugin) {
        this.plugin = plugin;
        this.activeChests = new ConcurrentHashMap<Location, ChestData>();
        this.pendingRespawn = ConcurrentHashMap.newKeySet();
        this.currentRoundId = null;
    }

    public void initialize() {
        this.plugin.getLogger().info("TreasureChestManager initialized");
    }

    public void spawnTreasureChests(World world, Location center, int radius, Integer roundId) {
        this.currentRoundId = roundId;
        this.removeAllChests();
        if (roundId != null) {
            this.deleteOldChestsFromDatabase(roundId, world);
        }
        int chestsPerChunk = this.plugin.getConfigManager().getTreasurePerChunk();
        int centerChunkX = center.getBlockX() >> 4;
        int centerChunkZ = center.getBlockZ() >> 4;
        int chunkRadius = radius >> 4;
        this.plugin.getLogger().info("Attempting to spawn chests in radius " + chunkRadius + " chunks (" + radius + " blocks)");
        this.plugin.getLogger().info("Center chunk: " + centerChunkX + ", " + centerChunkZ);
        this.plugin.getLogger().info("Chests per chunk: " + chestsPerChunk);
        ArrayList<int[]> chunkCoordinates = new ArrayList<int[]>();
        for (int chunkX = centerChunkX - chunkRadius; chunkX <= centerChunkX + chunkRadius; ++chunkX) {
            int chunkZ = centerChunkZ - chunkRadius;
            while (chunkZ <= centerChunkZ + chunkRadius) {
                chunkCoordinates.add(new int[]{chunkX, chunkZ++});
            }
        }
        AtomicInteger totalChests = new AtomicInteger(0);
        AtomicInteger attemptedChunks = new AtomicInteger(0);
        AtomicInteger failedLocations = new AtomicInteger(0);
        this.spawnChestsAsync(world, center, radius, chestsPerChunk, chunkCoordinates, 0, totalChests, attemptedChunks, failedLocations);
        this.startRespawnTask();
    }

    private void spawnChestsAsync(World world, Location center, int radius, int chestsPerChunk, List<int[]> coordinates, int index, AtomicInteger totalChests, AtomicInteger attemptedChunks, AtomicInteger failedLocations) {
        if (index >= coordinates.size()) {
            this.plugin.getLogger().info("Spawned " + totalChests.get() + " treasure chests in the map");
            this.plugin.getLogger().info("Attempted chunks: " + attemptedChunks.get() + ", Failed locations: " + failedLocations.get());
            return;
        }
        int[] coord = coordinates.get(index);
        int chunkX = coord[0];
        int chunkZ = coord[1];
        world.getChunkAtAsync(chunkX, chunkZ).thenAccept(chunk -> {
            attemptedChunks.incrementAndGet();
            for (int i = 0; i < chestsPerChunk; ++i) {
                Location chestLoc = this.findSafeChestLocationSync((Chunk)chunk, center, radius);
                if (chestLoc != null) {
                    Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
                        this.spawnChest(chestLoc);
                        totalChests.incrementAndGet();
                    });
                    continue;
                }
                failedLocations.incrementAndGet();
            }
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.spawnChestsAsync(world, center, radius, chestsPerChunk, coordinates, index + 1, totalChests, attemptedChunks, failedLocations), 1L);
        });
    }

    private Location findSafeChestLocationSync(Chunk chunk, Location center, int radius) {
        Random random = new Random();
        int distanceRejects = 0;
        int groundRejects = 0;
        for (int attempt = 0; attempt < 10; ++attempt) {
            int x = (chunk.getX() << 4) + random.nextInt(16);
            int z = (chunk.getZ() << 4) + random.nextInt(16);
            double distance = Math.sqrt(Math.pow((double)x - center.getX(), 2.0) + Math.pow((double)z - center.getZ(), 2.0));
            if (distance > (double)radius) {
                ++distanceRejects;
                continue;
            }
            int groundY = chunk.getWorld().getHighestBlockYAt(x, z);
            int chestY = groundY + 1;
            Block ground = chunk.getWorld().getBlockAt(x, groundY, z);
            Block chest = chunk.getWorld().getBlockAt(x, chestY, z);
            if (this.isSafeGroundBlock(ground) && chest.getType() == Material.AIR) {
                return new Location(chunk.getWorld(), (double)x, (double)chestY, (double)z);
            }
            ++groundRejects;
        }
        return null;
    }

    private boolean isSafeGroundBlock(Block block) {
        Material type = block.getType();
        return type.isSolid() && type != Material.LAVA && type != Material.WATER && type != Material.CACTUS && type != Material.MAGMA_BLOCK;
    }

    private void spawnChest(Location location) {
        ChestType chestType = ChestType.random();
        Block block = location.getBlock();
        block.setType(chestType.getMaterial());
        ChestData data = new ChestData(location, chestType, System.currentTimeMillis());
        this.activeChests.put(location, data);
        if (this.currentRoundId != null) {
            this.saveChestToDatabase(location, chestType, this.currentRoundId);
        }
        if (chestType.isContainer()) {
            this.fillChestWithLootDirect(location, chestType);
        }
        this.plugin.getLogger().fine("Spawned " + String.valueOf((Object)chestType) + " at " + this.formatLocation(location));
    }

    private void fillChestWithLootDirect(Location location, ChestType chestType) {
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            Block block = location.getBlock();
            if (!chestType.getMaterial().equals((Object)block.getType())) {
                this.plugin.getLogger().warning("Block type mismatch at " + this.formatLocation(location) + " - Expected: " + String.valueOf(chestType.getMaterial()) + ", Actual: " + String.valueOf(block.getType()));
                return;
            }
            BlockState state = block.getState();
            if (!(state instanceof InventoryHolder)) {
                this.plugin.getLogger().warning("Block is not an InventoryHolder at " + this.formatLocation(location));
                return;
            }
            Inventory inv = ((InventoryHolder)state).getInventory();
            inv.clear();
            Random random = new Random();
            int itemCount = random.nextInt(4) + 2;
            ArrayList<CallSite> itemsAdded = new ArrayList<CallSite>();
            for (int i = 0; i < itemCount; ++i) {
                ItemStack food = this.getRandomFood(random);
                int slot = random.nextInt(inv.getSize());
                inv.setItem(slot, food);
                itemsAdded.add((CallSite)((Object)(food.getType().name() + "x" + food.getAmount())));
            }
            if (chestType.isTrapped()) {
                double equipmentChance = this.plugin.getConfigManager().getTrappedChestEquipmentChance();
                if (random.nextDouble() < equipmentChance) {
                    ItemStack equipment = this.getRandomEquipment(random);
                    int slot = random.nextInt(inv.getSize());
                    inv.setItem(slot, equipment);
                }
            }
            this.plugin.getLogger().info("Filled " + String.valueOf((Object)chestType) + " at " + this.formatLocation(location) + " with " + itemCount + " items: " + String.join((CharSequence)", ", itemsAdded));
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
                Block checkBlock = location.getBlock();
                if (checkBlock.getState() instanceof InventoryHolder) {
                    Inventory checkInv = ((InventoryHolder)checkBlock.getState()).getInventory();
                    int actualItems = 0;
                    for (ItemStack item : checkInv.getContents()) {
                        if (item == null || item.getType() == Material.AIR) continue;
                        ++actualItems;
                    }
                    if (actualItems == 0) {
                        this.plugin.getLogger().severe("VERIFICATION FAILED: Chest at " + this.formatLocation(location) + " is EMPTY after filling! Expected " + itemCount + " items.");
                    }
                }
            }, 5L);
        });
    }

    private void fillChestWithLoot(Location location, ChestType chestType) {
        boolean updated;
        Block block = location.getBlock();
        if (!(block.getState() instanceof Container)) {
            this.plugin.getLogger().warning("Block at " + this.formatLocation(location) + " is not a container: " + String.valueOf(block.getType()) + " (expected container for " + String.valueOf((Object)chestType) + ")");
            return;
        }
        BlockState state = block.getState();
        if (!(state instanceof Container)) {
            this.plugin.getLogger().warning("BlockState is not a container at " + this.formatLocation(location));
            return;
        }
        Container container = (Container)state;
        Inventory inv = container.getInventory();
        inv.clear();
        Random random = new Random();
        int itemCount = random.nextInt(4) + 2;
        ArrayList<ItemStack> items = new ArrayList<ItemStack>();
        for (int i = 0; i < itemCount; ++i) {
            ItemStack food = this.getRandomFood(random);
            items.add(food);
            int slot = random.nextInt(inv.getSize());
            inv.setItem(slot, food);
        }
        if (chestType.isTrapped()) {
            double equipmentChance = this.plugin.getConfigManager().getTrappedChestEquipmentChance();
            if (random.nextDouble() < equipmentChance) {
                ItemStack equipment = this.getRandomEquipment(random);
                int slot = random.nextInt(inv.getSize());
                inv.setItem(slot, equipment);
            }
        }
        if (updated = container.update(true, false)) {
            this.plugin.getLogger().info("Filled " + String.valueOf((Object)chestType) + " at " + this.formatLocation(location) + " with " + itemCount + " items: " + items.stream().map(item -> item.getType().name() + "x" + item.getAmount()).collect(Collectors.joining(", ")));
        } else {
            this.plugin.getLogger().warning("Failed to update container at " + this.formatLocation(location));
        }
        Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> {
            Block checkBlock = location.getBlock();
            if (checkBlock.getState() instanceof Container) {
                Container checkContainer = (Container)checkBlock.getState();
                int actualItems = 0;
                for (ItemStack item : checkContainer.getInventory().getContents()) {
                    if (item == null || item.getType() == Material.AIR) continue;
                    ++actualItems;
                }
                if (actualItems == 0) {
                    this.plugin.getLogger().severe("VERIFICATION FAILED: Chest at " + this.formatLocation(location) + " is EMPTY after filling! Expected " + itemCount + " items.");
                } else {
                    this.plugin.getLogger().fine("Verified: " + actualItems + " items in chest at " + this.formatLocation(location));
                }
            }
        }, 5L);
    }

    private ItemStack getRandomFood(Random random) {
        Material food;
        double candyChance = random.nextDouble();
        if (candyChance < 0.7) {
            Material[] candies = new Material[]{Material.COOKIE, Material.COOKIE, Material.COOKIE, Material.PUMPKIN_PIE, Material.CAKE, Material.SWEET_BERRIES, Material.GLOW_BERRIES, Material.GOLDEN_APPLE, Material.GOLDEN_CARROT, Material.MELON_SLICE, Material.APPLE};
            food = candies[random.nextInt(candies.length)];
        } else {
            Material[] normalFoods = new Material[]{Material.BREAD, Material.COOKED_BEEF, Material.COOKED_CHICKEN, Material.COOKED_MUTTON, Material.COOKED_PORKCHOP, Material.COOKED_SALMON, Material.BAKED_POTATO, Material.CARROT};
            food = normalFoods[random.nextInt(normalFoods.length)];
        }
        int amount = random.nextInt(8) + 1;
        return new ItemStack(food, amount);
    }

    private ItemStack getRandomEquipment(Random random) {
        Material[] equipment = new Material[]{Material.IRON_HELMET, Material.IRON_CHESTPLATE, Material.IRON_LEGGINGS, Material.IRON_BOOTS, Material.IRON_SWORD, Material.BOW, Material.SHIELD, Material.DIAMOND_HELMET, Material.DIAMOND_CHESTPLATE, Material.DIAMOND_LEGGINGS, Material.DIAMOND_BOOTS, Material.DIAMOND_SWORD};
        Material item = equipment[random.nextInt(equipment.length)];
        return new ItemStack(item, 1);
    }

    public void onChestOpened(Location location) {
        ChestData data = this.activeChests.get(location);
        if (data == null) {
            return;
        }
        this.pendingRespawn.add(location);
        this.activeChests.remove(location);
        this.plugin.getLogger().fine("Chest opened at " + this.formatLocation(location));
    }

    private void startRespawnTask() {
        if (this.respawnTask != null) {
            this.respawnTask.cancel();
        }
        int respawnDelay = this.plugin.getConfigManager().getTreasureRespawnDelay();
        this.respawnTask = Bukkit.getScheduler().runTaskTimer((Plugin)this.plugin, () -> {
            ArrayList<Location> toRespawn = new ArrayList<Location>(this.pendingRespawn);
            for (Location loc : toRespawn) {
                if (loc.getBlock().getType() != Material.AIR) {
                    this.pendingRespawn.remove(loc);
                    continue;
                }
                this.spawnChest(loc);
                this.pendingRespawn.remove(loc);
                this.plugin.getLogger().fine("Respawned chest at " + this.formatLocation(loc));
            }
        }, (long)respawnDelay * 20L, (long)respawnDelay * 20L);
    }

    private void cleanupExistingChests(World world, Location center, int radius) {
        int centerChunkX = center.getBlockX() >> 4;
        int centerChunkZ = center.getBlockZ() >> 4;
        int chunkRadius = radius >> 4;
        int removedCount = 0;
        for (int chunkX = centerChunkX - chunkRadius; chunkX <= centerChunkX + chunkRadius; ++chunkX) {
            for (int chunkZ = centerChunkZ - chunkRadius; chunkZ <= centerChunkZ + chunkRadius; ++chunkZ) {
                if (!world.isChunkLoaded(chunkX, chunkZ)) continue;
                Chunk chunk = world.getChunkAt(chunkX, chunkZ);
                for (int x = 0; x < 16; ++x) {
                    for (int z = 0; z < 16; ++z) {
                        for (int y = world.getMinHeight(); y < world.getMaxHeight(); ++y) {
                            Location blockLoc;
                            double distance;
                            Block block = chunk.getBlock(x, y, z);
                            Material type = block.getType();
                            if (!this.isChestType(type) || !((distance = (blockLoc = block.getLocation()).distance(center)) <= (double)radius)) continue;
                            block.setType(Material.AIR);
                            ++removedCount;
                        }
                    }
                }
            }
        }
        if (removedCount > 0) {
            this.plugin.getLogger().info("Cleaned up " + removedCount + " existing chest blocks in the map");
        }
    }

    private boolean isChestType(Material type) {
        return type == Material.CHEST || type == Material.TRAPPED_CHEST || type == Material.BARREL || type == Material.FURNACE || type == Material.BLAST_FURNACE || type == Material.SMOKER || type == Material.BREWING_STAND || type == Material.HOPPER || type == Material.DROPPER || type == Material.DISPENSER;
    }

    public void removeAllChests() {
        for (Location loc : this.activeChests.keySet()) {
            Block block = loc.getBlock();
            if (block.getType() == Material.AIR) continue;
            block.setType(Material.AIR);
        }
        this.activeChests.clear();
        this.pendingRespawn.clear();
        if (this.respawnTask != null) {
            this.respawnTask.cancel();
            this.respawnTask = null;
        }
        this.plugin.getLogger().info("All treasure chests removed");
    }

    private String formatLocation(Location loc) {
        return String.format("%d, %d, %d", loc.getBlockX(), loc.getBlockY(), loc.getBlockZ());
    }

    public void shutdown() {
        this.removeAllChests();
        this.plugin.getLogger().info("TreasureChestManager shutdown complete");
    }

    private void deleteOldChestsFromDatabase(int newRoundId, World world) {
        block23: {
            try (Connection conn = this.plugin.getDatabaseInitializer().getConnection();
                 PreparedStatement stmt = conn.prepareStatement("SELECT id, world, x, y, z, chest_type FROM treasure_chests WHERE round_id != ? AND world = ?");){
                stmt.setInt(1, newRoundId);
                stmt.setString(2, world.getName());
                ResultSet rs = stmt.executeQuery();
                ArrayList<Location> chestsToDelete = new ArrayList<Location>();
                ArrayList<Integer> dbIdsToDelete = new ArrayList<Integer>();
                while (rs.next()) {
                    int id = rs.getInt("id");
                    String worldName = rs.getString("world");
                    int x = rs.getInt("x");
                    int y = rs.getInt("y");
                    int z = rs.getInt("z");
                    Location loc = new Location(world, (double)x, (double)y, (double)z);
                    chestsToDelete.add(loc);
                    dbIdsToDelete.add(id);
                }
                rs.close();
                int deletedBlocks = 0;
                for (Location loc : chestsToDelete) {
                    Block block;
                    int chunkZ;
                    int chunkX = loc.getBlockX() >> 4;
                    if (!world.isChunkLoaded(chunkX, chunkZ = loc.getBlockZ() >> 4) || !this.isChestType((block = loc.getBlock()).getType())) continue;
                    block.setType(Material.AIR);
                    ++deletedBlocks;
                }
                if (dbIdsToDelete.isEmpty()) break block23;
                String placeholders = String.join((CharSequence)",", Collections.nCopies(dbIdsToDelete.size(), "?"));
                try (PreparedStatement deleteStmt = conn.prepareStatement("DELETE FROM treasure_chests WHERE id IN (" + placeholders + ")");){
                    for (int i = 0; i < dbIdsToDelete.size(); ++i) {
                        deleteStmt.setInt(i + 1, (Integer)dbIdsToDelete.get(i));
                    }
                    int deleted = deleteStmt.executeUpdate();
                    this.plugin.getLogger().info("Deleted " + deleted + " old chest records from database (" + deletedBlocks + " physical blocks removed, " + (deleted - deletedBlocks) + " in unloaded chunks)");
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to delete old chests from database", e);
            }
        }
    }

    private void saveChestToDatabase(Location location, ChestType chestType, int roundId) {
        try (Connection conn = this.plugin.getDatabaseInitializer().getConnection();
             PreparedStatement stmt = conn.prepareStatement("INSERT INTO treasure_chests (round_id, world, x, y, z, chest_type, spawned_at) VALUES (?, ?, ?, ?, ?, ?, ?)");){
            stmt.setInt(1, roundId);
            stmt.setString(2, location.getWorld().getName());
            stmt.setInt(3, location.getBlockX());
            stmt.setInt(4, location.getBlockY());
            stmt.setInt(5, location.getBlockZ());
            stmt.setString(6, chestType.name());
            stmt.setLong(7, System.currentTimeMillis() / 1000L);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to save chest to database at " + this.formatLocation(location), e);
        }
    }

    public void cleanupChestOnChunkLoad(World world, int chunkX, int chunkZ) {
        block24: {
            if (this.currentRoundId == null) {
                return;
            }
            try (Connection conn = this.plugin.getDatabaseInitializer().getConnection();
                 PreparedStatement stmt = conn.prepareStatement("SELECT id, x, y, z FROM treasure_chests WHERE round_id != ? AND world = ? AND x >= ? AND x < ? AND z >= ? AND z < ?");){
                int minX = chunkX << 4;
                int maxX = minX + 16;
                int minZ = chunkZ << 4;
                int maxZ = minZ + 16;
                stmt.setInt(1, this.currentRoundId);
                stmt.setString(2, world.getName());
                stmt.setInt(3, minX);
                stmt.setInt(4, maxX);
                stmt.setInt(5, minZ);
                stmt.setInt(6, maxZ);
                ResultSet rs = stmt.executeQuery();
                ArrayList<Integer> idsToDelete = new ArrayList<Integer>();
                while (rs.next()) {
                    int z;
                    int y;
                    int id = rs.getInt("id");
                    int x = rs.getInt("x");
                    Location loc = new Location(world, (double)x, (double)(y = rs.getInt("y")), (double)(z = rs.getInt("z")));
                    Block block = loc.getBlock();
                    if (this.isChestType(block.getType())) {
                        block.setType(Material.AIR);
                        this.plugin.getLogger().fine("Cleaned up old chest at " + this.formatLocation(loc) + " on chunk load");
                    }
                    idsToDelete.add(id);
                }
                rs.close();
                if (idsToDelete.isEmpty()) break block24;
                String placeholders = String.join((CharSequence)",", Collections.nCopies(idsToDelete.size(), "?"));
                try (PreparedStatement deleteStmt = conn.prepareStatement("DELETE FROM treasure_chests WHERE id IN (" + placeholders + ")");){
                    for (int i = 0; i < idsToDelete.size(); ++i) {
                        deleteStmt.setInt(i + 1, (Integer)idsToDelete.get(i));
                    }
                    deleteStmt.executeUpdate();
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().log(Level.WARNING, "Failed to cleanup chests on chunk load", e);
            }
        }
    }

    private static class ChestData {
        private final Location location;
        private final ChestType type;
        private final long spawnTime;

        public ChestData(Location location, ChestType type, long spawnTime) {
            this.location = location;
            this.type = type;
            this.spawnTime = spawnTime;
        }

        public Location getLocation() {
            return this.location;
        }

        public ChestType getType() {
            return this.type;
        }

        public long getSpawnTime() {
            return this.spawnTime;
        }
    }
}

