package com.mythicmetals.block;

import com.google.common.collect.*;
import com.mythicmetals.MythicMetals;
import com.mythicmetals.misc.RegistryHelper;
import io.wispforest.owo.util.Maldenhagen;
import io.wispforest.owo.util.TagInjector;
import net.minecraft.block.*;
import net.minecraft.class_2199;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2431;
import net.minecraft.class_2498;
import net.minecraft.class_2960;
import net.minecraft.class_4970;
import net.minecraft.class_4970.class_2251;
import net.minecraft.class_5541;
import net.minecraft.class_6016;
import net.minecraft.class_6019;
import net.minecraft.class_7923;
import java.util.*;
import java.util.function.Consumer;

/**
 * This class is a container which is used for the creation for all the blocks in Mythic Metals.
 * For creating blocks using this you want to start with looking at the {@link Builder}, which
 * contains all the methods for creating blocks.
 *
 * @author glisco
 * @author Noaaan
 */
@SuppressWarnings({"unused"})
public class BlockSet {
    private final class_2431 ore;
    private final class_2248 storageBlock;
    private final class_2248 oreStorageBlock;
    private final class_2199 anvil;

    private final String name;
    private final boolean fireproof;

    private final Multimap<class_2248, class_2960> miningLevels;
    private final Multimap<class_2199, class_2960> anvilMap;
    private final Map<String, class_2431> oreVariants;
    private final boolean uncommon;

    /**
     * This constructor collects the smaller constructors from the {@link Builder} and creates a set of blocks.
     * Use {@link Builder#begin(String, boolean) BlockSet.Builder.begin} to begin,
     * and call {@link Builder#finish()} when you are done.
     *
     * @param name            Common name for the entire set of blocks, applies to every block created.
     * @param ore             Contains a vanilla {@link class_2431}.
     * @param storageBlock    Contains a {@link class_2248} which is used as a storage block.
     * @param oreStorageBlock Contains a {@link class_2248} which is used as a ore storage block.
     * @param anvil           Contains an {@link class_2199}
     * @param oreVariants     A map of a string and {@link class_2431} which is used for variant ores.
     * @param fireproof       Boolean for creating fireproof block sets.
     * @param miningLevels    A map containing all the blocks being registered with their corresponding mining levels.
     * @param anvilMap        A map containing all anvils and their levels, so that they can be disabled.
     * @param uncommon        Boolean for setting the block item to Uncommon Rarity, changing the color of the text
     */
    private BlockSet(String name,
                     class_2431 ore,
                     class_2248 storageBlock,
                     class_2248 oreStorageBlock,
                     class_2199 anvil,
                     Map<String, class_2431> oreVariants,
                     boolean fireproof,
                     Multimap<class_2248, class_2960> miningLevels,
                     Multimap<class_2199, class_2960> anvilMap, boolean uncommon) {

        this.name = name;
        this.fireproof = fireproof;

        this.ore = ore;
        this.storageBlock = storageBlock;
        this.oreStorageBlock = oreStorageBlock;
        this.anvil = anvil;

        this.oreVariants = oreVariants;
        this.miningLevels = miningLevels;
        this.anvilMap = anvilMap;
        this.uncommon = uncommon;
    }

    private void register() {

        if (ore != null) {
            RegistryHelper.block(name + "_ore", ore, fireproof, uncommon);
        }

        oreVariants.forEach((s, block) -> RegistryHelper.block(s + "_" + name + "_ore", block, fireproof, uncommon));

        if (oreStorageBlock != null) {
            RegistryHelper.block("raw_" + name + "_block", oreStorageBlock, fireproof, uncommon);
        }
        if (storageBlock != null) {
            RegistryHelper.block(name + "_block", storageBlock, fireproof, uncommon);
        }
        if (anvil != null) {
            RegistryHelper.block(name + "_anvil", anvil, fireproof, uncommon);
        }
        // Inject all the mining levels into their tags.
        if (MythicMetals.CONFIG.enableAnvils()) {
            anvilMap.forEach(((anvilBlock, level) -> {
                TagInjector.inject(class_7923.field_41175, RegistryHelper.id("anvils"), anvilBlock);
                TagInjector.inject(class_7923.field_41175, level, anvilBlock);
                TagInjector.inject(class_7923.field_41175, class_2960.method_60654("anvil"), anvilBlock);
                TagInjector.inject(class_7923.field_41178, class_2960.method_60654("anvil"), anvilBlock.method_8389());
            }));
        }
        miningLevels.forEach((block, level) -> {
            TagInjector.inject(class_7923.field_41175, level, block);
            TagInjector.inject(class_7923.field_41175, RegistryHelper.id("blocks"), block);
        });
    }

    /**
     * @return Returns the ore block in the set
     */
    public class_2431 getOre() {
        return ore;
    }

    /**
     * @return Returns the storage block from the set
     */
    public class_2248 getStorageBlock() {
        return storageBlock;
    }

    /**
     * @return Returns the ore storage block from the set
     */
    public class_2248 getOreStorageBlock() {
        return oreStorageBlock;
    }

    /**
     * @param variant The string of the ore variants name
     * @return Returns the specified ore variant from the variant map in the blockset
     */
    public class_2431 getOreVariant(String variant) {
        return oreVariants.get(variant);
    }

    /**
     * @return Returns the anvil from the set
     */
    public class_2199 getAnvil() {
        return anvil;
    }

    public Set<class_2248> getOreVariants() {
        return ImmutableSet.copyOf(oreVariants.values());
    }

    public Map<String, class_2248> getOreVariantsMap() {
        return Map.copyOf(oreVariants);
    }

    public String getName() {
        return this.name;
    }

    /**
     * This is the BlockSet Builder, which is used for constructing new sets of blocks.
     * <p>
     * To begin creating BlockSets you want to call:
     * {@code public static final BlockSet SETNAME = }{@link Builder#begin(String, boolean) BlockSet.Builder.begin()}
     * where you provide a {@code string} for the name/key, and the {@code fireproof} boolean.
     * <p>
     * When creating blocks it's important to call {@link #strength(float)} before creating a block or any set.
     * This is because the values are grabbed from this method. You can call it multiple times if you wish to
     * specifically tailor the values for individual blocks.
     * <p>
     * When you are finished with adding your blocks to the set,
     * call {@link Builder#finish() Builder.finish} when you are done.
     * If you need any examples on how to apply this builder in practice, see {@link MythicBlocks}.
     *
     * @author glisco
     * @author Noaaan
     * @see Builder#begin(String, boolean)
     * @see MythicBlocks
     */
    public static class Builder {

        private static final List<BlockSet> toBeRegistered = new ArrayList<>();

        private final String name;
        private final boolean fireproof;
        private final Map<String, class_2431> oreVariants = new LinkedHashMap<>();
        private class_2431 ore = null;
        private class_2248 storageBlock = null;
        private class_2248 oreStorageBlock = null;
        private class_2199 anvil = null;
        private class_2498 currentSounds = class_2498.field_11544;
        private float currentHardness = -1;
        private float currentResistance = -1;
        private final Multimap<class_2248, class_2960> miningLevels = HashMultimap.create();
        private final Multimap<class_2199, class_2960> anvilMap = HashMultimap.create();
        private final Consumer<class_4970.class_2251> settingsProcessor = settings -> {
        };

        private final class_2960 SHOVEL = class_2960.method_60654("mineable/shovel");
        private final class_2960 PICKAXE = class_2960.method_60654("mineable/pickaxe");
        private boolean uncommon = false;

        /**
         * @see #begin(String, boolean)
         */
        private Builder(String name, boolean fireproof) {
            this.name = name;
            this.fireproof = fireproof;
        }

        /**
         * This method begins the creation of a block set.
         * You can add as many blocks as you want in the set
         * Call {@link Builder#finish()} when you are done.
         *
         * @param name      The name of the new block set
         * @param fireproof Boolean of whether the entire set should be fireproof
         */
        public static Builder begin(String name, boolean fireproof) {
            return new Builder(name, fireproof);
        }

        public static void register() {
            toBeRegistered.forEach(blockSet -> {
                MythicBlocks.BLOCKSET_MAP.put(blockSet.name, blockSet);
                blockSet.register();
            });
            toBeRegistered.clear();
        }

        /**
         * Used internally for configuring blocks
         *
         * @param hardness   Determines the breaking time of the block.
         * @param resistance Determines blast resistance of a block.
         * @param sounds     Determines the sounds that blocks play when interacted with.
         */
        public static class_4970.class_2251 blockSettings(float hardness, float resistance, class_2498 sounds) {
            return class_4970.class_2251.method_9637()
                .method_9629(hardness, resistance)
                .method_9626(sounds)
                .method_51369()
                .method_29292();
        }

        /**
         * Puts an ore, a storage block, an ore storage block, and an anvil in the blockset.
         *
         * @param strength    Sets the strength of the blocks in the set.
         * @param miningLevel Mining level of the blocks. The ore sets the raw value,
         *                    while every other block recieves + 1 to their level.
         * @see #strength(float)    Strength
         */
        public Builder createDefaultSet(float strength, class_2960 miningLevel, class_2960 higherMiningLevel) {
            return strength(strength)
                .createOre(miningLevel)
                .strength(strength + 1.0F)
                .createOreStorageBlock(miningLevel)
                .createStorageBlock(higherMiningLevel)
                .createAnvil(miningLevel);
        }

        /**
         * Puts an ore, a storage block, and an ore storage block in the blockset.
         *
         * @param strength           Sets the strength of the blocks in the set.
         * @param miningLevel        The mining level of the ore block
         * @param storageMiningLevel The mining level of both storage blocks
         * @see #strength(float)
         */
        public Builder createBlockSet(float strength, class_2960 miningLevel, class_2960 storageMiningLevel) {
            return strength(strength)
                .createOre(miningLevel)
                .strength(strength + 1.0F)
                .createStorageBlock(storageMiningLevel)
                .createOreStorageBlock(storageMiningLevel);
        }

        /**
         * Puts an ore, a storage block and an ore storage block in the set, with slightly more configurable settings.
         *
         * @param oreStrength        The strength of the ore block.
         * @param oreMiningLevel     The mining level of the ore block.
         * @param storageStrength    The strength of the storage block and ore storage block.
         * @param storageMiningLevel The mining level of the storage block and ore storage block.
         * @see #strength(float)        oreStrength and storageStrength
         */
        public Builder createDefaultSet(float oreStrength, class_2960 oreMiningLevel, float storageStrength, class_2960 storageMiningLevel) {
            return strength(oreStrength)
                .createOre(oreMiningLevel)
                .strength(storageStrength)
                .createStorageBlock(storageMiningLevel)
                .createOreStorageBlock(storageMiningLevel);
        }

        /**
         * Puts a storage block and an anvil in the blockset.
         *
         * @param miningLevel The mining level of the anvil and the storage block
         * @see #strength(float)
         */
        public Builder createAnvilSet(float strength, class_2960 miningLevel) {
            return strength(strength)
                .sounds(class_2498.field_11533)
                .createStorageBlock(miningLevel)
                .createAnvil(miningLevel);
        }

        /**
         * Puts a storage block and an anvil in the blockset, where the storage block is configurable.
         *
         * @param hardness    The hardness of the storage block.
         * @param resistance  The blast resistance of the storage block.
         * @param miningLevel The mining level of the anvil and the storage block.
         * @see #createAnvil(class_2960)  createAnvil
         */
        public Builder createAnvilSet(float hardness, float resistance, class_2960 miningLevel) {
            return strength(hardness, resistance)
                .createStorageBlock(this.currentSounds, miningLevel)
                .createAnvil(miningLevel);
        }

        /**
         * Applies sounds to the block(s) in the set.
         *
         * @param sounds The {@link class_2498} which should be played.
         */
        public Builder sounds(class_2498 sounds) {
            this.currentSounds = sounds;
            return this;
        }

        /**
         * A simplified method to create a hardness and resistance value from a single int.
         *
         * @param strength The base int value for the blocks' strength.
         * @return hardness, resistance (strength + 1)
         */
        public Builder strength(float strength) {
            return strength(strength, strength + 1);
        }

        /**
         * Gives the block(s) in the set the specified strength.
         *
         * @param hardness   Hardness of the block, determines breaking speed.
         * @param resistance Blast resistance of the block.
         */
        public Builder strength(float hardness, float resistance) {
            this.currentHardness = hardness;
            this.currentResistance = resistance;
            return this;
        }

        /**
         * Creates an ore block.
         *
         * @param miningLevel The mining level of the ore block.
         * @see Builder
         */
        public Builder createOre(class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.ore = new class_2431(class_6016.field_29942, settings);
            miningLevels.put(ore, miningLevel);
            miningLevels.put(ore, PICKAXE);
            return this;
        }

        /**
         * Creates an ore block, which drops experience.
         *
         * @param miningLevel The mining level of the ore block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         * @see Builder
         */
        public Builder createOre(class_2960 miningLevel, class_6019 experience) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.ore = new class_2431(experience, settings);
            miningLevels.put(ore, miningLevel);
            miningLevels.put(ore, PICKAXE);
            return this;
        }

        /**
         * Creates an ore block, which drops experience.
         *
         * @param miningLevel The mining level of the ore block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         * @see Builder
         */
        public Builder createLuminantOre(class_2960 miningLevel, class_6019 experience, int luminance) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds).method_9631(blockState -> luminance);
            settingsProcessor.accept(settings);
            this.ore = new class_2431(class_6016.field_29942, settings);
            miningLevels.put(ore, miningLevel);
            miningLevels.put(ore, PICKAXE);
            Maldenhagen.injectCopium(this.ore);
            return this;
        }

        /**
         * Creates an ore variant.
         *
         * @param name        The name/key for the variant.
         * @param miningLevel The mining level of the ore variant.
         * @see Builder
         */
        public Builder createOreVariant(String name, class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.oreVariants.put(name, new class_2431(class_6016.field_29942, settings));
            miningLevels.put(oreVariants.get(name), miningLevel);
            miningLevels.put(oreVariants.get(name), PICKAXE);
            return this;
        }

        /**
         * Creates an ore variant, which drops experience.
         *
         * @param name        The name/key for the variant.
         * @param miningLevel The mining level of the variant ore block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         */
        public Builder createOreVariant(String name, class_2960 miningLevel, class_6019 experience) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.oreVariants.put(name, new class_2431(experience, settings));
            miningLevels.put(oreVariants.get(name), miningLevel);
            miningLevels.put(oreVariants.get(name), PICKAXE);
            return this;
        }

        /**
         * Creates an ore variant, which drops experience.
         *
         * @param name        The name/key for the variant.
         * @param miningLevel The mining level of the variant ore block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         */
        public Builder createOreVariant(String name, class_2960 miningLevel, class_6019 experience, int luminance) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds).method_9631(blockState -> luminance);
            settingsProcessor.accept(settings);
            this.oreVariants.put(name, new class_2431(experience, settings));
            miningLevels.put(oreVariants.get(name), miningLevel);
            miningLevels.put(oreVariants.get(name), PICKAXE);
            Maldenhagen.injectCopium(this.oreVariants.get(name));
            return this;
        }

        /**
         * A special ore creator for the creation of a {@link StarriteOreBlock}.
         *
         * @param miningLevel The mining level of the block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         */
        public Builder createStarriteOre(class_2960 miningLevel, class_6019 experience) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.ore = new StarriteOreBlock(settings, experience);
            miningLevels.put(ore, miningLevel);
            miningLevels.put(ore, PICKAXE);
            return this;
        }

        /**
         * A special ore creator for the creation of a {@link BanglumOreBlock}.
         *
         * @param miningLevel The mining level of the block.
         */
        public Builder createBanglumOre(class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.ore = new BanglumOreBlock(settings);
            miningLevels.put(ore, miningLevel);
            miningLevels.put(ore, PICKAXE);
            return this;
        }

        /**
         * A special method for the creation of variants from {@link StarriteOreBlock}.
         *
         * @param name        The name/key for the variant.
         * @param miningLevel The mining level of the block.
         * @param experience  An {@link class_6019}, which holds the range of xp that can drop.
         */
        public Builder createStarriteOreVariant(String name, class_2960 miningLevel, class_6019 experience) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.oreVariants.put(name, new StarriteOreBlock(settings, experience));
            miningLevels.put(oreVariants.get(name), miningLevel);
            miningLevels.put(oreVariants.get(name), PICKAXE);
            return this;
        }

        /**
         * A special method for the creation of variants from {@link BanglumOreBlock}.
         *
         * @param name        The name/key for the variant.
         * @param miningLevel The mining level of the block.
         */
        public Builder createBanglumOreVariant(String name, class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.oreVariants.put(name, new BanglumOreBlock(settings));
            miningLevels.put(oreVariants.get(name), miningLevel);
            miningLevels.put(oreVariants.get(name), PICKAXE);
            return this;
        }

        /**
         * A special method for the creation of storage blocks that copies Amethyst Blocks.
         *
         * @param miningLevel The mining level of the block.
         * @see class_2246#field_27159
         * @see class_5541
         */
        public Builder createAmethystStorageBlock(class_2960 miningLevel) {
            this.storageBlock = new class_5541(blockSettings(currentHardness, currentResistance, class_2498.field_27197));
            miningLevels.put(storageBlock, miningLevel);
            miningLevels.put(storageBlock, PICKAXE);
            return this;
        }

        /**
         * Create a storage block, with a specific material in mind.
         *
         * @param miningLevel The mining level of the storage block.
         */
        public Builder createStorageBlock(class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.storageBlock = new class_2248(settings);
            miningLevels.put(storageBlock, miningLevel);
            miningLevels.put(storageBlock, PICKAXE);
            return this;
        }

        /**
         * Create a storage block, with a specific sound in mind.
         *
         * @param sounds      A {@link class_2498}, which determines block sounds.
         * @param miningLevel The mining level of the storage block.
         */
        public Builder createStorageBlock(class_2498 sounds, class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, sounds);
            settingsProcessor.accept(settings);
            this.storageBlock = new class_2248(settings);
            miningLevels.put(storageBlock, miningLevel);
            miningLevels.put(storageBlock, PICKAXE);
            return this;
        }

        /**
         * Create a raw ore storage block.
         *
         * @param miningLevel The mining level of the raw storage block.
         */
        public Builder createOreStorageBlock(class_2960 miningLevel) {
            final var settings = blockSettings(currentHardness, currentResistance, currentSounds);
            settingsProcessor.accept(settings);
            this.oreStorageBlock = new class_2248(settings);
            miningLevels.put(oreStorageBlock, miningLevel);
            miningLevels.put(oreStorageBlock, PICKAXE);
            return this;
        }

        /**
         * Creates an anvil for a blockset.
         * Only requires a mining level, since hardness and resistance match vanilla values.
         *
         * @param miningLevel Mining level of the anvil.
         */
        public Builder createAnvil(class_2960 miningLevel) {
            if (MythicMetals.CONFIG.enableAnvils()) {
                final var settings = blockSettings(5.0f, 15000f, class_2498.field_11531);
                settingsProcessor.accept(settings);
                this.anvil = new class_2199(settings);
                anvilMap.put(anvil, miningLevel);
                anvilMap.put(anvil, PICKAXE);
            }
            return this;
        }

        /**
         * Kinda manual at this point ngl
         */
        public <T extends class_2248> Builder createCustomStorageBlock(T block, class_2960 miningLevel) {
            this.storageBlock = block;
            miningLevels.put(storageBlock, miningLevel);
            miningLevels.put(storageBlock, PICKAXE);
            return this;
        }

        public Builder createCustomStorageBlock(class_2960 miningLevel, class_4970.class_2251 settings) {
            settingsProcessor.accept(settings);
            this.storageBlock = new class_2248(settings);
            miningLevels.put(storageBlock, miningLevel);
            miningLevels.put(storageBlock, PICKAXE);
            return this;
        }

        public Builder uncommon() {
            this.uncommon = true;
            return this;
        }

        /**
         * Finishes the creation of the block set, and returns the entire set using the settings declared.
         * For registering the blocks call {@link Builder#register() Builder.register} during mod initialization.
         *
         * @return BlockSet
         */
        public BlockSet finish() {
            final var set = new BlockSet(this.name, this.ore,
                this.storageBlock, this.oreStorageBlock, this.anvil,
                this.oreVariants, this.fireproof, this.miningLevels, this.anvilMap, this.uncommon);
            Builder.toBeRegistered.add(set);
            return set;
        }
    }
}
