package com.mythicmetals.command;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.RequiredArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.suggestion.Suggestions;
import com.mojang.brigadier.suggestion.SuggestionsBuilder;
import com.mojang.brigadier.tree.ArgumentCommandNode;
import com.mojang.brigadier.tree.LiteralCommandNode;
import com.mythicmetals.MythicMetals;
import com.mythicmetals.armor.ArmorSet;
import com.mythicmetals.armor.MythicArmor;
import com.mythicmetals.block.BlockSet;
import com.mythicmetals.block.MythicBlocks;
import com.mythicmetals.config.MythicOreConfigs;
import com.mythicmetals.config.OreConfig;
import com.mythicmetals.item.tools.MythicTools;
import com.mythicmetals.item.tools.ToolSet;
import com.mythicmetals.misc.RegistryHelper;
import com.mythicmetals.misc.StringUtilsAtHome;
import io.wispforest.owo.util.ReflectionUtils;
import net.fabricmc.fabric.api.command.v2.ArgumentTypeRegistry;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1299;
import net.minecraft.class_1531;
import net.minecraft.class_173;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_181;
import net.minecraft.class_1937;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2319;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2561;
import net.minecraft.class_3039;
import net.minecraft.class_3218;
import net.minecraft.class_3489;
import net.minecraft.class_6880;
import net.minecraft.class_7924;
import net.minecraft.class_8053;
import net.minecraft.class_8056;
import net.minecraft.class_8113;
import net.minecraft.class_8113.class_8122;
import net.minecraft.class_8567;
import net.minecraft.class_9334;
import net.minecraft.class_9433;
import net.minecraft.loot.context.*;
import net.minecraft.server.command.*;
import org.apache.commons.lang3.mutable.MutableInt;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.io.IOException;
import java.nio.file.*;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

@SuppressWarnings({"UnstableApiUsage", "CodeBlock2Expr"})
public final class MythicCommands {

    public static BiMap<String, OreConfig> ORE_CONFIG = HashBiMap.create();

    private MythicCommands() {
    }

    @SuppressWarnings("UnreachableCode")
    public static void init() {
        ReflectionUtils.iterateAccessibleStaticFields(MythicOreConfigs.class, OreConfig.class, (value, name, field) -> {
            ORE_CONFIG.put(name, value);
        });
        ArgumentTypeRegistry.registerArgumentType(RegistryHelper.id("toolset"), ToolSetArgumentType.class, class_2319.method_41999(ToolSetArgumentType::toolSet));
        ArgumentTypeRegistry.registerArgumentType(RegistryHelper.id("armorset"), ArmorSetArgumentType.class, class_2319.method_41999(ArmorSetArgumentType::armorSet));
        ArgumentTypeRegistry.registerArgumentType(RegistryHelper.id("ore-config"), OreConfigArgumentType.class, class_2319.method_41999(OreConfigArgumentType::oreConfig));
        ArgumentTypeRegistry.registerArgumentType(RegistryHelper.id("blockset"), BlockSetArgumentType.class, class_2319.method_41999(BlockSetArgumentType::blockSet));
    }

    // TODO - Add new command for grabbing the data-generated ore features, and create a datapack skeleton
    public static void registerCommands() {
        CommandRegistrationCallback.EVENT.register((dispatcher, access, env) -> {
            var mythicRoot = class_2170.method_9247("mythicmetals").requires(src -> src.method_9259(2)).build();
            var range = class_2170.method_9247("range").build();
            var tools = class_2170.method_9247("tools").build();
            var allTools = class_2170.method_9247("tools-all").executes(MythicCommands::exportAllTools).build();
            var allArmor = class_2170.method_9247("armor-all").executes(MythicCommands::exportAllArmor).build();
            var ores = class_2170.method_9247("ores").build();
            var armor = class_2170.method_9247("armor").build();
            var wiki = class_2170.method_9247("wiki").build();
            var armorStand = class_2170.method_9247("armor-stand").build();
            var loot = class_2170.method_9247("test-loot-table").build();
            var display = class_2170.method_9247("place-display").build();
            var placeBlocks = class_2170.method_9247("place-all-blocks").executes(context -> placeAllBlocksets(context, Map.of()))
                .build();

            // TODO - Make this useful command more useful for current use-cases:
            // TODO -- Allow dumping output in a spreadsheet friendly format
            var rangeType = class_2170.method_9244("type", StringArgumentType.word())
                .suggests(MythicCommands::dumpType)
                .executes(MythicCommands::dumpAllOreConfigs)
                .build();

            var exportOres = class_2170.method_9244("ore-config", OreConfigArgumentType.oreConfig())
                .executes(MythicCommands::exportOreData)
                .build();

            var exportTools = class_2170.method_9244("toolset", ToolSetArgumentType.toolSet())
                .executes(MythicCommands::exportTools)
                .build();

            var exportArmor = class_2170.method_9244("armorset", ArmorSetArgumentType.armorSet())
                .executes(MythicCommands::exportArmor)
                .build();

            var placeDisplay = class_2170.method_9244("material", StringArgumentType.word())
                .suggests(MythicCommands::material)
                .executes(MythicCommands::placeMythicDisplay)
                .build();

            var lootTables = class_2170.method_9244("loot_table", class_9433.class_9436.method_58482(access))
                .suggests(class_3039.field_13605)
                .then(class_2170.method_9244("rolls", IntegerArgumentType.integer())
                    .executes(MythicCommands::testLootTable))
                .build();

            var trimPattern = class_2170.method_9244("trim_pattern", StringArgumentType.word())
                .suggests(MythicCommands::trimTypes)
                .executes(context -> {
                    String matQuery = StringArgumentType.getString(context, "material");
                    String trimQuery = StringArgumentType.getString(context, "trim_pattern");
                    return armorStandCommand(context, matQuery, trimQuery);
                });

            var summonTrims = class_2170.method_9244("material", StringArgumentType.word())
                .suggests(MythicCommands::armorMaterial)
                .executes(context -> {
                    String mat = StringArgumentType.getString(context, "material");
                    return armorStandCommand(context, mat, null);
                })
                .then(trimPattern)
                .build();

            // Wiki nodes
            ores.addChild(exportOres);
            tools.addChild(exportTools);
            armor.addChild(exportArmor);
            wiki.addChild(ores);
            wiki.addChild(tools);
            wiki.addChild(allTools);
            wiki.addChild(armor);
            wiki.addChild(allArmor);

            // Misc nodes
            range.addChild(rangeType);
            loot.addChild(lootTables);
            armorStand.addChild(summonTrims);
            display.addChild(placeDisplay);

            // Add commands to root
            mythicRoot.addChild(range);
            mythicRoot.addChild(wiki);
            mythicRoot.addChild(armorStand);
            mythicRoot.addChild(loot);
            mythicRoot.addChild(placeBlocks);
            mythicRoot.addChild(display);

            dispatcher.getRoot().addChild(mythicRoot);
        });
    }

    private static int exportAllArmor(CommandContext<class_2168> context) {
        var folder = Path.of(FabricLoader.getInstance().getConfigDir() + "/mythicmetals");
        try {
            Files.createDirectory(folder);
        } catch (FileAlreadyExistsException ignored) {
            MythicMetals.LOGGER.debug("Folder already exists");
        } catch (IOException e) {
            MythicMetals.LOGGER.error("Failed to create folder", e);
        }
        MythicArmor.ARMOR_MAP.forEach((name, armorSet) -> {
            var file = Path.of(FabricLoader.getInstance().getConfigDir() + "/mythicmetals/" + name.toLowerCase(Locale.ROOT) + ".md");
            try {
                Files.createFile(file);
            } catch (FileAlreadyExistsException ignored) {
                // no-op
            }
            catch (IOException e) {
                MythicMetals.LOGGER.error("Failed to write wiki data");
                context.getSource().method_9226(() -> class_2561.method_43470("Failed to %s wiki data to disk!".formatted(name)), false);
                return;
            }
            try {
                Files.writeString(file, WikiExporter.computeArmorSet(armorSet));
                var logString = "Successfully exported equipment/%s-tools".formatted(name.toLowerCase(Locale.ROOT));
                MythicMetals.LOGGER.info(logString);
            } catch (IOException e) {
                MythicMetals.LOGGER.error("Failed to write wiki data");
                context.getSource().method_9226(() -> class_2561.method_43470("Failed to %s wiki data to disk!".formatted(name)), false);
            }
        });

        context.getSource().method_9226(() -> class_2561.method_43470("Exported all the armor to wiki format into the config folder"), false);
        return 0;
    }

    // TODO - Definitely the most lazy approach.
    //  At least make it overwrite the files instead of forcing you to delete the folder every time
    private static int exportAllTools(CommandContext<class_2168> context) {
        var folder = Path.of(FabricLoader.getInstance().getConfigDir() + "/mythicmetals");
        try {
            Files.createDirectory(folder);
        } catch (FileAlreadyExistsException ignored) {
            MythicMetals.LOGGER.debug("Folder already exists");
        } catch (IOException e) {
            MythicMetals.LOGGER.error("Failed to create folder", e);
        }
        ReflectionUtils.iterateAccessibleStaticFields(MythicTools.class, ToolSet.class, (value, name, field) -> {
            var file = Path.of(FabricLoader.getInstance().getConfigDir() + "/mythicmetals/" + name.toLowerCase(Locale.ROOT) + "-tools.md");
            try {
                Files.createFile(file);
            } catch (FileAlreadyExistsException ignored) {
                // no-op
            } catch (IOException e) {
                MythicMetals.LOGGER.error("Failed to write wiki data");
                context.getSource().method_9226(() -> class_2561.method_43470("Failed to %s wiki data to disk!".formatted(name)), false);
                return;
            }
            try {
                Files.writeString(file, WikiExporter.computeToolset(value));
                var logString = "Successfully exported equipment/%s-tools".formatted(name.toLowerCase(Locale.ROOT));
                MythicMetals.LOGGER.info(logString);
            } catch (IOException e) {
                MythicMetals.LOGGER.error("Failed to write wiki data");
                context.getSource().method_9226(() -> class_2561.method_43470("Failed to %s wiki data to disk!".formatted(name)), false);
            }
        });

        context.getSource().method_9226(() -> class_2561.method_43470("Exported all the tools (in the shed) to wiki format into the config folder"), false);
        return 0;
    }

    /**
     * Place every block set from {@link MythicBlocks} across the YZ axis
     *
     * @param context     ServerCommandSource Command Context
     * @param extraBlocks Map which can be used to insert extra blocks for a specific block set
     */
    public static int placeAllBlocksets(CommandContext<class_2168> context, Map<String, ArrayList<class_2248>> extraBlocks) {
        var source = context.getSource();
        var world = source.method_9225();
        AtomicInteger x = new AtomicInteger(((int) source.method_9222().field_1352));
        AtomicInteger y = new AtomicInteger(((int) source.method_9222().field_1351));
        int z = ((int) source.method_9222().field_1350);
        ReflectionUtils.iterateAccessibleStaticFields(MythicBlocks.class, BlockSet.class, (blockSet, name, field) -> {
            y.set(((int) source.method_9222().field_1351));
            if (blockSet.getOre() != null) {
                world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), blockSet.getOre().method_9564());
            }
            blockSet.getOreVariants().forEach(block -> {
                world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), block.method_9564());
            });
            if (blockSet.getOreStorageBlock() != null) {
                world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), blockSet.getOreStorageBlock().method_9564());
            }
            if (blockSet.getStorageBlock() != null) {
                world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), blockSet.getStorageBlock().method_9564());
            }
            if (blockSet.getAnvil() != null) {
                world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), blockSet.getAnvil().method_9564());
            }
            if (extraBlocks.containsKey(name)) {
                extraBlocks.get(name).forEach(extraBlock -> {
                    world.method_8501(class_2338.method_49637(x.get(), y.getAndIncrement(), z), extraBlock.method_9564());
                });
            }
            x.incrementAndGet();
        });
        source.method_9226(() -> class_2561.method_43470("Placed all blocksets starting at %s,%s,%s".formatted(source.method_9222().field_1352, source.method_9222().field_1351, source.method_9222().field_1350)), true);
        return 0;
    }

    public static int placeMythicDisplay(CommandContext<class_2168> context) {
        int placements = -1;
        var material = StringArgumentType.getString(context, "material");

        if (!MythicTools.TOOL_MAP.containsKey(material) && !MythicArmor.ARMOR_MAP.containsKey(material) && MythicBlocks.BLOCKSET_MAP.containsKey(material)) {
            MythicMetals.LOGGER.error("Failed to find material: {}", material);
            context.getSource().method_9226(() -> class_2561.method_43470("Could not find any items for the material %s".formatted(material)), false);
            return -1;
        }

        // place the base structure
        var world = context.getSource().method_9225();
        var startPos = context.getSource().method_9228().method_24515();

        placeStructure(world, startPos);

        if (material.equals("all")) {
            // oh dear god
        }

        if (MythicTools.TOOL_MAP.containsKey(material)) {
            // TODO - I like item frames more, unfortunately...
            var toolSet = MythicTools.TOOL_MAP.get(material);
            var displayEntitySword = new class_8113.class_8122(class_1299.field_42456, world);
            var displayEntityPickaxe = new class_8113.class_8122(class_1299.field_42456, world);
            var displayEntityAxe = new class_8113.class_8122(class_1299.field_42456, world);
            var displayEntityShovel = new class_8113.class_8122(class_1299.field_42456, world);
            var displayEntityHoe = new class_8113.class_8122(class_1299.field_42456, world);

            displayEntitySword.method_48897(toolSet.getSword().method_7854());
            displayEntityPickaxe.method_48897(toolSet.getPickaxe().method_7854());
            displayEntityAxe.method_48897(toolSet.getAxe().method_7854());
            displayEntityShovel.method_48897(toolSet.getShovel().method_7854());
            displayEntityHoe.method_48897(toolSet.getHoe().method_7854());

            displayEntitySword.method_23327(startPos.method_10263() + 1.25, startPos.method_10264() + 3.5, startPos.method_10260() + 1.2);
            displayEntityPickaxe.method_23327(startPos.method_10263() + 2.5, startPos.method_10264() + 3.5, startPos.method_10260() + 1.2);
            displayEntityAxe.method_23327(startPos.method_10263() + 3.75, startPos.method_10264() + 3.5, startPos.method_10260() + 1.2);
            displayEntityShovel.method_23327(startPos.method_10263() + 5, startPos.method_10264() + 3.5, startPos.method_10260() + 1.2);
            displayEntityHoe.method_23327(startPos.method_10263() + 6.25, startPos.method_10264() + 3.5, startPos.method_10260() + 1.2);

            displayEntitySword.method_36456(180);
            displayEntityPickaxe.method_36456(180);
            displayEntityAxe.method_36456(180);
            displayEntityShovel.method_36456(180);
            displayEntityHoe.method_36456(180);

            world.method_8649(displayEntitySword);
            world.method_8649(displayEntityPickaxe);
            world.method_8649(displayEntityAxe);
            world.method_8649(displayEntityShovel);
            world.method_8649(displayEntityHoe);
        }

        if (MythicArmor.ARMOR_MAP.containsKey(material)) {

        }

        if (MythicBlocks.BLOCKSET_MAP.containsKey(material)) {

        }

        return placements;
    }

    private static void placeStructure(class_1937 world, class_2338 start) {
        // floor
        for (int i = 0; i < 8; i++) {
            for (int j = 0; j < 5; j++) {
                world.method_8501(class_2338.method_49637(start.method_10263() + i, start.method_10264(), start.method_10260() + j), class_2246.field_10107.method_9564());

                if (j == 0) {
                    for (int y = 0; y < 6; y++) {
                        world.method_8501(class_2338.method_49637(start.method_10263() + i, start.method_10264() + y, start.method_10260() + j), class_2246.field_10107.method_9564());
                    }
                }
            }
        }
    }

    /**
     * Command which generates loot from a loot table X amount of times, and prints the output to the console
     */
    private static int testLootTable(CommandContext<class_2168> ctx) {
        var source = ctx.getSource();
        try {
            var lootTable = class_9433.class_9436.method_58481(ctx, "loot_table");

            int rolls = IntegerArgumentType.getInteger(ctx, "rolls");

            class_8567 lootContextParameterSet = new class_8567.class_8568(ctx.getSource().method_9225())
                .method_51877(class_181.field_1226, source.method_9228())
                .method_51874(class_181.field_24424, source.method_9222())
                .method_51875(class_173.field_1179);

            HashMap<class_1792, Integer> map = new HashMap<>();

            for (int i = 0; i < rolls; i++) {
                List<class_1799> list = lootTable.comp_349().method_51878(lootContextParameterSet);
                list.forEach(itemStack -> {
                    int count = map.getOrDefault(itemStack.method_7909(), 0);
                    map.put(itemStack.method_7909(), itemStack.method_7947() + count);
                });
            }

            map.forEach((item, integer) -> {
                source.method_9226(() -> (class_2561.method_43470(item + ": " + integer.toString())), false);
            });
        } catch (CommandSyntaxException e) {
            throw new RuntimeException(e);
        }

        return 0;
    }

    /**
     * ArmorSet exporter for the Mythic Metals Wiki
     * <br><br>
     * Note that it does not handle abilities or custom attributes, like Lava Swim Speed
     */
    private static int exportArmor(CommandContext<class_2168> context) {
        var armorSet = ArmorSetArgumentType.getArmorSet(context, "armorset");
        var source = context.getSource();
        var output = WikiExporter.computeArmorSet(armorSet);

        MythicMetals.LOGGER.info(output);
        source.method_9226(() -> class_2561.method_43470("Exported armor to wiki format in logs"), false);
        return 0;
    }

    /**
     * Ore/Material exporter for the Mythic Metals Wiki
     */
    private static int exportOreData(CommandContext<class_2168> context) {
        var oreConfig = OreConfigArgumentType.getOreConfig(context, "ore-config");
        var source = context.getSource();
        var blockSet = MythicBlocks.BLOCKSET_MAP.get(ORE_CONFIG.inverse().get(oreConfig));

        String oreName = StringUtilsAtHome.toTitleCase(blockSet.getName() + " Ores");

        String template = WikiExporter.createOreTemplate(oreName, blockSet, oreConfig);

        source.method_9226(() -> class_2561.method_43470("Exported ore stats for %s to wiki format".formatted(oreName)), false);
        MythicMetals.LOGGER.info(template);

        return 2;
    }

    /**
     * Tool exporter for the Mythic Metals Wiki
     */
    private static int exportTools(CommandContext<class_2168> context) {
        var toolset = ToolSetArgumentType.getToolSet(context, "toolset");

        var source = context.getSource();
        MythicMetals.LOGGER.info(WikiExporter.computeToolset(toolset));
        source.method_9226(() -> class_2561.method_43470("Exported tools to wiki format"), false);
        return 0;
    }

    /**
     * Summons an armor stand with a specific armor set and trim on top of the world <br>
     * Create the {@link class_8053} using a pattern from {@link net.minecraft.class_8057}
     * and a material from {@link net.minecraft.class_8055}
     *
     * @param world    The world where you want to summon the armor stand, needs to be on the server
     * @param trim     {@link class_8053} you wish to use on the armor.
     * @param armorSet {@link ArmorSet} that you wish to equip on the armor stand.
     * @param x        x-coordinate where the armor stand should spawn
     * @param z        z-coordinate where the armor stand should spawn
     * @return Returns whether the armor set was successfully created and summoned
     * @see class_8053
     * @see MythicArmor
     * @see ArmorSet
     */
    public static boolean summonArmorStandWithTrim(class_1937 world, @Nullable class_8053 trim, ArmorSet armorSet, float x, float z) {
        if (world.field_9236) return false;
        if (armorSet.equals(MythicArmor.TIDESINGER)) return false; // This has custom "trims", ignore it
        AtomicBoolean success = new AtomicBoolean(true);

        var armorStand = new class_1531(world, x, world.method_31600() - 50, z);
        armorSet.getArmorItems().forEach(armorItem -> {
            var armorStack = new class_1799(armorItem);
            if (!armorStack.method_31573(class_3489.field_41890)) {
                MythicMetals.LOGGER.debug("Armor Item %s is not trimmable".formatted(armorStack.method_7964()));
            }
            if (trim != null) {
                armorStack.method_57379(class_9334.field_49607, trim);
            }
            if (success.get()) armorStand.method_5673(armorItem.method_7685(), armorStack);
        });
        world.method_8649(armorStand);
        return success.get();
    }

    /**
     * @param world The world to get the DynamicRegistryManager from. This should only happen on the server
     * @return Returns all armor trims in a sorted ArrayList
     */
    public static ArrayList<class_8053> getAllArmorTrims(class_1937 world) {
        if (world.field_9236) return new ArrayList<>();

        var list = new ArrayList<class_8053>();
        world.method_30349().method_30530(class_7924.field_42083).method_40270().forEach(armorMaterialEntry -> {
            world.method_30349().method_30530(class_7924.field_42082).method_40270().forEach(armorTrimEntry -> {
                list.add(new class_8053(armorMaterialEntry, armorTrimEntry));
            });
        });
        return list;
    }

    /**
     * @param world The world to get the DynamicRegistryManager from. This should only happen on the server
     * @return Returns all the trim patterns in the registry as a list of strings
     */
    public static ArrayList<String> getAllTrimPatternStrs(class_1937 world) {
        if (world.field_9236) return new ArrayList<>();

        var list = new ArrayList<String>();
        world.method_30349().method_30530(class_7924.field_42082).method_40270().forEach(armorTrimEntry -> list.add(armorTrimEntry.comp_349().comp_1213().method_12832()));
        return list;
    }

    /**
     * Suggest which format/location to use when dumping all the ore configs in {@link MythicOreConfigs}
     */
    private static CompletableFuture<Suggestions> dumpType(CommandContext<class_2168> ctx, SuggestionsBuilder suggestion) {
        suggestion.suggest("console");
        return suggestion.buildFuture();
    }

    /**
     * Suggests armor materials from all the armor sets defined in {@link MythicArmor}
     * Includes one extra suggestion for "all"
     */
    private static CompletableFuture<Suggestions> armorMaterial(CommandContext<class_2168> ctx, SuggestionsBuilder suggestion) {
        MythicArmor.ARMOR_MAP.forEach((s, armorSet) -> suggestion.suggest(s));
        suggestion.suggest("all");
        return suggestion.buildFuture();
    }

    private static CompletableFuture<Suggestions> material(CommandContext<class_2168> ctx, SuggestionsBuilder suggestion) {
        var placeableMaterials = new HashSet<String>();
        placeableMaterials.addAll(MythicTools.TOOL_MAP.keySet());
        placeableMaterials.addAll(MythicArmor.ARMOR_MAP.keySet());
        placeableMaterials.add("all");
        placeableMaterials.forEach(suggestion::suggest);
        return suggestion.buildFuture();
    }

    /**
     * Suggests all the armor trim types available in the dynamic registry
     * Includes two extra suggestions for "all" and "none"
     */
    private static CompletableFuture<Suggestions> trimTypes(CommandContext<class_2168> ctx, SuggestionsBuilder suggestion) {
        var list = new ArrayList<class_8056>();
        ctx.getSource().method_30497().method_30530(class_7924.field_42082).method_40270().forEach(armorTrimEntry -> list.add(armorTrimEntry.comp_349()));
        list.forEach(trimPattern -> suggestion.suggest(trimPattern.comp_1213().method_12832()));
        suggestion.suggest("all").suggest("none");
        return suggestion.buildFuture();
    }

    private static int dumpAllOreConfigs(CommandContext<class_2168> context) {
        if (StringArgumentType.getString(context, "type").equals("console")) {
            ReflectionUtils.iterateAccessibleStaticFields(MythicOreConfigs.class, OreConfig.class, (feature, name, field) -> {
                if (!feature.offset && !feature.trapezoid) {
                    context.getSource().method_9226(() -> class_2561.method_43470(
                        name.toUpperCase(Locale.ROOT)
                            + " has the range between "
                            + feature.bottom
                            + " to "
                            + feature.top
                            + ", with a discard chance of "
                            + feature.discardChance * 100 + "%"), false);
                }
                if (feature.offset) {
                    context.getSource().method_9226(() -> class_2561.method_43470(
                        name.toUpperCase(Locale.ROOT)
                            + " has the range between "
                            + feature.bottom
                            + "(offset) to "
                            + feature.top
                            + ", with a discard chance of "
                            + feature.discardChance * 100 + "%"), false);
                }
                if (feature.trapezoid) {
                    context.getSource().method_9226(() -> class_2561.method_43470(
                        name.toUpperCase(Locale.ROOT)
                            + " has a triangle range between "
                            + feature.bottom
                            + " to "
                            + feature.top
                            + ", where the sweet spot is at Y = "
                            + ((feature.bottom + feature.top) / 2)
                            + " with a discard chance of "
                            + feature.discardChance * 100 + "%"), false);
                }

            });
            return 1;
        }
        return -1;
    }

    private static int armorStandCommand(CommandContext<class_2168> context, @NotNull String material, @Nullable String rawTrim) {
        var armorTrims = new ArrayList<class_8053>();
        var world = context.getSource().method_9225();
        var pos = context.getSource().method_9222();
        float x = (int) pos.field_1352 + 0.5f;
        float z = (int) pos.field_1350 + 0.5f;

        final String trimQuery = rawTrim == null ? "none" : rawTrim;

        if (trimQuery.equals("none")) {
            // Summon all permutations of the armor at once. Not recommended due to the sheer density of them
            if (material.equals("all")) {

                int count = 0;
                var armorSetStrings = new TreeSet<>(MythicArmor.ARMOR_MAP.keySet());
                for (var armorSetName : armorSetStrings) {
                    if (summonArmorStandWithTrim(world, null, MythicArmor.ARMOR_MAP.get(armorSetName), x, z)) {
                        x++;
                        count++;
                    }
                }
                int finalCount = count;
                context.getSource().method_9226(() -> class_2561.method_43470("Summoned and dropping %d armorstands".formatted(finalCount)), true);
                return finalCount;
            } else {
                if (summonArmorStandWithTrim(world, null, MythicArmor.ARMOR_MAP.get(material), x, z)) {
                    context.getSource().method_9226(() -> class_2561.method_43470("Summoned and dropping one armorstand"), true);
                    return 1;
                } else {
                    context.getSource().method_9226(() -> class_2561.method_43470("Unable to summon the armor stand. It might be untrimmable"), false);
                }
            }
            return -1;
        }

        if (material.equals("all")) {
            if (MythicArmor.ARMOR_MAP.isEmpty()) {
                context.getSource().method_9226(() -> class_2561.method_43470("Unable to summon. Somehow the armor map is empty..."), false);
                return -1; // "how did this happen?" "a long time ago, actually never..."
            }

            if (trimQuery.equals("all")) {
                armorTrims.addAll(
                    getAllArmorTrims(world).stream().toList()
                );
            } else if (getAllTrimPatternStrs(world).contains(trimQuery)) {
                armorTrims.addAll(getAllArmorTrims(world).stream().filter(trim -> trim.method_48424().comp_349().comp_1213().method_12832().equals(trimQuery)).toList());
            }

            // lambda moment
            MutableInt mutX = new MutableInt(pos.field_1352);
            MutableInt mutZ = new MutableInt(pos.field_1350);
            MutableInt count = new MutableInt(0);

            var armorSetStrings = new TreeSet<>(MythicArmor.ARMOR_MAP.keySet());
            for (var armorSetName : armorSetStrings) {
                armorTrims.forEach(armorTrim -> {
                    if (summonArmorStandWithTrim(world, armorTrim, MythicArmor.ARMOR_MAP.get(armorSetName), mutX.getValue(), mutZ.getValue())) {
                        mutX.increment();
                        count.increment();
                    }
                });
                mutZ.increment();
                mutX.setValue(pos.field_1352);
            }
            context.getSource().method_9226(() -> class_2561.method_43470("Summoned and dropping %d armorstands with trims".formatted(count.getValue())), true);

            return count.getValue();
        } else if (MythicArmor.ARMOR_MAP.get(material) != null) {
            if (trimQuery.equals("all")) {
                armorTrims = getAllArmorTrims(world);
            } else {
                var trims = getAllArmorTrims(world).stream().filter(trim -> trim.method_48424().comp_349().comp_1213().method_12832().equals(trimQuery)).toList();
                armorTrims.addAll(trims);
            }

            // Split the armor stands into groups using these offsets
            int splitPoint = armorTrims.size() / world.method_30349().method_30530(class_7924.field_42083).method_10204();

            // flip-flop
            int xOffset = 0;
            int zOffset = 0;
            int count = 0;

            for (int i = 0; i < armorTrims.size(); i++) {
                if (i % splitPoint == 0) {
                    xOffset += 2;
                    zOffset = 0;
                }
                var armorSet = MythicArmor.ARMOR_MAP.get(material);
                if (summonArmorStandWithTrim(world, armorTrims.get(i), armorSet, (int) pos.field_1352 + xOffset, (int) pos.field_1350 + zOffset)) {
                    count++;
                    zOffset += 2;
                } else {
                    xOffset -= 2;
                }
            }
            String feedback = "Summoned and dropping %d armorstands".formatted(count);
            context.getSource().method_9226(() -> class_2561.method_43470(feedback), true);
            return count;
        }
        return -1;
    }
}
