package com.mythicmetals.command;

import F;
import com.mythicmetals.armor.ArmorSet;
import com.mythicmetals.block.BlockSet;
import com.mythicmetals.config.OreConfig;
import com.mythicmetals.item.tools.MythicTools;
import com.mythicmetals.item.tools.ToolSet;
import com.mythicmetals.misc.StringUtilsAtHome;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import net.minecraft.class_156;
import net.minecraft.class_1738;
import net.minecraft.class_1831;
import net.minecraft.class_2477;
import net.minecraft.class_2960;
import net.minecraft.class_7923;

/**
 * Helper class that contains all the page layouts for the Mythic Metals Wiki
 */
public class WikiExporter {
    private WikiExporter() {}

    static final String ADMONITION_HEADER = """
        !!! info inline end ""
            <center class=tooltip>
        """;

    static final String ADMONIITION_TOP_IMAGE = """
            <h3>**%s**</h3>
            ![WRITE ALT TEXT HERE](%s)<br>
        """;

    static final String ADMONITION_TOOL_IMAGE = """
            <h3>**%s**</h3>
            ![WRITE ALT TEXT HERE]%s<br>
        """;

    static final String RECIPE_SCALING = "{ .sized-image style=\"--image-width: 40%;\" }";
    static final String ICON_SCALE = "{ .sized-image style=\"--image-width: 8%;\" }";

    public static String computeArmorSet(ArmorSet armorSet) {
        return createArmorTemplate(
            armorSet.getTitlecaseName(),
            armorSet.getMaterialId(),
            computeArmorAdmonition(armorSet),
            computeArmorRecipes(armorSet)
        );
    }

    public static String computeToolset(ToolSet toolSet) {
        return createToolTemplate(
            toolSet.getTitlecaseName(),
            computeToolAdmonition(toolSet),
            computeToolRecipes(toolSet)
        );
    }

    public static String createOreTemplate(String oreName, BlockSet blockSet, OreConfig oreConfig) {
        return createOreTemplate(
            oreName,
            computeOreAdmonition(blockSet, oreConfig)
        );
    }

    static String createOreTemplate(String name, String admonition) {
        return """
            
            ---
            title: %s
            project: mythicmetals
            summary: A summary of %s, their history, and where to find them.
            ---
            
            %s
            
            ## Generation
            
            ## Usages
            
            ## Trivia
            
            ## History
            
            """.formatted(name, name, admonition);
    }

    static String createToolTemplate(String name, String admonition, String recipes) {
        return """
            
            ---
            title: %s Tools
            project: mythicmetals
            summary: A summary of %s Tools, their abilities, their history, and how to craft them.
            ---
            
            %s
            ## Obtaining
            
            ### Crafting
            
            Tools can be crafted from [TODO - LINK TO MATERIAL.]
            
            %s
            ## Usages
            
            TODO - Remove if irrelevant, for example if it does not craft into anything
            
            ## Trivia
            
            ## History
            
            """.formatted(name, name, admonition, recipes);
    }

    static String createArmorTemplate(String name, String lowercaseName, String admonition, String recipes) {
        return """
            ---
            title: %s Armor
            project: mythicmetals
            summary: The armor does protect you (TODO).
            armoricon: %s
            ---
            
            %s
            
            ## Obtaining
            
            ### Crafting
            
            This armor can be crafted from [TODO - LINK TO MATERIAL.]
            
            %s
            ## Usages
            
            TODO - Remove if irrelevant, for example if it does not craft into anything
            
            ## Trivia
            
            ## History
            
            """.formatted(name, lowercaseName + ".png", admonition, recipes);
    }

    static String computeToolAdmonition(ToolSet toolSet) {
        var output = new StringBuilder();
        var translationStorage = class_2477.method_10517();
        // tool stats are really annoying to get
        Deque<Integer> damageDeque = new ArrayDeque<>(Arrays.stream(MythicTools.DEFAULT_DAMAGE).boxed().toList());
        var atkSpd = new ArrayDeque<>(toolSet.getAttackSpeed());
        output.append(ADMONITION_HEADER);
        toolSet.get().forEach(tool -> {
            String id = class_7923.field_41178.method_10221(tool).method_12832();
            output.append(ADMONITION_TOOL_IMAGE.formatted(
                translationStorage.method_48307(tool.method_7876()),
                "(../../assets/mythicmetals/%s.png)".formatted(id) + RECIPE_SCALING
            ));
            output.append("""
                    +%s Attack Damage, %s Attack Speed<br>
                    %s Durability<br>
                """.formatted(
                tool.method_8022().method_8028() + damageDeque.pop() + 1,
                BigDecimal.valueOf(atkSpd.pop()).setScale(1, RoundingMode.HALF_UP).toPlainString(),
                tool.method_7854().method_7936()
            ));
        });

        return output.toString();
    }

    static String computeToolRecipes(ToolSet toolSet) {
        StringBuilder output = new StringBuilder();
        for (class_1831 tool : toolSet.get()) {
            String id = class_7923.field_41178.method_10221(tool).method_12832();
            String name = StringUtilsAtHome.toTitleCase(id.replace('_', ' '));
            output.append(("""
                    ![Image of the recipe for %s](../../assets/mythicmetals/recipes/tools/%s.png)%s
                    """
                ).formatted(name, id, RECIPE_SCALING));
        }
        return output.toString();
    }

    static String computeOreAdmonition(BlockSet blockSet, OreConfig oreConfig) {
        var translationStorage = class_2477.method_10517();
        var output = new StringBuilder();

        output.append(ADMONITION_HEADER);
        output.append(ADMONIITION_TOP_IMAGE.formatted(
            translationStorage.method_48307(blockSet.getOre().method_9539()),
            "../../assets/mythicmetals/%s.png".formatted(blockSet.getName() + "_ore")
        ));

        blockSet.getOreVariantsMap().forEach((variantName, block) -> {
            String variantOreName = translationStorage.method_48307(block.method_9539());
            output.append(ADMONIITION_TOP_IMAGE.formatted(
                variantOreName,
                "../../assets/mythicmetals/" + variantName + "_" + blockSet.getName() + "_ore.png"
            ));
        });

        var oreStatsTemplate = """
                ---
                **Mining Level**: X (Y for variant)<br>
                **Max Vein Size**: %s<br>
                **Attempts Per Chunk**: %s<br>
                **Spawn Range**: %s to %s<br>
                **Discard Chance**: %s<br>
            """.formatted(
            oreConfig.veinSize,
            oreConfig.perChunk,
            oreConfig.bottom + (oreConfig.offset ? "(Offset)" : ""),
            oreConfig.top + (oreConfig.trapezoid ? " (Triangle Range)" : ""),
            oreConfig.discardChance == 0 ? "Never discarded" : oreConfig.discardChance * 100 + "%"
        );

        output.append(oreStatsTemplate);
        return output.toString();
    }

    static String computeArmorAdmonition(ArmorSet armorSet) {
        var translationStorage = class_2477.method_10517();
        var output = new StringBuilder();
        output.append(ADMONITION_HEADER);
        final String armorTitleName = armorSet.getTitlecaseName();
        var armorModelImage = "../../assets/armor-models/256/%s".formatted(armorSet.getMaterialId() + "_256.png");
        output.append(ADMONIITION_TOP_IMAGE.formatted(armorTitleName + " Armor", armorModelImage));

        for (var armor : armorSet.getArmorItems()) {
            var item = class_7923.field_41178.method_10221(armor);
            String name = translationStorage.method_48307(class_156.method_646("item", item));
            String id = item.method_12832();

            int protection = armor.method_7687();

            output.append("\n");
            output.append("\t<h4>**").append(name).append("**</h4>").append("\n");
            output.append("\t![Image of %s](../../assets/mythicmetals/%s.png)".formatted(name, id)).append(RECIPE_SCALING).append("<br>");
            for (int i = 1; i < protection; i = i + 2) {
                output.append("\t![armor](../../assets/icon/full_armor_icon.png)").append(ICON_SCALE).append("\n");
            }
            if ((protection & 1) == 1) {
                output.append("\t![armor](../../assets/icon/half_armor_icon.png)").append(ICON_SCALE).append("\n");
            }
            output.append("\t<br>\n");
            // +5 Armor, +2 Toughness
            output.append("\t+%s Armor".formatted(protection));
            if (armor.method_26353() > 0) {
                output.append(", +%s Toughness".formatted(armor.method_26353()));
            }
            output.append("<br>\n");
            var kbRes = armor.method_7686().comp_349().comp_2304();
            if (kbRes > 0) {
                output.append("\t+%s Knockback Resistance".formatted(kbRes)).append("<br>\n");
            }
            // 350 Durability
            output.append("\t%s Durability".formatted(armor.method_7854().method_7936())).append("<br>\n");
        }
        return output.toString();
    }

    static String computeArmorRecipes(ArmorSet armorSet) {
        StringBuilder output = new StringBuilder();
        for (class_1738 armor : armorSet.getArmorItems()) {
            String id = class_7923.field_41178.method_10221(armor).method_12832();
            String name = StringUtilsAtHome.toTitleCase(id.replace('_', ' '));
            output.append(("""
                    ![Image of the recipe for %s](../../assets/mythicmetals/recipes/armor/%s.png)%s
                    """
                ).formatted(name, id, RECIPE_SCALING));
        }
        return output.toString();
    }
}
