/*
 * Decompiled with CFR 0.152.
 */
package com.celestialswords.dependency;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;

public class DependencyManager {
    private final JavaPlugin plugin;
    private final Logger logger;
    private final Map<String, PluginDependency> dependencies;
    private final File pluginsFolder;
    private final File updateFolder;

    public DependencyManager(JavaPlugin plugin) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
        this.dependencies = new HashMap<String, PluginDependency>();
        this.pluginsFolder = new File(plugin.getDataFolder().getParentFile(), "");
        this.updateFolder = plugin.getServer().getUpdateFolderFile();
    }

    public DependencyManager registerDependency(String pluginName, String projectId, String version) {
        this.dependencies.put(pluginName.toLowerCase(), new PluginDependency(pluginName, projectId, version));
        return this;
    }

    public DependencyManager registerDependency(String pluginName, String projectId, String version, String checksum) {
        this.dependencies.put(pluginName.toLowerCase(), new PluginDependency(pluginName, projectId, version, checksum));
        return this;
    }

    public void checkAndInstallDependencies() {
        this.logger.info("Checking plugin dependencies...");
        this.checkForSelfUpdate();
        for (PluginDependency dependency : this.dependencies.values()) {
            String pluginName = dependency.getName();
            if (pluginName.equalsIgnoreCase("FaithBreak")) {
                Plugin existingFaithBreak = Bukkit.getPluginManager().getPlugin(pluginName);
                if (existingFaithBreak != null && existingFaithBreak.isEnabled()) {
                    this.logger.info("FaithBreak is already loaded and enabled. Skipping reinstallation to avoid conflicts.");
                    this.logger.info("Current FaithBreak version: " + existingFaithBreak.getDescription().getVersion());
                    continue;
                }
                this.logger.info("Installing FaithBreak dependency...");
                this.downloadAndInstallPlugin(dependency);
                continue;
            }
            Plugin dependencyPlugin = Bukkit.getPluginManager().getPlugin(pluginName);
            if (dependencyPlugin == null) {
                this.logger.warning("Required dependency '" + pluginName + "' is missing. Attempting to download and install...");
                this.downloadAndInstallPlugin(dependency);
                continue;
            }
            this.logger.info("Dependency '" + pluginName + "' is already installed.");
        }
    }

    private void checkForSelfUpdate() {
        String projectId = "bqBp9WRo";
        String currentVersion = this.plugin.getDescription().getVersion();
        this.logger.info("Checking for CelestialSwords updates...");
        this.logger.info("Current version: " + currentVersion);
        ModrinthVersionInfo versionInfo = this.fetchLatestVersionFromModrinth("CelestialSwords", projectId);
        if (versionInfo == null || versionInfo.version == null || versionInfo.downloadUrl == null) {
            this.logger.warning("Failed to fetch version information for CelestialSwords from Modrinth.");
            return;
        }
        String latestVersion = versionInfo.version;
        String downloadUrl = versionInfo.downloadUrl;
        this.logger.info("Latest version available: " + latestVersion);
        if (this.shouldUpdate(currentVersion, latestVersion)) {
            this.logger.info("A newer version of CelestialSwords is available: " + latestVersion);
            this.logger.info("Downloading update automatically...");
            this.downloadSelfUpdate(latestVersion, downloadUrl);
        } else {
            this.logger.info("CelestialSwords is up to date.");
        }
    }

    private void downloadSelfUpdate(String version, String downloadUrl) {
        try {
            String fileName;
            File outputFile;
            if (!this.updateFolder.exists()) {
                this.updateFolder.mkdirs();
            }
            if ((outputFile = new File(this.updateFolder, fileName = "CelestialSwords-" + version + ".jar")).exists()) {
                this.logger.info("Update version " + version + " is already downloaded. It will be installed on next restart.");
                return;
            }
            URL url = new URL(downloadUrl);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestProperty("User-Agent", "CelestialSwords-DependencyManager/1.0");
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(60000);
            connection.setInstanceFollowRedirects(true);
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                this.logger.severe("Failed to download CelestialSwords update. Server returned response code: " + responseCode);
                return;
            }
            try (InputStream in = connection.getInputStream();
                 ReadableByteChannel rbc = Channels.newChannel(in);
                 FileOutputStream fos = new FileOutputStream(outputFile);){
                fos.getChannel().transferFrom(rbc, 0L, Long.MAX_VALUE);
                this.logger.info("Successfully downloaded CelestialSwords update v" + version + " to update folder!");
                this.logger.info("The update will be automatically installed when the server restarts.");
                this.logger.info("Update file location: " + outputFile.getAbsolutePath());
                this.plugin.getServer().getScheduler().runTaskLater((Plugin)this.plugin, () -> this.plugin.getServer().broadcast(String.valueOf(ChatColor.GREEN) + "[CelestialSwords] " + String.valueOf(ChatColor.WHITE) + "A new version (" + String.valueOf(ChatColor.GREEN) + version + String.valueOf(ChatColor.WHITE) + ") has been downloaded and will be installed on next restart!", "astral.admin"), 20L);
            }
        }
        catch (IOException e) {
            this.logger.log(Level.SEVERE, "Error downloading CelestialSwords update", e);
        }
    }

    private boolean shouldUpdate(String currentVersion, String latestVersion) {
        if (currentVersion.equals(latestVersion)) {
            return false;
        }
        try {
            String[] currentParts = currentVersion.split("\\.");
            String[] latestParts = latestVersion.split("\\.");
            int maxLength = Math.max(currentParts.length, latestParts.length);
            for (int i = 0; i < maxLength; ++i) {
                int latestPart;
                int currentPart = i < currentParts.length ? Integer.parseInt(currentParts[i]) : 0;
                int n = latestPart = i < latestParts.length ? Integer.parseInt(latestParts[i]) : 0;
                if (latestPart > currentPart) {
                    return true;
                }
                if (latestPart >= currentPart) continue;
                return false;
            }
        }
        catch (NumberFormatException e) {
            this.logger.warning("Could not parse version numbers, using string comparison");
            return !currentVersion.equals(latestVersion);
        }
        return false;
    }

    private void downloadAndInstallPlugin(PluginDependency dependency) {
        Plugin existingPlugin;
        String pluginName = dependency.getName();
        String projectId = dependency.getProjectId();
        String expectedChecksum = dependency.getChecksum();
        if (pluginName.equalsIgnoreCase("FaithBreak") && (existingPlugin = Bukkit.getPluginManager().getPlugin(pluginName)) != null) {
            this.logger.info("FaithBreak is already loaded. Skipping download to avoid duplicate plugin conflicts.");
            return;
        }
        ModrinthVersionInfo versionInfo = this.fetchLatestVersionFromModrinth(pluginName, projectId);
        if (versionInfo == null || versionInfo.version == null || versionInfo.downloadUrl == null) {
            this.logger.severe("Failed to fetch version information for '" + pluginName + "' from Modrinth.");
            return;
        }
        String version = versionInfo.version;
        String downloadUrl = versionInfo.downloadUrl;
        File outputFile = new File(this.pluginsFolder, pluginName + "-" + version + ".jar");
        File[] existingFiles = this.pluginsFolder.listFiles();
        if (existingFiles != null) {
            for (File existingFile : existingFiles) {
                if (!existingFile.getName().startsWith(pluginName + "-") || !existingFile.getName().endsWith(".jar")) continue;
                this.logger.info("Found existing file for '" + pluginName + "': " + existingFile.getName());
                if (!this.loadExistingPlugin(existingFile, pluginName)) continue;
                return;
            }
        }
        try {
            URL url = new URL(downloadUrl);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestProperty("User-Agent", "CelestialSwords-DependencyManager/1.0");
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(60000);
            connection.setInstanceFollowRedirects(true);
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                this.logger.severe("Failed to download '" + pluginName + "'. Server returned response code: " + responseCode);
                return;
            }
            try (InputStream in = connection.getInputStream();
                 ReadableByteChannel rbc = Channels.newChannel(in);
                 FileOutputStream fos = new FileOutputStream(outputFile);){
                fos.getChannel().transferFrom(rbc, 0L, Long.MAX_VALUE);
                this.logger.info("Successfully downloaded '" + pluginName + "' to " + outputFile.getAbsolutePath());
                if (expectedChecksum != null && !expectedChecksum.isEmpty()) {
                    String actualChecksum = this.calculateSHA256(outputFile);
                    if (!actualChecksum.equalsIgnoreCase(expectedChecksum)) {
                        this.logger.severe("Checksum verification failed for '" + pluginName + "'!");
                        this.logger.severe("Expected: " + expectedChecksum);
                        this.logger.severe("Actual: " + actualChecksum);
                        this.logger.severe("The downloaded file may be corrupted or tampered with. Deleting it for security.");
                        outputFile.delete();
                        return;
                    }
                    this.logger.info("Checksum verification passed for '" + pluginName + "'.");
                }
                File pluginFile = outputFile;
                this.plugin.getServer().getScheduler().runTaskLater((Plugin)this.plugin, () -> {
                    try {
                        Plugin loadedPlugin = Bukkit.getPluginManager().loadPlugin(pluginFile);
                        if (loadedPlugin != null) {
                            Bukkit.getPluginManager().enablePlugin(loadedPlugin);
                            this.logger.info("Successfully loaded and enabled '" + pluginName + "' v" + version + "!");
                            if (!loadedPlugin.isEnabled()) {
                                this.logger.severe("Plugin '" + pluginName + "' was loaded but could not be enabled. Please check for compatibility issues or conflicts.");
                            }
                        } else {
                            this.logger.severe("Failed to load '" + pluginName + "'. The downloaded file may be corrupted or incompatible.");
                        }
                    }
                    catch (Exception e) {
                        this.logger.log(Level.SEVERE, "Critical error enabling downloaded plugin '" + pluginName + "'.", e);
                    }
                }, 20L);
            }
        }
        catch (IOException e) {
            this.logger.log(Level.SEVERE, "Error downloading plugin '" + pluginName + "'", e);
        }
    }

    private ModrinthVersionInfo fetchLatestVersionFromModrinth(String pluginName, String projectId) {
        try {
            String line;
            URL url = new URL("https://api.modrinth.com/v2/project/" + projectId + "/version");
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("GET");
            connection.setRequestProperty("User-Agent", "CelestialSwords-DependencyManager/1.0");
            connection.setConnectTimeout(5000);
            connection.setReadTimeout(5000);
            if (connection.getResponseCode() != 200) {
                this.logger.warning("Failed to fetch version info for '" + pluginName + "': HTTP " + connection.getResponseCode());
                return null;
            }
            BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
            StringBuilder response = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            reader.close();
            String jsonResponse = response.toString();
            if (jsonResponse.isEmpty() || jsonResponse.equals("[]")) {
                this.logger.warning("No versions found for '" + pluginName + "' with project ID: " + projectId);
                return null;
            }
            String versionNumber = this.extractJsonField(jsonResponse, "\"version_number\":");
            String downloadUrl = this.extractJsonField(jsonResponse, "\"url\":");
            if (versionNumber == null || downloadUrl == null) {
                this.logger.warning("Failed to parse version or download URL for '" + pluginName + "'");
                return null;
            }
            return new ModrinthVersionInfo(versionNumber, downloadUrl);
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error fetching version info for '" + pluginName + "'", e);
            return null;
        }
    }

    private String extractJsonField(String json, String fieldName) {
        int fieldIndex = json.indexOf(fieldName);
        if (fieldIndex == -1) {
            return null;
        }
        int startIndex = fieldIndex + fieldName.length();
        int endIndex = json.indexOf(",", startIndex);
        if (endIndex == -1) {
            endIndex = json.indexOf("}", startIndex);
        }
        if (endIndex == -1) {
            return null;
        }
        String value = json.substring(startIndex, endIndex).trim();
        if (value.startsWith("\"") && value.endsWith("\"")) {
            value = value.substring(1, value.length() - 1);
        }
        return value;
    }

    private boolean loadExistingPlugin(File pluginFile, String pluginName) {
        try {
            Plugin existingPlugin = Bukkit.getPluginManager().getPlugin(pluginName);
            if (existingPlugin != null && existingPlugin.isEnabled()) {
                this.logger.info("Plugin '" + pluginName + "' is already loaded and enabled.");
                return true;
            }
            if (existingPlugin != null) {
                this.logger.info("Plugin '" + pluginName + "' is already loaded but not enabled. Skipping to avoid conflicts.");
                return true;
            }
            this.plugin.getServer().getScheduler().runTaskLater((Plugin)this.plugin, () -> {
                try {
                    Plugin loadedPlugin = Bukkit.getPluginManager().loadPlugin(pluginFile);
                    if (loadedPlugin != null) {
                        Bukkit.getPluginManager().enablePlugin(loadedPlugin);
                        this.logger.info("Successfully loaded and enabled existing '" + pluginName + "' from " + pluginFile.getName());
                    } else {
                        this.logger.warning("Failed to load existing '" + pluginName + "' from " + pluginFile.getName());
                    }
                }
                catch (Exception e) {
                    this.logger.log(Level.WARNING, "Error loading existing plugin '" + pluginName + "' from " + pluginFile.getName(), e);
                }
            }, 10L);
            return true;
        }
        catch (Exception e) {
            this.logger.log(Level.WARNING, "Error loading existing plugin '" + pluginName + "' from " + pluginFile.getName(), e);
            return false;
        }
    }

    private void unloadExistingPlugin(String pluginName) {
        Plugin existingPlugin = Bukkit.getPluginManager().getPlugin(pluginName);
        if (existingPlugin != null) {
            this.logger.info("Found existing '" + pluginName + "' plugin. Attempting to unload it...");
            try {
                if (existingPlugin.isEnabled()) {
                    Bukkit.getPluginManager().disablePlugin(existingPlugin);
                    this.logger.info("Successfully disabled existing '" + pluginName + "' plugin.");
                }
                this.logger.info("Existing '" + pluginName + "' plugin has been disabled.");
                this.logger.info("Skipping reinstallation to avoid duplicate plugin conflicts.");
            }
            catch (Exception e) {
                this.logger.log(Level.WARNING, "Error while unloading existing '" + pluginName + "' plugin", e);
                this.logger.warning("The existing plugin may still be active. Server restart may be required for clean installation.");
            }
        }
    }

    private void cleanupOldPluginFiles(String pluginName) {
        File[] files = this.pluginsFolder.listFiles();
        if (files == null) {
            this.logger.warning("Could not list files in plugins folder for cleanup.");
            return;
        }
        for (File file : files) {
            String fileName = file.getName();
            if (!fileName.toLowerCase().startsWith(pluginName.toLowerCase()) || !fileName.endsWith(".jar") && !fileName.endsWith(".jar.disabled")) continue;
            this.logger.info("Removing old '" + pluginName + "' file: " + fileName);
            try {
                if (file.delete()) {
                    this.logger.info("Successfully removed old file: " + fileName);
                    continue;
                }
                this.logger.warning("Failed to remove old file: " + fileName + ". It may be in use.");
                File disabledFile = new File(file.getParent(), fileName + ".disabled");
                if (file.renameTo(disabledFile)) {
                    this.logger.info("Renamed old file to: " + disabledFile.getName());
                    continue;
                }
                this.logger.warning("Could not remove or disable old file: " + fileName);
            }
            catch (Exception e) {
                this.logger.log(Level.WARNING, "Error removing old plugin file: " + fileName, e);
            }
        }
    }

    private String calculateSHA256(File file) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] fileBytes = Files.readAllBytes(file.toPath());
            byte[] hashBytes = digest.digest(fileBytes);
            StringBuilder hexString = new StringBuilder();
            for (byte hashByte : hashBytes) {
                String hex = Integer.toHexString(0xFF & hashByte);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }
            return hexString.toString();
        }
        catch (IOException | NoSuchAlgorithmException e) {
            this.logger.log(Level.SEVERE, "Error calculating file checksum", e);
            return "";
        }
    }

    private static class PluginDependency {
        private final String name;
        private final String projectId;
        private final String version;
        private final String checksum;

        public PluginDependency(String name, String projectId, String version) {
            this(name, projectId, version, null);
        }

        public PluginDependency(String name, String projectId, String version, String checksum) {
            this.name = name;
            this.projectId = projectId;
            this.version = version;
            this.checksum = checksum;
        }

        public String getName() {
            return this.name;
        }

        public String getProjectId() {
            return this.projectId;
        }

        public String getVersion() {
            return this.version;
        }

        public String getChecksum() {
            return this.checksum;
        }
    }

    private static class ModrinthVersionInfo {
        private final String version;
        private final String downloadUrl;

        public ModrinthVersionInfo(String version, String downloadUrl) {
            this.version = version;
            this.downloadUrl = downloadUrl;
        }
    }
}

