/*
 * Decompiled with CFR 0.152.
 */
package eu.prismm;

import eu.prismm.FilterConfig;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ExceptionLogger {
    private static final Logger LOGGER = LogManager.getLogger((String)"noconsolespam");
    private static final String ERROR_DIRECTORY = "Console Errors";
    private final FilterConfig.ExceptionSettings settings;
    private boolean isInitialized = false;
    private static final Pattern MOD_PACKAGE_PATTERN = Pattern.compile("(net\\.minecraft|net\\.fabricmc|io\\.fabric|com\\.mojang|net\\.minecraftforge|org\\.bukkit|org\\.spigotmc|cpw\\.mods|mezz\\.jei|vazkii\\.botania|joptsimple|com\\.google|org\\.apache|org\\.slf4j|java\\.util|java\\.lang|java\\.io|java\\.nio|oshi\\..|joml\\..|[a-z0-9_]+\\.[a-z0-9_]+\\.[a-z0-9_]+)", 2);
    private static final Pattern FABRIC_EXCEPTION_PATTERN = Pattern.compile("(net\\.fabricmc|io\\.fabric)", 2);
    private static final Pattern MINECRAFT_EXCEPTION_PATTERN = Pattern.compile("(net\\.minecraft|com\\.mojang)", 2);
    private static final Map<String, Pattern> EXCEPTION_CATEGORIES = new HashMap<String, Pattern>();

    public ExceptionLogger(FilterConfig.ExceptionSettings settings) {
        this.settings = settings;
        if (settings.isCaptureExceptions()) {
            try {
                this.ensureErrorDirectoryExists();
                this.isInitialized = true;
                LOGGER.info("Exception logger initialized with directory: {}", (Object)new File(ERROR_DIRECTORY).getAbsolutePath());
            }
            catch (Exception e) {
                LOGGER.error("Failed to initialize exception logger", (Throwable)e);
            }
        }
    }

    private void ensureErrorDirectoryExists() {
        File errorDir = new File(ERROR_DIRECTORY);
        if (!errorDir.exists()) {
            if (errorDir.mkdirs()) {
                LOGGER.info("Created error directory: {}", (Object)errorDir.getAbsolutePath());
            } else {
                LOGGER.error("Failed to create error directory: {}", (Object)errorDir.getAbsolutePath());
                this.isInitialized = false;
            }
        }
    }

    private File ensureSourceDirectoryExists(String source) {
        if (!this.settings.isOrganizeBySource()) {
            return new File(ERROR_DIRECTORY);
        }
        File sourceDir = new File(ERROR_DIRECTORY, source);
        if (!sourceDir.exists()) {
            if (sourceDir.mkdirs()) {
                LOGGER.debug("Created source-specific error directory: {}", (Object)sourceDir.getAbsolutePath());
            } else {
                LOGGER.error("Failed to create source-specific error directory: {}", (Object)sourceDir.getAbsolutePath());
                return new File(ERROR_DIRECTORY);
            }
        }
        return sourceDir;
    }

    private String extractExceptionSource(Throwable throwable) {
        StackTraceElement[] stackTrace;
        if (throwable == null) {
            return "Unknown";
        }
        Object source = throwable.getClass().getSimpleName();
        String category = this.categorizeException(throwable);
        if (category != null) {
            source = category + "-" + (String)source;
        }
        if ((stackTrace = throwable.getStackTrace()) != null && stackTrace.length > 0) {
            for (StackTraceElement element : stackTrace) {
                String packageName;
                String[] parts;
                String className = element.getClassName();
                Matcher matcher = MOD_PACKAGE_PATTERN.matcher(className);
                if (!matcher.find() || (parts = (packageName = matcher.group(1)).split("\\.")).length < 2) continue;
                source = parts.length >= 3 ? parts[2] : parts[1];
                break;
            }
        }
        return ((String)source).replaceAll("[^a-zA-Z0-9.-]", "_");
    }

    private String categorizeException(Throwable throwable) {
        if (throwable == null) {
            return null;
        }
        String className = throwable.getClass().getName();
        for (Map.Entry<String, Pattern> entry : EXCEPTION_CATEGORIES.entrySet()) {
            if (!entry.getValue().matcher(className).matches()) continue;
            return entry.getKey();
        }
        if (throwable instanceof RuntimeException) {
            return "Runtime";
        }
        if (throwable instanceof IOException) {
            return "IO";
        }
        if (throwable instanceof Error) {
            return "Error";
        }
        if (throwable instanceof Exception) {
            return "General";
        }
        return null;
    }

    private boolean shouldCaptureException(Throwable throwable) {
        if (throwable == null) {
            return false;
        }
        if (throwable instanceof RuntimeException) {
            return this.settings.isCaptureRuntimeExceptions();
        }
        if (throwable instanceof IOException) {
            return this.settings.isCaptureIOExceptions();
        }
        if (throwable instanceof Error) {
            if (throwable instanceof AssertionError) {
                return true;
            }
            return true;
        }
        if (throwable instanceof ClassNotFoundException || throwable instanceof CloneNotSupportedException || throwable instanceof IllegalAccessException || throwable instanceof InstantiationException || throwable instanceof InterruptedException || throwable instanceof NoSuchFieldException || throwable instanceof NoSuchMethodException || throwable instanceof ReflectiveOperationException) {
            return true;
        }
        if (this.settings.isCaptureFabricExceptions()) {
            if (FABRIC_EXCEPTION_PATTERN.matcher(throwable.getClass().getName()).find()) {
                return true;
            }
            for (StackTraceElement element : throwable.getStackTrace()) {
                if (!FABRIC_EXCEPTION_PATTERN.matcher(element.getClassName()).find()) continue;
                return true;
            }
        }
        if (this.settings.isCaptureMinecraftExceptions()) {
            if (MINECRAFT_EXCEPTION_PATTERN.matcher(throwable.getClass().getName()).find()) {
                return true;
            }
            for (StackTraceElement element : throwable.getStackTrace()) {
                if (!MINECRAFT_EXCEPTION_PATTERN.matcher(element.getClassName()).find()) continue;
                return true;
            }
        }
        return true;
    }

    public void logException(String message, Throwable exception) {
        if (!(this.isInitialized && this.settings.isCaptureExceptions() && this.shouldCaptureException(exception))) {
            return;
        }
        String source = this.extractExceptionSource(exception);
        String timestamp = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss-SSS").format(new Date());
        String exceptionType = exception.getClass().getSimpleName();
        String filename = String.format("%s_%s.log", timestamp, exceptionType);
        File sourceDir = this.ensureSourceDirectoryExists(source);
        File logFile = new File(sourceDir, filename);
        try (FileWriter fw = new FileWriter(logFile);
             PrintWriter pw = new PrintWriter(fw);){
            pw.println("Date: " + new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(new Date()));
            pw.println("Exception Type: " + exception.getClass().getName());
            pw.println("Category: " + (this.categorizeException(exception) != null ? this.categorizeException(exception) : "Uncategorized"));
            pw.println("Source: " + source);
            pw.println("Message: " + message);
            pw.println("Exception Message: " + exception.getMessage());
            Throwable cause = exception.getCause();
            if (cause != null) {
                pw.println("Caused by: " + cause.getClass().getName() + ": " + cause.getMessage());
            }
            pw.println("\nStackTrace:");
            exception.printStackTrace(pw);
            Throwable[] suppressed = exception.getSuppressed();
            if (suppressed != null && suppressed.length > 0) {
                pw.println("\nSuppressed Exceptions:");
                for (Throwable t : suppressed) {
                    pw.println("  Suppressed: " + t.getClass().getName() + ": " + t.getMessage());
                    t.printStackTrace(pw);
                }
            }
            LOGGER.debug("Logged exception to file: {}", (Object)logFile.getAbsolutePath());
            this.enforceMaxFileCount();
        }
        catch (IOException e) {
            LOGGER.error("Failed to write exception to log file", (Throwable)e);
        }
    }

    private void enforceMaxFileCount() {
        File dir = new File(ERROR_DIRECTORY);
        this.enforceMaxFileCountInDir(dir);
        File[] subdirs = dir.listFiles(File::isDirectory);
        if (subdirs != null) {
            for (File subdir : subdirs) {
                this.enforceMaxFileCountInDir(subdir);
            }
        }
    }

    private void enforceMaxFileCountInDir(File dir) {
        File[] files = dir.listFiles(File::isFile);
        if (files != null && files.length > this.settings.getMaxBackupIndex()) {
            Arrays.sort(files, (f1, f2) -> Long.compare(f1.lastModified(), f2.lastModified()));
            for (int i = 0; i < files.length - this.settings.getMaxBackupIndex(); ++i) {
                if (files[i].delete()) {
                    LOGGER.debug("Deleted old log file: {}", (Object)files[i].getName());
                    continue;
                }
                LOGGER.warn("Failed to delete old log file: {}", (Object)files[i].getName());
            }
        }
    }

    public boolean isInitialized() {
        return this.isInitialized;
    }

    static {
        EXCEPTION_CATEGORIES.put("Runtime", Pattern.compile(".*RuntimeException"));
        EXCEPTION_CATEGORIES.put("IO", Pattern.compile(".*IOException"));
        EXCEPTION_CATEGORIES.put("Concurrent", Pattern.compile(".*ConcurrentModificationException|.*InterruptedException"));
        EXCEPTION_CATEGORIES.put("Reflection", Pattern.compile(".*ReflectiveOperationException|.*IllegalAccessException|.*NoSuchMethodException|.*InstantiationException"));
        EXCEPTION_CATEGORIES.put("Security", Pattern.compile(".*SecurityException"));
        EXCEPTION_CATEGORIES.put("Network", Pattern.compile(".*SocketException|.*ConnectException|.*UnknownHostException"));
        EXCEPTION_CATEGORIES.put("Parse", Pattern.compile(".*ParseException|.*NumberFormatException|.*DateTimeException"));
        EXCEPTION_CATEGORIES.put("Data", Pattern.compile(".*SQLException|.*JsonException|.*JsonSyntaxException"));
        EXCEPTION_CATEGORIES.put("Minecraft", Pattern.compile(".*CommandSyntaxException"));
        EXCEPTION_CATEGORIES.put("Graphics", Pattern.compile(".*RenderException|.*TextureException"));
        EXCEPTION_CATEGORIES.put("Memory", Pattern.compile(".*OutOfMemoryError"));
    }
}

