package com.github.mkram17.bazaarutils.features;


import com.github.mkram17.bazaarutils.BazaarUtils;
import com.github.mkram17.bazaarutils.config.BUConfig;
import com.github.mkram17.bazaarutils.events.*;
import com.github.mkram17.bazaarutils.events.handlers.BUListener;
import com.github.mkram17.bazaarutils.misc.CustomItemButton;
import com.github.mkram17.bazaarutils.misc.orderinfo.BazaarOrder;
import com.github.mkram17.bazaarutils.misc.orderinfo.OrderInfoContainer;
import com.github.mkram17.bazaarutils.misc.orderinfo.PriceInfoContainer;
import com.github.mkram17.bazaarutils.utils.GUIUtils;
import com.github.mkram17.bazaarutils.utils.ScreenInfo;
import com.github.mkram17.bazaarutils.utils.SoundUtil;
import com.github.mkram17.bazaarutils.utils.Util;
import dev.isxander.yacl3.api.Option;
import lombok.Getter;
import lombok.Setter;
import meteordevelopment.orbit.EventHandler;
import meteordevelopment.orbit.EventPriority;
import net.minecraft.class_124;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_476;
import net.minecraft.class_9290;
import net.minecraft.class_9334;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.github.mkram17.bazaarutils.BazaarUtils.EVENT_BUS;

//TODO switch to finding market price without finding the OrderData first. Then, OrderUpdater should handle fixing it. Or just do it that way for redundancy.
public class FlipHelper extends CustomItemButton implements BUListener {

    private static final int FLIP_ORDER_SLOT = 15;
    private static final Pattern PRICE_PATTERN = Pattern.compile("([\\d,.]+) coins");
    private static final Pattern VOLUME_PATTERN = Pattern.compile("([\\d,]+)");
    private static final String FLIP_ORDER_IDENTIFIER = "Flip Order";
    private static final String CANNOT_CANCEL_IDENTIFIER = "can't be flipped";
    private static final int LORE_LINE_VOLUME = 1;
    private static final int LORE_LINE_PRICE = 3;


    private boolean shouldAddToSign = false;
    @Getter @Setter
    private boolean enabled;
    @Getter
    private static final class_1792 BUTTON_ITEM = class_1802.field_42709;
    private BazaarOrder order;

    public FlipHelper(boolean enabled, int slotNumber) {
        this.enabled = enabled;
        this.slotNumber = slotNumber;
    }

    @EventHandler(priority = EventPriority.HIGHEST)
    public void onChestLoaded(ChestLoadedEvent e) {
        if (!enabled) {
            return;
        }
        if(!inCorrectScreen()){
            resetState();
            return;
        }

        try {
            class_1799 flipOrderSign = getFlipSign(e.getItemStacks()).orElse(new class_1799(class_1802.field_8077, 1));
            Optional<BazaarOrder> orderOptional = matchToUserOrder(flipOrderSign.method_57353().method_58694(class_9334.field_49632));
            if (orderOptional.isEmpty()) {
                return;
            }
            order = orderOptional.get();
        } catch (Exception ex) {
            Util.notifyError("Error while trying to find flip item in Flip Helper", ex);
        }
    }

    @EventHandler
    public void onSlotClicked(SlotClickEvent event) {
        if (!enabled || event.slot.method_34266() != slotNumber || !inCorrectScreen() || order == null) {
            return;
        }

        SoundUtil.playSound(BUTTON_SOUND, BUTTON_VOLUME);
        GUIUtils.clickSlot(FLIP_ORDER_SLOT,0);
        shouldAddToSign = true;
    }

    @EventHandler
    public void onSignOpen(SignOpenEvent e){
        if(!shouldAddToSign) return;
        handleFlip();
        shouldAddToSign = false;
    }

    @EventHandler(priority = EventPriority.HIGHEST)
    public void replaceItemEvent(ReplaceItemEvent event) {
        if(!enabled || !(event.getSlotId() == slotNumber) || !inCorrectScreen() || order == null)
            return;

        class_1799 itemStack = new class_1799(BUTTON_ITEM, 1);
        itemStack.method_57379(class_9334.field_49631, getButtonText());
        itemStack.method_57379(BazaarUtils.CUSTOM_SIZE_COMPONENT, getButtonStackSize());
        event.setReplacement(itemStack);
    }

    private class_2561 getButtonText() {
        double flipPrice = order.getFlipPrice();
        if (flipPrice == 0) {
            return class_2561.method_43470("There are no competing sell offers.").method_27692(class_124.field_1064);
        } else if (order == null) {
            return class_2561.method_43470("Could not find order").method_27692(class_124.field_1064);
        } else {
            return class_2561.method_43470("Flip order for " + Util.getPrettyString(flipPrice) + " coins").method_27692(class_124.field_1064);
        }
    }

    private String getButtonStackSize() {
        double flipPrice = order.getFlipPrice();
        if (order.getFlipPrice() == 0) {
            return "ANY";
        } else if (order == null) {
            return "???";
        } else {
            return String.valueOf(Util.truncateNum(flipPrice));
        }
    }

    private void resetState() {
        this.order = null;
        this.shouldAddToSign = false;
    }

    private void handleFlip() {
        double flipPrice = order.getFlipPrice();
        ScreenInfo previousScreen = ScreenInfo.getCurrentScreenInfo().getPreviousScreenInfo();
        if(order != null && flipPrice != 0 && previousScreen.inMenu(ScreenInfo.BazaarMenuType.FLIP_GUI)) {
            GUIUtils.setSignText(Double.toString(Util.truncateNum(flipPrice)), true);
            order.flipItem(flipPrice);
        }
    }

    private Optional<class_1799> getFlipSign(List<class_1799> chestItemStacks) {
        for (class_1799 itemStack : chestItemStacks) {
            if (itemStack == null || itemStack.method_7960()) {
                continue;
            }

            if (itemStack.method_7964().getString().contains(FLIP_ORDER_IDENTIFIER)) {
                class_9290 lore = itemStack.method_57353().method_58694(class_9334.field_49632);
                if (lore != null) {
                    return Optional.of(itemStack);
                }
            }
        }
        return Optional.empty();
    }

    private Optional<PriceInfoContainer> getOrderPriceInfo(class_9290 lore) {
        if (lore.comp_2400().size() <= LORE_LINE_PRICE) return Optional.empty();

        String priceLine = lore.comp_2400().get(LORE_LINE_PRICE).getString();
        Matcher matcher = PRICE_PATTERN.matcher(priceLine);

        if (matcher.find()) {
            try {
                double orderPrice = Double.parseDouble(matcher.group(1).replace(",", ""));
                return Optional.of(new PriceInfoContainer(orderPrice, PriceInfoContainer.PriceType.INSTASELL));
            } catch (NumberFormatException e) {
                Util.notifyError("Error while trying to parse order price in Flip Helper", e);
            }
        }
        return Optional.empty();
    }

    private Optional<Integer> getVolumeUnclaimed(class_9290 lore) {
        if (lore.comp_2400().size() <= LORE_LINE_VOLUME) return Optional.empty();

        String volumeLine = lore.comp_2400().get(LORE_LINE_VOLUME).getString();
        Matcher matcher = VOLUME_PATTERN.matcher(volumeLine);

        if (matcher.find()) {
            try {
                return Optional.of(Integer.parseInt(matcher.group(1).replace(",", "")));
            } catch (NumberFormatException e) {
                Util.notifyError("Error while trying to parse order volume in Flip Helper", e);
            }
        }
        return Optional.empty();
    }

    private Optional<BazaarOrder> matchToUserOrder(class_9290 lore) {
        Optional<PriceInfoContainer> priceInfoOpt = getOrderPriceInfo(lore);
        Optional<Integer> orderVolumeFilledOpt = getVolumeUnclaimed(lore);

        if (priceInfoOpt.isPresent() && orderVolumeFilledOpt.isPresent()) {
            PriceInfoContainer priceInfoContainer = priceInfoOpt.get();
            OrderInfoContainer tempOrder = new OrderInfoContainer(null, orderVolumeFilledOpt.get(), priceInfoContainer.getPricePerItem(), priceInfoContainer.getPriceType(), null);
            return tempOrder.findOrderInList(BUConfig.get().userOrders);
        }
        return Optional.empty();
    }

    private static boolean inCorrectScreen(){
        ScreenInfo screenInfo = ScreenInfo.getCurrentScreenInfo();
        return screenInfo.inMenu(ScreenInfo.BazaarMenuType.FLIP_GUI) && !inCancelOrderScreen();
    }

    private static boolean inCancelOrderScreen() {
        if (!(class_310.method_1551().field_1755 instanceof class_476 inventory)) {
            return false;
        }

        try {
            return cantBeFlippedLineIsPresent(inventory, FLIP_ORDER_SLOT);
        } catch (Exception ex) {
            Util.notifyError("Error while checking if in cancel screen", ex);
            return false;
        }
    }

    private static boolean cantBeFlippedLineIsPresent(class_476 inventory, int slot){
        class_1799 itemStack = inventory.method_17577().method_7629().method_5438(slot);
        if (itemStack.method_7960()) {
            return false;
        }

        class_2561 customName = itemStack.method_58694(class_9334.field_49631);
        if (customName == null || !customName.getString().contains(FLIP_ORDER_IDENTIFIER)) {
            return false;
        }

        class_9290 lore = itemStack.method_58694(class_9334.field_49632);
        if (lore == null || lore.comp_2400().isEmpty()) {
            return false;
        }
        return Util.findComponentWith(lore.comp_2400(), CANNOT_CANCEL_IDENTIFIER) != null;
    }

    //an item to cancel the order being present means that the order has not been filled or is otherwise not ready to be flipped
//    private static boolean isCancelItem(GenericContainerScreen inventory, int slot) {
//        ItemStack itemStack = inventory.getScreenHandler().getInventory().getStack(slot);
//        if (itemStack.isEmpty()) {
//            return false;
//        }
//
//        Text customName = itemStack.get(DataComponentTypes.CUSTOM_NAME);
//        if (customName != null && customName.getString().contains(CANCEL_ORDER_IDENTIFIER)) {
//            return true;
//        }
//
//        LoreComponent lore = itemStack.get(DataComponentTypes.LORE);
//        if (lore != null) {
//            return lore.lines().stream()
//                    .noneMatch(line -> line.getString().contains(CANNOT_CANCEL_IDENTIFIER));
//        }
//
//        return false;
//    }

    public Option<Boolean> createOption() {
        return super.createOption("Flip Helper",
                "Button in flip order menu to undercut market prices for items.",
                this::isEnabled,
                this::setEnabled);
    }

    @Override
    public void subscribe() {
        EVENT_BUS.subscribe(this);
    }
}

