package com.github.mkram17.bazaarutils.utils;

import com.github.mkram17.bazaarutils.config.BUConfig;
import com.github.mkram17.bazaarutils.data.BazaarData;
import com.github.mkram17.bazaarutils.features.CustomOrder;
import com.github.mkram17.bazaarutils.features.OutbidOrderHandler;
import com.github.mkram17.bazaarutils.features.restrictsell.RestrictSell;
import com.github.mkram17.bazaarutils.features.restrictsell.RestrictSellControl;
import com.github.mkram17.bazaarutils.misc.orderinfo.BazaarOrder;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.DoubleArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.tree.CommandNode;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandManager;
import net.fabricmc.fabric.api.client.command.v2.FabricClientCommandSource;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_407;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import static com.github.mkram17.bazaarutils.config.BUConfig.openGUI;

public class BUCommands {

    private static final List<LiteralArgumentBuilder<FabricClientCommandSource>> developerCommands = List.of(
            ClientCommandManager.literal("remove")
                    .then(ClientCommandManager.argument("index", IntegerArgumentType.integer())
                            .executes(BUCommands::executeRemove)
                    ),
            ClientCommandManager.literal("info")
                    .then((ClientCommandManager.argument("index", IntegerArgumentType.integer())
                                    .executes(BUCommands::executeInfo)
                            )
                    ),
            ClientCommandManager.literal("outdated")
                    .executes((source) -> {
                        for (BazaarOrder item : OutbidOrderHandler.getOutbidOrders()) {
                            PlayerActionUtil.notifyAll(item.getName() + " is outdated. Market Price: " + item.getInstaSellPrice() + " Order Price: " + item.getPricePerItem());
                        }
                        return 1;
                    }),
            ClientCommandManager.literal("convertname")
                    .then((ClientCommandManager.argument("item name", StringArgumentType.string())
                            .executes((context) -> {
                                String name = StringArgumentType.getString(context, "item name")
                                        .replaceAll("_", " ");
                                String productID = BazaarData.findProductId(name);

                                if(productID == null){
                                    PlayerActionUtil.notifyAll("Could not find product ID for " + name);
                                } else {
                                    PlayerActionUtil.notifyAll(name + ": " + productID);
                                }
                                return 1;
                            })
                    )
                    ),
            ClientCommandManager.literal("list")
                    .executes(context -> {
                                PlayerActionUtil.notifyAll(BazaarOrder.getVariables(BazaarOrder::getName).toString());
                                return 1;
                            }
                    )
    );
    private static final LiteralArgumentBuilder<FabricClientCommandSource> bazaarutils = ClientCommandManager.literal("bazaarutils").executes(context -> {
        openGUI();
        return 1;
    });

    public static void register(CommandDispatcher<FabricClientCommandSource> dispatcher) {




        bazaarutils.then(ClientCommandManager.literal("help")
                .executes((context) -> {
                            PlayerActionUtil.notifyAll(Util.HELP_MESSAGE);
                            return 1;
                        }
                ));

        bazaarutils.then(ClientCommandManager.literal("tax")
                .then((ClientCommandManager.argument("amount", DoubleArgumentType.doubleArg(1, 1.25))
                                .executes((context) -> {
                                    BUConfig.get().bzTax = DoubleArgumentType.getDouble(context, "amount") / 100;
                                    return 1;
                                })
                        )
                )
        );
        bazaarutils.then(ClientCommandManager.literal("discord")
                .executes((context) -> {
                    class_310 client = class_310.method_1551();
                    client.method_63588(() -> {
                        context.getSource().getClient().method_1507(new class_407((confirmed) -> {
                            if (confirmed) {
                                    try {
                                        net.minecraft.class_156.method_668().method_673(new URI(Util.DISCORD_LINK));
                                    } catch (URISyntaxException e) {
                                        throw new RuntimeException(e);
                                    }
                            }
                            class_310.method_1551().method_1507(null);
                        }, Util.DISCORD_LINK, true));
                    });
                    return 1;
                })
        );
        bazaarutils.then(ClientCommandManager.literal("developer")
                .executes((context) -> {
                    BUConfig.get().developerMode = !BUConfig.get().developerMode;
                    Util.scheduleConfigSave();
                    //TODO register new commands so they can be used without restarting
                    PlayerActionUtil.notifyAll(BUConfig.get().developerMode ? "Developer mode enabled." : "Developer mode disabled. Restart for all changes to take effect");

                    if(BUConfig.get().developerMode) {
                        registerDeveloperCommands(dispatcher);
                    }
                    return 1;
                })
        );

        bazaarutils.then(ClientCommandManager.literal("customorder")
                .then(ClientCommandManager.literal("add")
                        .then(ClientCommandManager.argument("order amount", IntegerArgumentType.integer(1, 71680))
                                .then(ClientCommandManager.argument("slot number", IntegerArgumentType.integer(1, 36))
                                        .executes(context -> {
                                            int orderAmount = IntegerArgumentType.getInteger(context, "order amount");
                                            int slotNumber = IntegerArgumentType.getInteger(context, "slot number");

                                            if (orderAmount < 1 || orderAmount > 71679) {
                                                context.getSource().sendError(class_2561.method_43470("Order amount must be 1-71,679"));
                                                return 0;
                                            }

                                            if (slotNumber < 1 || slotNumber > 36) {
                                                context.getSource().sendError(class_2561.method_43470("Slot number must be 1-36"));
                                                return 0;
                                            }
                                            CustomOrder orderToAdd = new CustomOrder(
                                                    true,
                                                    orderAmount,
                                                    slotNumber - 1,
                                                    CustomOrder.getNextColoredPane()
                                            );

                                            BUConfig.get().customOrders.add(orderToAdd);
                                            PlayerActionUtil.notifyAll("Added order for " + orderToAdd.getOrderAmount() + " in slot number " + slotNumber);
                                            Util.scheduleConfigSave();
                                            return 1;
                                        })
                                )
                        )
                )
        );
        bazaarutils.then(ClientCommandManager.literal("customorder")
                .then(ClientCommandManager.literal("remove")
                        .then(ClientCommandManager.argument("order number", IntegerArgumentType.integer(1))
                                .executes(context -> {
                                    int orderNum = IntegerArgumentType.getInteger(context, "order number") - 1;
                                    if (BUConfig.get().customOrders.size() < orderNum) {
                                        PlayerActionUtil.notifyAll("There is no Custom Order #" + orderNum + ". The Custom Order # is based on the order they are displayed in the config.");
                                        return 0;
                                    }
                                    CustomOrder customOrder = BUConfig.get().customOrders.get(orderNum);
                                    if (customOrder.getOrderAmount() != 71680) {
                                        PlayerActionUtil.notifyAll("Removed Custom Order for " + BUConfig.get().customOrders.get(orderNum).getOrderAmount());
                                        BUConfig.get().customOrders.get(orderNum).remove();
                                    } else {
                                        PlayerActionUtil.notifyAll("Cannot remove Max Buy Order.");
                                        return 0;
                                    }
                                    Util.scheduleConfigSave();
                                    return 1;
                                })
                        )
                )
        );
        bazaarutils.then(ClientCommandManager.literal("rule")
                .then(ClientCommandManager.literal("add")
                        // Volume branch
                        .then(ClientCommandManager.literal("volume")
                                .then(ClientCommandManager.argument("limit", DoubleArgumentType.doubleArg(0.1))
                                        .executes(context -> {
                                            double limit = DoubleArgumentType.getDouble(context, "limit");
                                            BUConfig.get().restrictSell.addRule(RestrictSell.restrictBy.VOLUME, limit);
                                            Util.scheduleConfigSave();
                                            PlayerActionUtil.notifyAll("Added rule: VOLUME" + ": " + limit);
                                            return 1;
                                        })
                                )
                        )
                        // Price branch
                        .then(ClientCommandManager.literal("price")
                                .then(ClientCommandManager.argument("limit", DoubleArgumentType.doubleArg(0.1))
                                        .executes(context -> {
                                            double limit = DoubleArgumentType.getDouble(context, "limit");
                                            BUConfig.get().restrictSell.addRule(RestrictSell.restrictBy.PRICE, limit);
                                            Util.scheduleConfigSave();
                                            PlayerActionUtil.notifyAll("Added rule: PRICE" + ": " + limit);
                                            return 1;
                                        })
                                )
                        )
                        // Name branch
                        .then(ClientCommandManager.literal("name")
                                .then(ClientCommandManager.argument("itemName", StringArgumentType.string())
                                        .executes(context -> {
                                            String name = StringArgumentType.getString(context, "itemName");
                                            BUConfig.get().restrictSell.addRule(RestrictSell.restrictBy.NAME, name);
                                            Util.scheduleConfigSave();
                                            PlayerActionUtil.notifyAll("Added rule: NAME" + ": " + name);
                                            return 1;
                                        })
                                )
                        )
                )
        );
        bazaarutils.then(ClientCommandManager.literal("rule")
                .then(ClientCommandManager.literal("remove")
                        .then(ClientCommandManager.argument("rule number", IntegerArgumentType.integer(1))
                                .executes(context -> {
                                    int restrictNum = IntegerArgumentType.getInteger(context, "rule number") - 1;
                                    RestrictSellControl rule = BUConfig.get().restrictSell.getControls().get(restrictNum);
                                    if (rule == null)
                                        context.getSource().sendError(class_2561.method_43470("Invalid rule number. Check the order in /bu"));
                                    if (rule.getRule() != null) {
                                        PlayerActionUtil.notifyAll(rule.getRule() == RestrictSell.restrictBy.NAME ? "Removed rule: NAME: " + rule.getName() : "Removed rule: " + rule.getRule() + ": " + rule.getAmount());
                                    }
                                    BUConfig.get().restrictSell.getControls().remove(restrictNum);
                                    Util.scheduleConfigSave();
                                    return 1;
                                })
                        )
                )
        );
        bazaarutils.then(ClientCommandManager.literal("updateresources")
                .executes(context -> {
                    PlayerActionUtil.notifyAll("Checking for resource updates...");
                    ResourceManager.checkForUpdates(true);
                    return 1;
                })
        );


        if (BUConfig.get().developerMode) {
            registerDeveloperCommands(dispatcher);
        }
//
        CommandNode<FabricClientCommandSource> bazaarutilsNode = dispatcher.register(bazaarutils);
        dispatcher.register(
                ClientCommandManager.literal("bu")
                        .executes(context -> {
                            // Forward execution to the main command's handler
                            return bazaarutils.getCommand().run(context);
                        })
                        .redirect(bazaarutilsNode) // Inherit subcommands
        );
    }

    private static void registerDeveloperCommands(CommandDispatcher<FabricClientCommandSource> dispatcher){
        for(LiteralArgumentBuilder<FabricClientCommandSource> command : developerCommands) {
            bazaarutils.then(command);
        }
        dispatcher.register(bazaarutils);
    }

    private static int executeRemove(CommandContext<FabricClientCommandSource> context) {
        int index = IntegerArgumentType.getInteger(context, "index");
        BazaarOrder bazaarOrder = BUConfig.get().userOrders.get(index);
        bazaarOrder.removeFromWatchedItems();
        PlayerActionUtil.notifyAll("Removed " + bazaarOrder, Util.notificationTypes.COMMAND);
        return 1;
    }

    private static int executeInfo(CommandContext<FabricClientCommandSource> context) {
        int index = IntegerArgumentType.getInteger(context, "index");
        PlayerActionUtil.notifyAll(BUConfig.get().userOrders.get(index).toString());
        return 1;
    }
}