package com.github.mkram17.bazaarutils.misc.orderinfo;

import com.github.mkram17.bazaarutils.config.BUConfig;
import com.github.mkram17.bazaarutils.events.BazaarDataUpdateEvent;
import com.github.mkram17.bazaarutils.events.UserOrdersChangeEvent;
import com.github.mkram17.bazaarutils.features.OutbidOrderHandler;
import com.github.mkram17.bazaarutils.utils.PlayerActionUtil;
import com.github.mkram17.bazaarutils.utils.ScreenInfo;
import com.github.mkram17.bazaarutils.utils.SoundUtil;
import com.github.mkram17.bazaarutils.utils.Util;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import meteordevelopment.orbit.EventHandler;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_5250;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static com.github.mkram17.bazaarutils.BazaarUtils.EVENT_BUS;


//TODO figure out how to handle rounding with price
//TODO sometimes a BazaarOrder is created that is for the same order as one in userOrders, but not the same object. This causes problems sometimes, and should be fixed.
//TODO use last viewed item in bazaar to help with finding accurate price instead of just chat message
//An eventful OrderInfoContainer, also tracks info only needed for actual user orders
@Slf4j
public class BazaarOrder extends OrderInfoContainer {

    public static final int OUTBID_ORDER_NOTIFICATIONS = 3; // number of notifications to send when an order becomes outdated

    @Getter @Setter
    private Statuses fillStatus; //only used to determine if order is set or filled, not outdated, competitive, or matched
    @Getter @Setter
    private int amountClaimed = 0;
    @Getter
    private int amountFilled = 0;


    public BazaarOrder(String name, Integer volume, Double pricePerItem, PriceType priceType) {
        this(name, volume, pricePerItem, priceType, null);
    }

    public BazaarOrder(String name, Integer volume, Double pricePerItem, PriceType priceType, ItemInfo itemInfo) {
        super(name, volume, pricePerItem, priceType, itemInfo);
        this.fillStatus = Statuses.SET;
        startTracking();
    }

    private void startTracking(){
        handleOutbidStatusChange();
        subscribe();
        scheduleHealthCheck();
        updateMarketPrice();
    }

    @EventHandler
    private void onDataUpdate(BazaarDataUpdateEvent e){
        updateMarketPrice();
        handleOutbidStatusChange();
    }

    @EventHandler
    private void onUserOrderChange(UserOrdersChangeEvent e) {
        if(e.getChangeType() == UserOrdersChangeEvent.ChangeTypes.REMOVE || e.getOrder() != this)
            return;
        updateMarketPrice();
        handleOutbidStatusChange();
    }

    private void handleOutbidStatusChange(){
        Optional<Statuses> outbidOptional = findOutbidStatus();
        if(outbidOptional.isEmpty()) return;

        Statuses newStatus = outbidOptional.get();
        if(outbidStatus != newStatus){
            outbidStatus = newStatus;
            onOutbid(newStatus == Statuses.OUTBID);
        }
    }

    private void onOutbid(boolean isOutbid){
        boolean shouldNotifyUser = BUConfig.get().outbidOrderHandler.isNotifyOutbid();
        boolean shouldPlayNotificationSound = BUConfig.get().outbidOrderHandler.isNotificationSound();
        boolean shouldAutoOpenBazaar = BUConfig.get().outbidOrderHandler.isAutoOpenEnabled();

        if(!shouldNotifyUser || !BUConfig.get().userOrders.contains(this)) return;
        if(this.getFillStatus() == OrderInfoContainer.Statuses.FILLED) return;

        class_5250 message;
        if (isOutbid) {
            message = OutbidOrderHandler.getOutbidMessage(this);
            if (BUConfig.get().developerMode) {
                message.method_27693(class_2561.method_43470(". Market Price: " + this.getMarketPrice(this.getPriceType()) + " Order Price: " + this.getPricePerItem()));
            }
            if(shouldAutoOpenBazaar){
                openBazaar();
            }
            class_310 client = class_310.method_1551();
            var player = client.field_1724;
            if (shouldPlayNotificationSound && player != null) {
                SoundUtil.notifyMultipleTimes(OUTBID_ORDER_NOTIFICATIONS);
            }
            Util.tickExecuteLater(2, () -> PlayerActionUtil.notifyChatCommand(message, "managebazaarorders"));
        } else if(this.getOutbidStatus() == OrderInfoContainer.Statuses.COMPETITIVE) {
            message = OutbidOrderHandler.getCompetitiveMessage(this);
            Util.tickExecuteLater(2, () -> PlayerActionUtil.notifyAll(message));
        } else {
            message = OutbidOrderHandler.getMatchedMessage(this);
            Util.tickExecuteLater(2, () -> PlayerActionUtil.notifyAll(message));
        }
    }

    public void openBazaar() {
        ScreenInfo screenInfo = ScreenInfo.getCurrentScreenInfo();
        if(screenInfo.inBazaar() )
            return;
        CompletableFuture.runAsync(() ->{
            for(int i = 3; i >= 1; i--) {
                try {
                    if(i == 3)
                        PlayerActionUtil.notifyAll("Opening bazaar in 3");
                    else
                        PlayerActionUtil.notifyAll(String.valueOf(i));
                    Thread.sleep(1000);
                } catch (InterruptedException ex) {
                    throw new RuntimeException(ex);
                }
            }

            PlayerActionUtil.runCommand("managebazaarorders");
        });
    }


    public int getIndex() {
        return BUConfig.get().userOrders.indexOf(this);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append(name).append("[").append(getIndex()).append("]");
        if (amountClaimed != 0) {
            sb.append(", amount claimed: ").append(amountClaimed);
        }
        sb.append(", type: ").append(getPriceType().getString());
        if (fillStatus == Statuses.FILLED) {
            sb.append(", status: ").append(fillStatus);
        }
        sb.append(")");
        return sb.toString();
    }

    public void flipItem(double newPrice) {
        flipPrices(newPrice);
        updateMarketPrice();
        this.amountFilled = 0;
        this.fillStatus = Statuses.SET;
        EVENT_BUS.post(new UserOrdersChangeEvent(UserOrdersChangeEvent.ChangeTypes.UPDATE, this));
    }

    public double getFlipPrice() {
        updateMarketPrice();
        Double marketPrice = getMarketPrice(priceType);
        Double marketOppositePrice = getMarketPrice(priceType.getOpposite());
        if (marketPrice == 0) {
            return 0;
        }
        if (getPriceType() == PriceType.INSTABUY) {
            return (marketOppositePrice + .1);
        } else {
            return (marketOppositePrice - .1);
        }
    }

    public void setAmountFilled(int amountFilled) {
        this.amountFilled = amountFilled;
        if (this.amountFilled >= volume) {
            setFilled();
        } else {
            this.fillStatus = Statuses.SET;
        }
    }

    public void setFilled() {
        amountFilled = volume;
        fillStatus = Statuses.FILLED;
    }

    public void removeFromWatchedItems() {
        if (!BUConfig.get().userOrders.remove(this)) {
            PlayerActionUtil.notifyAll("Error removing " + name + " from watched items. Item couldn't be found.");
        }
        EVENT_BUS.post(new UserOrdersChangeEvent(UserOrdersChangeEvent.ChangeTypes.REMOVE, this));
        Util.scheduleConfigSave();
    }
}