/*
 * Decompiled with CFR 0.152.
 */
package com.nyakotech.hibernateforge;

import com.nyakotech.hibernateforge.Constants;
import com.nyakotech.hibernateforge.Hibernation;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.minecraft.core.BlockPos;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerChunkCache;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.border.WorldBorder;
import net.minecraft.world.level.entity.EntityTypeTest;
import net.minecraft.world.phys.AABB;

public class MemoryManager {
    private static final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    private static boolean memoryOptimizationActive = false;
    private static long lastGCTime = 0L;
    private static final long GC_INTERVAL_MS = 30000L;
    private static final double MEMORY_THRESHOLD = 0.8;

    public static void startMemoryOptimization(MinecraftServer server) {
        if (memoryOptimizationActive) {
            return;
        }
        memoryOptimizationActive = true;
        Constants.LOG.info("Starting Memory Optimization for hibernation");
        scheduler.scheduleAtFixedRate(() -> {
            if (Hibernation.isHibernating()) {
                MemoryManager.performMemoryCleanup(server);
            }
        }, 10L, 30L, TimeUnit.SECONDS);
        MemoryManager.performGarbageCollection();
    }

    public static void stopMemoryOptimization() {
        if (!memoryOptimizationActive) {
            return;
        }
        memoryOptimizationActive = false;
        Constants.LOG.info("Stopping Memory Optimization for hibernation");
    }

    private static void performMemoryCleanup(MinecraftServer server) {
        try {
            MemoryManager.unloadUnnecessaryChunks(server);
            MemoryManager.cleanupInactiveEntities(server);
            if (MemoryManager.shouldForceGC()) {
                MemoryManager.performGarbageCollection();
            }
            MemoryManager.logMemoryUsage();
        }
        catch (Exception e) {
            Constants.LOG.error("Error during memory cleanup: ", (Throwable)e);
        }
    }

    private static void unloadUnnecessaryChunks(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            ServerChunkCache chunkSource = level.getChunkSource();
            BlockPos spawnPos = level.getSharedSpawnPos();
            int spawnX = spawnPos.getX() >> 4;
            int spawnZ = spawnPos.getZ() >> 4;
            CompletableFuture.runAsync(() -> {
                try {
                    chunkSource.save(true);
                }
                catch (RuntimeException e) {
                    Constants.LOG.warn("Error saving chunks: ", (Throwable)e);
                }
            });
        }
    }

    private static void cleanupInactiveEntities(MinecraftServer server) {
        for (ServerLevel level : server.getAllLevels()) {
            List entities = level.getEntities(EntityTypeTest.forClass(Entity.class), MemoryManager.getWorldBorderBoundingBox(level), entity -> true);
            List<Entity> entitiesToRemove = entities.stream().filter(MemoryManager::canEntityBeRemovedDuringHibernation).toList();
            for (Entity entity2 : entitiesToRemove) {
                entity2.discard();
            }
            if (entitiesToRemove.isEmpty()) continue;
            Constants.LOG.debug("{} inactive entities removed from the level {}", (Object)entitiesToRemove.size(), (Object)level.dimension().location());
        }
    }

    private static AABB getWorldBorderBoundingBox(ServerLevel level) {
        WorldBorder border = level.getWorldBorder();
        double centerX = border.getCenterX();
        double centerZ = border.getCenterZ();
        double size = border.getSize();
        double halfSize = size / 2.0;
        return new AABB(centerX - halfSize, Double.NEGATIVE_INFINITY, centerZ - halfSize, centerX + halfSize, Double.POSITIVE_INFINITY, centerZ + halfSize);
    }

    private static boolean canEntityBeRemovedDuringHibernation(Entity entity) {
        String entityType = entity.getType().toString();
        return entityType.contains("experience_orb") || entityType.contains("firework") || entityType.contains("arrow") || entity.tickCount > 6000 && entityType.contains("item");
    }

    private static boolean shouldForceGC() {
        long maxMemory;
        long freeMemory;
        Runtime runtime = Runtime.getRuntime();
        long totalMemory = runtime.totalMemory();
        long usedMemory = totalMemory - (freeMemory = runtime.freeMemory());
        double memoryUsagePercent = (double)usedMemory / (double)(maxMemory = runtime.maxMemory());
        return memoryUsagePercent > 0.8 || System.currentTimeMillis() - lastGCTime > 30000L;
    }

    private static void performGarbageCollection() {
        long beforeGC = MemoryManager.getUsedMemoryMB();
        long startTime = System.currentTimeMillis();
        System.gc();
        try {
            Thread.sleep(100L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        System.gc();
        long afterGC = MemoryManager.getUsedMemoryMB();
        long gcTime = System.currentTimeMillis() - startTime;
        long memoryFreed = beforeGC - afterGC;
        lastGCTime = System.currentTimeMillis();
        Constants.LOG.info("GC executed: {}MB freed in {}ms (Before: {}MB, After: {}MB)", new Object[]{memoryFreed, gcTime, beforeGC, afterGC});
    }

    private static long getUsedMemoryMB() {
        Runtime runtime = Runtime.getRuntime();
        return (runtime.totalMemory() - runtime.freeMemory()) / 0x100000L;
    }

    private static void logMemoryUsage() {
        Runtime runtime = Runtime.getRuntime();
        long totalMemory = runtime.totalMemory() / 0x100000L;
        long freeMemory = runtime.freeMemory() / 0x100000L;
        long usedMemory = totalMemory - freeMemory;
        long maxMemory = runtime.maxMemory() / 0x100000L;
        double usagePercent = (double)usedMemory / (double)maxMemory * 100.0;
        double formattedUsagePercent = (double)Math.round(usagePercent * 10.0) / 10.0;
        Constants.LOG.info("Memory: {}MB used / {}MB max ({}%) \u2014 Free: {}MB", new Object[]{usedMemory, maxMemory, formattedUsagePercent, freeMemory});
    }

    public static void saveImportantData(MinecraftServer server) {
        Constants.LOG.info("Saving important data before hibernation...");
        try {
            server.saveEverything(false, false, true);
            server.getPlayerList().saveAll();
            Constants.LOG.info("Data saved successfully");
        }
        catch (Exception e) {
            Constants.LOG.error("Error while saving important data: ", (Throwable)e);
        }
    }

    public static void shutdown() {
        memoryOptimizationActive = false;
        scheduler.shutdown();
        try {
            if (!scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}

