package io.wispforest.owo.ui.hud;

import com.mojang.blaze3d.platform.Window;
import io.wispforest.owo.Owo;
import io.wispforest.owo.client.OwoClient;
import io.wispforest.owo.ui.container.FlowLayout;
import io.wispforest.owo.ui.core.Component;
import io.wispforest.owo.ui.core.OwoUIAdapter;
import io.wispforest.owo.ui.event.ClientRenderCallback;
import io.wispforest.owo.ui.event.WindowResizeCallback;
import net.minecraft.client.Minecraft;
import net.minecraft.resources.ResourceLocation;
import net.neoforged.bus.api.Event;
import net.neoforged.neoforge.client.gui.GuiLayer;
import net.neoforged.neoforge.client.gui.GuiLayerManager;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * A utility for displaying owo-ui components on the
 * in-game HUD - rendered during {@link HudRenderCallback}
 */
public class Hud {

    static @Nullable OwoUIAdapter<FlowLayout> adapter = null;
    static boolean suppress = false;

    private static final Map<ResourceLocation, Component> activeComponents = new HashMap<>();
    private static final List<Consumer<FlowLayout>> pendingActions = new ArrayList<>();

    /**
     * Add a new component to be rendered on the in-game HUD.
     * The root container used by the HUD does not support layout
     * positioning - the component supplied by {@code component}
     * must be explicitly positioned via either {@link io.wispforest.owo.ui.core.Positioning#absolute(int, int)}
     * or {@link io.wispforest.owo.ui.core.Positioning#relative(int, int)}
     *
     * @param id        An ID uniquely describing this HUD component
     * @param component A function creating the component
     *                  when the HUD is first rendered
     */
    public static void add(ResourceLocation id, Supplier<Component> component) {
        pendingActions.add(flowLayout -> {
            var instance = component.get();

            flowLayout.child(instance);
            activeComponents.put(id, instance);
        });
    }

    /**
     * Remove the HUD component described by the given ID
     *
     * @param id The ID of the HUD component to remove
     */
    public static void remove(ResourceLocation id) {
        pendingActions.add(flowLayout -> {
            var component = activeComponents.get(id);
            if (component == null) return;

            flowLayout.removeChild(component);
            activeComponents.remove(id);
        });
    }

    /**
     * Get the HUD component described by the given ID
     *
     * @param id The ID of the HUD component to query
     * @return The relevant HUD component, or {@code null} if there is none
     */
    public static @Nullable Component getComponent(ResourceLocation id) {
        return activeComponents.get(id);
    }

    /**
     * @return {@code true} if there is an active HUD component described by {@code id}
     */
    public static boolean hasComponent(ResourceLocation id) {
        return activeComponents.containsKey(id);
    }

    private static void initializeAdapter() {
        var window = Minecraft.getInstance().getWindow();
        adapter = OwoUIAdapter.createWithoutScreen(
            0, 0, window.getGuiScaledWidth(), window.getGuiScaledHeight(), HudContainer::new
        );

        adapter.inflateAndMount();
    }

    static {
        WindowResizeCallback.EVENT.register((client, window) -> {
            if (adapter == null) return;
            adapter.moveAndResize(0, 0, window.getGuiScaledWidth(), window.getGuiScaledHeight());
        });

        ClientRenderCallback.BEFORE.register(client -> {
            if (client.level == null) return;
            if (!pendingActions.isEmpty()) {
                if (adapter == null) initializeAdapter();

                pendingActions.forEach(action -> action.accept(adapter.rootComponent));
                pendingActions.clear();
            }
        });
    }

    public static final GuiLayerManager.NamedLayer HUD_LAYER = new GuiLayerManager.NamedLayer(ResourceLocation.fromNamespaceAndPath("owo", "owo_ui_hud"), (context, tickCounter) -> {
        if (adapter == null || suppress || Minecraft.getInstance().options.hideGui) return;
        adapter.render(context, -69, -69, tickCounter.getGameTimeDeltaPartialTick(false));
    });
}
