package io.wispforest.owo.moddata;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import io.wispforest.owo.Owo;
import net.minecraft.resources.ResourceLocation;
import net.neoforged.fml.ModList;
import net.neoforged.fml.loading.FMLLoader;
import net.neoforged.neoforgespi.language.ModFileScanData;
import org.apache.commons.io.FilenameUtils;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

/**
 * Contains the logic to load JSON from all other mods' data directories
 * when {@link #load(ModDataConsumer)} is called. This should ideally be done
 * one and in a {@link net.fabricmc.api.ModInitializer}
 */
public final class ModDataLoader {

    private static final Gson GSON = new Gson();

    private static final Path DATA_PATH = FMLLoader.getCurrent().getGameDir().resolve("moddata");

    private ModDataLoader() {}

    /**
     * Loads the data the {@code consumer} requests
     *
     * @param consumer The consumer to load data for
     */
    public static void load(ModDataConsumer consumer) {
        Map<ResourceLocation, JsonObject> foundFiles = new HashMap<>();

        ModList.get().getMods().forEach(modInfo -> {
            for (var rootPath : modInfo.getOwningFile().getFile().getContents().getContentRoots()) {
                final var targetPath = rootPath.resolve(String.format("data/%s/%s", modInfo.getModId(), consumer.getDataSubdirectory()));
                tryLoadFilesFrom(foundFiles, modInfo.getModId(), targetPath);
            }
        });

        try {
            Files.createDirectories(DATA_PATH);

            try (var stream = Files.list(DATA_PATH)) {
                stream.forEach(nsPath -> {
                    if (!Files.isDirectory(nsPath)) return;

                    var namespace = nsPath.getFileName().toString();
                    var targetPath = nsPath.resolve(consumer.getDataSubdirectory());
                    if (!Files.exists(targetPath)) return;

                    tryLoadFilesFrom(foundFiles, namespace, targetPath);
                });
            }
        } catch (IOException e) {
            Owo.LOGGER.error("### Unable to traverse global data tree ++ Stacktrace below ###", e);
        }

        foundFiles.forEach(consumer::acceptParsedFile);
    }

    private static void tryLoadFilesFrom(Map<ResourceLocation, JsonObject> foundFiles, String namespace, Path targetPath) {
        try {
            if (!Files.exists(targetPath)) return;

            try (var stream = Files.walk(targetPath)) {
                stream.forEach(path -> {
                    if (!path.toString().endsWith(".json")) return;
                    try {
                        final InputStreamReader tabData = new InputStreamReader(Files.newInputStream(path));

                        foundFiles.put(ResourceLocation.fromNamespaceAndPath(namespace, FilenameUtils.removeExtension(targetPath.relativize(path).toString())), GSON.fromJson(tabData, JsonObject.class));
                    } catch (IOException e) {
                        Owo.LOGGER.warn("### Unable to open data file {} ++ Stacktrace below ###", path, e);
                    }
                });
            }

        } catch (IOException e) {
            Owo.LOGGER.error("### Unable to traverse data tree {} ++ Stacktrace below ###", targetPath, e);
        }
    }
}
