package io.wispforest.owo.network.neoforge;

import io.wispforest.owo.Owo;
import io.wispforest.owo.client.screens.ScreenInternals;
import io.wispforest.owo.network.OwoHandshake;
import io.wispforest.owo.network.OwoNetChannel.MessagePayload;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.network.protocol.PacketFlow;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload.Type;
import net.minecraft.world.entity.player.Player;
import net.neoforged.neoforge.network.event.RegisterPayloadHandlersEvent;
import net.neoforged.neoforge.network.handling.IPayloadHandler;
import net.neoforged.neoforge.network.registration.PayloadRegistrar;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;

import static io.wispforest.owo.network.OwoNetChannel.MessagePayload;

public class NeoOwoNetworking {

    private static final Map<CustomPacketPayload.Type<?>, PayloadCodec<?>> PAYLOAD_ID_TO_CLIENT_CODEC = new ConcurrentHashMap<>();
    private static final Map<CustomPacketPayload.Type<?>, PayloadHandler<?>> PAYLOAD_ID_TO_CLIENT_HANDLER = new ConcurrentHashMap<>();

    private static final Map<CustomPacketPayload.Type<MessagePayload>, SidedPacketCodec<MessagePayload>> PAYLOAD_ID_TO_SIDED_CODEC = new ConcurrentHashMap<>();

    private static final Map<CustomPacketPayload.Type<MessagePayload>, PayloadHandler<MessagePayload>> PAYLOAD_ID_TO_SERVER_PAYLOAD_HANDLER = new ConcurrentHashMap<>();
    private static final Map<CustomPacketPayload.Type<MessagePayload>, PayloadHandler<MessagePayload>> PAYLOAD_ID_TO_CLIENT_PAYLOAD_HANDLER = new ConcurrentHashMap<>();

    private static boolean hasNetworkRegistrationTakenPlace = false;

    public static void onNetworkRegister(RegisterPayloadHandlersEvent event) {
        var registrar = event.registrar("1.0.0");

        for (var entry : PAYLOAD_ID_TO_CLIENT_CODEC.entrySet()) {
            var id = entry.getKey();

            var handler = Objects.requireNonNull(PAYLOAD_ID_TO_CLIENT_HANDLER.get(id), "Unable to register the given client play packet due to missing the needed handler! Id: " + id);

            entry.getValue().registerPlayPayload(registrar, handler);
        }

        OwoHandshake.register(registrar);
        ScreenInternals.init(registrar);

        /* | \/ Optional Section Below \/ | */

        registrar = registrar.optional();

        for (var entry : PAYLOAD_ID_TO_SIDED_CODEC.entrySet()) {
            var id = entry.getKey();

            if (!PAYLOAD_ID_TO_SERVER_PAYLOAD_HANDLER.containsKey(id)) {
                throw new IllegalStateException("Unable to get the required Payload Handler as its missing for the Server! Id: " + id);
            } else if (!PAYLOAD_ID_TO_CLIENT_PAYLOAD_HANDLER.containsKey(id)) {
                throw new IllegalStateException("Unable to get the required Payload Handler as its missing for the Client! Id: " + id);
            }

            IPayloadHandler<MessagePayload> biDiHandler = (arg, iPayloadContext) -> {
                iPayloadContext.enqueueWork(() -> {
                    var player = iPayloadContext.player();

                    var handler = (!player.level().isClientSide())
                        ? PAYLOAD_ID_TO_SERVER_PAYLOAD_HANDLER.get(id)
                        : PAYLOAD_ID_TO_CLIENT_PAYLOAD_HANDLER.get(id);

                    handler.accept(arg, iPayloadContext.player());
                });
            };

            registrar.playBidirectional(id, entry.getValue(), biDiHandler, biDiHandler);
        }

        hasNetworkRegistrationTakenPlace = true;
    }

    public static void registerMessageCodecs(CustomPacketPayload.Type<MessagePayload> id, StreamCodec<FriendlyByteBuf, MessagePayload> serverCodec, StreamCodec<FriendlyByteBuf, MessagePayload> clientCodec) {
        if (PAYLOAD_ID_TO_SIDED_CODEC.containsKey(id)) {
            throw new IllegalStateException("Unable to register the given codec as such already exists within codec map! Id: " + id);
        }

        if (hasNetworkRegistrationTakenPlace) {
            throw new IllegalStateException("Unable to register the given codec as network registration has already occurred! Id: " + id);
        }

        PAYLOAD_ID_TO_SIDED_CODEC.put(id, new SidedPacketCodec<>(serverCodec, clientCodec));
    }

    public static <T extends CustomPacketPayload> void registerClientCodec(CustomPacketPayload.Type<T> id, StreamCodec<FriendlyByteBuf, T> codec) {
        if (PAYLOAD_ID_TO_CLIENT_CODEC.containsKey(id)) {
            throw new IllegalStateException("Unable to register the given codec as such already exists within codec map! Id: " + id);
        }

        if (hasNetworkRegistrationTakenPlace) {
            throw new IllegalStateException("Unable to register the given codec as network registration has already occurred! Id: " + id);
        }

        PAYLOAD_ID_TO_CLIENT_CODEC.put(id, new PayloadCodec<T>(id, codec, Optional.of(PacketFlow.CLIENTBOUND)));
    }

    public static <T extends CustomPacketPayload> void registerClientPayload(CustomPacketPayload.Type<T> id, PayloadHandler<T> payloadHandler) {
        if (PAYLOAD_ID_TO_CLIENT_HANDLER.containsKey(id)) {
            throw new IllegalStateException("Unable to register the given payload handler as such already exists within payload handler map! Id: " + id);
        }

        if (hasNetworkRegistrationTakenPlace) {
            throw new IllegalStateException("Unable to register the given payload handler as network registration has already occurred! Id: " + id);
        }

        PAYLOAD_ID_TO_CLIENT_HANDLER.put(id, payloadHandler);
    }

    public record PayloadCodec<T extends CustomPacketPayload>(CustomPacketPayload.Type<T> id, StreamCodec<FriendlyByteBuf, T> codec, Optional<PacketFlow> possibleSide) {
        public void registerPlayPayload(PayloadRegistrar registrar, PayloadHandler<?> handler) {
            var castedHandler = (PayloadHandler<T>) handler;

            possibleSide.ifPresentOrElse(side -> {
                if (side.isClientbound()) {
                    registrar.playToClient(id, codec, (arg, context) -> context.enqueueWork(() -> castedHandler.accept(arg, context.player())));
                } else {
                    registrar.playToServer(id, codec, (arg, context) -> context.enqueueWork(() -> castedHandler.accept(arg, context.player())));
                }
            }, () -> {
                IPayloadHandler<T> biDiHandler = (arg, context) -> context.enqueueWork(() -> castedHandler.accept(arg, context.player()));

                registrar.playBidirectional(id, codec, biDiHandler, biDiHandler);
            });
        }
    }

    public static void registerServerMessageHandler(CustomPacketPayload.Type<MessagePayload> id, PayloadHandler<MessagePayload> handler) {
        if (PAYLOAD_ID_TO_SERVER_PAYLOAD_HANDLER.containsKey(id)) {
            throw new IllegalStateException("Unable to register the given server handler as such already exists within handler map! Id: " + id);
        }

        if (hasNetworkRegistrationTakenPlace) {
            throw new IllegalStateException("Unable to register the given payload handler as network registration has already occurred! Id: " + id);
        }

        PAYLOAD_ID_TO_SERVER_PAYLOAD_HANDLER.put(id, handler);
    }

    public static void registerClientMessageHandler(CustomPacketPayload.Type<MessagePayload> id, PayloadHandler<MessagePayload> handler) {
        if (PAYLOAD_ID_TO_CLIENT_PAYLOAD_HANDLER.containsKey(id)) {
            throw new IllegalStateException("Unable to register the given client handler as such already exists within handler map! Id: " + id);
        }

        if (hasNetworkRegistrationTakenPlace) {
            throw new IllegalStateException("Unable to register the given payload handler as network registration has already occurred! Id: " + id);
        }

        PAYLOAD_ID_TO_CLIENT_PAYLOAD_HANDLER.put(id, handler);
    }

    public interface PayloadHandler<T extends CustomPacketPayload> extends BiConsumer<T, Player> {
        static <P extends CustomPacketPayload> PayloadHandler<P> empty() { return (payload, player) -> {}; }

        @Override void accept(T payload, Player player);
    }
}
