package dev.dfonline.flint.data;

import com.mojang.authlib.properties.Property;
import com.mojang.authlib.properties.PropertyMap;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_9282;
import net.minecraft.class_9288;
import net.minecraft.class_9290;
import net.minecraft.class_9296;
import net.minecraft.class_9334;

/**
 * Represents an item, with configurable data, lore, name, and more.
 */
public class DFItem {

    private final class_1799 item;
    private ItemData data;

    /**
     * Creates a new DFItem from an ItemStack.
     *
     * @param item The item to create the DFItem from.
     */
    public DFItem(class_1799 item) {
        this.item = item;
        this.data = new ItemData(item);
    }

    /**
     * Creates a new DFItem from an ItemStack.
     *
     * @param item The item to create the DFItem from.
     * @return The new DFItem.
     */
    public static DFItem of(class_1799 item) {
        return new DFItem(item);
    }

    /**
     * Gets the item's data.
     *
     * @return The item's data.
     */
    public ItemData getItemData() {
        return this.data;
    }

    /**
     * Sets the item's data.
     *
     * @param itemData The new data to set.
     */
    public void setItemData(ItemData itemData) {
        this.data = itemData;
    }

    /**
     * Edits the item's data with a consumer, creates the data if it doesn't exist.
     * <br>
     * Example:
     * <pre>{@code
     * item.editData(data -> {
     *    data.setStringValue("key", "value");
     * });
     * }</pre>
     *
     * @param consumer The consumer to edit the data with.
     */
    public void editData(Consumer<ItemData> consumer) {
        if (!this.data.hasCustomData()) {
            this.data = ItemData.getEmpty();
        }
        consumer.accept(this.data);
    }

    /**
     * Delegates to {@link ItemData#getHypercubeStringValue(String)}.
     *
     * @param key The key to get, without the hypercube: prefix.
     * @return The value of the key, or an empty string if it doesn't exist.
     */
    public String getHypercubeStringValue(String key) {
        ItemData itemData = this.getItemData();
        if (itemData == null) {
            return "";
        }
        return itemData.getHypercubeStringValue(key);
    }

    /**
     * Delegates to {@link ItemData#hasHypercubeKey(String)}.
     *
     * @param key The key to check, without the hypercube: prefix.
     * @return Whether the key exists.
     */
    public boolean hasHypercubeKey(String key) {
        ItemData itemData = this.getItemData();
        if (itemData == null) {
            return false;
        }
        return itemData.hasHypercubeKey(key);
    }

    /**
     * Converts the DFItem back into an ItemStack.
     *
     * @return The ItemStack.
     */
    public class_1799 getItemStack() {
        if (this.data != null) {
            this.item.method_57379(class_9334.field_49628, this.getItemData().toComponent());
        }
        return this.item;
    }

    /**
     * Delegates to {@link ItemData#getPublicBukkitValues()}.
     *
     * @return The PublicBukkitValues.
     */
    public PublicBukkitValues getPublicBukkitValues() {
        return this.getItemData().getPublicBukkitValues();
    }

    /**
     * Gets the lore of the item.
     *
     * @return The lore of the item.
     */
    public List<class_2561> getLore() {
        class_9290 loreComponent = this.item.method_58694(class_9334.field_49632);
        if (loreComponent == null) {
            return List.of();
        }
        return loreComponent.comp_2400();
    }

    /**
     * Sets the lore of the item.
     *
     * @param lore The new lore to set.
     */
    public void setLore(List<class_2561> lore) {
        this.item.method_57379(class_9334.field_49632, new class_9290(lore));
    }

    /**
     * Gets the name of the item.
     *
     * @return The name of the item.
     */
    public class_2561 getName() {
        return this.item.method_7964();
    }

    /**
     * Sets the name of the item.
     *
     * @param name The new name to set.
     */
    public void setName(class_2561 name) {
        this.item.method_57379(class_9334.field_49631, name);
    }

//    Since this is currently unused we can keep it that way for now
//    TODO: Reimplement
//    /**
//     * Hides additional information about the item, such as additional tooltip, jukebox playable, fireworks, and attribute modifiers.
//     */
//    public void hideFlags() {
//        this.item.set(DataComponentTypes.HIDE_ADDITIONAL_TOOLTIP, Unit.INSTANCE);
//        this.item.remove(DataComponentTypes.JUKEBOX_PLAYABLE);
//        this.item.remove(DataComponentTypes.FIREWORKS);
//        this.item.remove(DataComponentTypes.ATTRIBUTE_MODIFIERS);
//    }

    /**
     * Sets the dye color of the item.
     *
     * @param color The new dye color to set.
     */
    public void setDyeColor(int color) {
        this.item.method_57379(class_9334.field_49644, new class_9282(color));
    }

//    Since this is currently unused we can keep it that way for now
//    TODO: Reimplement (custom model data works differently now so this needs a full rework)
//    /**
//     * Sets the custom model data of the item.
//     *
//     * @param modelData The new custom model data to set.
//     */
//    public void setCustomModelData(int modelData) {
//        this.item.set(DataComponentTypes.CUSTOM_MODEL_DATA, new CustomModelDataComponent());
//    }

    /**
     * Sets the profile of the item, for use with player heads.
     *
     * @param uuid      The UUID of the player.
     * @param value     The value of the profile.
     * @param signature The signature of the profile.
     */
    public void setProfile(UUID uuid, String value, String signature) {
        PropertyMap map = new PropertyMap();
        map.put("textures", new Property("textures", value, signature));
        this.item.method_57379(class_9334.field_49617, new class_9296(Optional.empty(), Optional.ofNullable(uuid), map));
    }

    /**
     * Removes the item's data.
     */
    public void removeItemData() {
        this.item.method_57381(class_9334.field_49628);
        this.data = null;
    }


    // This method doesn't fit the theme of entire item data abstraction, but its use case is very specific.

    /**
     * Gets the container of the item.
     *
     * @return The container of the item.
     */
    @Nullable
    public class_9288 getContainer() {
        return this.item.method_58694(class_9334.field_49622);
    }

}
