package com.zurrtum.create.content.contraptions.actors.psi;

import com.zurrtum.create.api.behaviour.movement.MovementBehaviour;
import com.zurrtum.create.catnip.animation.LerpedFloat;
import com.zurrtum.create.catnip.animation.LerpedFloat.Chaser;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.content.contraptions.behaviour.MovementContext;
import com.zurrtum.create.content.trains.entity.CarriageContraption;

import java.util.Optional;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.Vec3;

public class PortableStorageInterfaceMovement extends MovementBehaviour {

    public static final String _workingPos_ = "WorkingPos";
    static final String _clientPrevPos_ = "ClientPrevPos";

    @Override
    public Vec3 getActiveAreaOffset(MovementContext context) {
        return Vec3.atLowerCornerOf(context.state.getValue(PortableStorageInterfaceBlock.FACING).getUnitVec3i()).scale(1.85f);
    }

    @Override
    public boolean disableBlockEntityRendering() {
        return true;
    }

    @Override
    public void visitNewPosition(MovementContext context, BlockPos pos) {
        boolean onCarriage = context.contraption instanceof CarriageContraption;
        if (onCarriage && context.motion.length() > 1 / 4f)
            return;
        if (!findInterface(context, pos))
            context.data.remove(_workingPos_);
    }

    @Override
    public void tick(MovementContext context) {
        if (context.world.isClientSide())
            getAnimation(context).tickChaser();

        boolean onCarriage = context.contraption instanceof CarriageContraption;
        if (onCarriage && context.motion.length() > 1 / 4f)
            return;

        if (context.world.isClientSide()) {
            BlockPos pos = BlockPos.containing(context.position);
            if (!findInterface(context, pos))
                reset(context);
            return;
        }

        if (!context.data.contains(_workingPos_)) {
            if (context.stall)
                cancelStall(context);
            return;
        }

        BlockPos pos = context.data.read(_workingPos_, BlockPos.CODEC).orElseThrow();
        Vec3 target = VecHelper.getCenterOf(pos);

        if (!context.stall && !onCarriage && context.position.closerThan(target, target.distanceTo(context.position.add(context.motion))))
            context.stall = true;

        Optional<Direction> currentFacingIfValid = getCurrentFacingIfValid(context);
        if (currentFacingIfValid.isEmpty()) {
            reset(context);
            return;
        }

        PortableStorageInterfaceBlockEntity stationaryInterface = getStationaryInterfaceAt(
            context.world,
            pos,
            context.state,
            currentFacingIfValid.get()
        );
        if (stationaryInterface == null) {
            reset(context);
            return;
        }

        if (stationaryInterface.connectedEntity == null)
            stationaryInterface.startTransferringTo(context.contraption, stationaryInterface.distance);

        boolean timerBelow = stationaryInterface.transferTimer <= PortableStorageInterfaceBlockEntity.ANIMATION;
        stationaryInterface.keepAlive = 2;
        if (context.stall && timerBelow) {
            context.stall = false;
        }
    }

    protected boolean findInterface(MovementContext context, BlockPos pos) {
        if (context.contraption instanceof CarriageContraption cc && !cc.notInPortal())
            return false;
        Optional<Direction> currentFacingIfValid = getCurrentFacingIfValid(context);
        if (!currentFacingIfValid.isPresent())
            return false;

        Direction currentFacing = currentFacingIfValid.get();
        PortableStorageInterfaceBlockEntity psi = findStationaryInterface(context.world, pos, context.state, currentFacing);

        if (psi == null)
            return false;
        if (psi.isPowered())
            return false;

        context.data.store(_workingPos_, BlockPos.CODEC, psi.getBlockPos());
        if (!context.world.isClientSide()) {
            Vec3 diff = VecHelper.getCenterOf(psi.getBlockPos()).subtract(context.position);
            diff = VecHelper.project(diff, Vec3.atLowerCornerOf(currentFacing.getUnitVec3i()));
            float distance = (float) (diff.length() + 1.85f - 1);
            psi.startTransferringTo(context.contraption, distance);
        } else {
            context.data.store(_clientPrevPos_, BlockPos.CODEC, pos);
            if (context.contraption instanceof CarriageContraption || context.contraption.entity.isStalled() || context.motion.lengthSqr() == 0)
                getAnimation(context).chase(psi.getConnectionDistance() / 2, 0.25f, Chaser.LINEAR);
        }

        return true;
    }

    @Override
    public void stopMoving(MovementContext context) {
        //		reset(context);
    }

    @Override
    public void cancelStall(MovementContext context) {
        reset(context);
    }

    public void reset(MovementContext context) {
        context.data.remove(_clientPrevPos_);
        context.data.remove(_workingPos_);
        context.stall = false;
        getAnimation(context).chase(0, 0.25f, Chaser.LINEAR);
    }

    private PortableStorageInterfaceBlockEntity findStationaryInterface(Level world, BlockPos pos, BlockState state, Direction facing) {
        for (int i = 0; i < 2; i++) {
            PortableStorageInterfaceBlockEntity interfaceAt = getStationaryInterfaceAt(world, pos.relative(facing, i), state, facing);
            if (interfaceAt == null)
                continue;
            return interfaceAt;
        }
        return null;
    }

    private PortableStorageInterfaceBlockEntity getStationaryInterfaceAt(Level world, BlockPos pos, BlockState state, Direction facing) {
        BlockEntity blockEntity = world.getBlockEntity(pos);
        if (!(blockEntity instanceof PortableStorageInterfaceBlockEntity psi))
            return null;
        BlockState blockState = world.getBlockState(pos);
        if (blockState.getBlock() != state.getBlock())
            return null;
        if (blockState.getValue(PortableStorageInterfaceBlock.FACING) != facing.getOpposite())
            return null;
        if (psi.isPowered())
            return null;
        return psi;
    }

    private Optional<Direction> getCurrentFacingIfValid(MovementContext context) {
        Vec3 directionVec = Vec3.atLowerCornerOf(context.state.getValue(PortableStorageInterfaceBlock.FACING).getUnitVec3i());
        directionVec = context.rotation.apply(directionVec);
        Direction facingFromVector = Direction.getApproximateNearest(directionVec.x, directionVec.y, directionVec.z);
        if (directionVec.distanceTo(Vec3.atLowerCornerOf(facingFromVector.getUnitVec3i())) > 1 / 2f)
            return Optional.empty();
        return Optional.of(facingFromVector);
    }

    public static LerpedFloat getAnimation(MovementContext context) {
        if (!(context.temporaryData instanceof LerpedFloat lf)) {
            LerpedFloat nlf = LerpedFloat.linear();
            context.temporaryData = nlf;
            return nlf;
        }
        return lf;
    }

}
