package com.zurrtum.create.content.decoration.placard;

import com.mojang.serialization.MapCodec;
import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.AllShapes;
import com.zurrtum.create.AllSoundEvents;
import com.zurrtum.create.api.schematic.requirement.SpecialBlockItemRequirement;
import com.zurrtum.create.content.equipment.wrench.IWrenchable;
import com.zurrtum.create.content.logistics.filter.FilterItem;
import com.zurrtum.create.content.logistics.filter.FilterItemStack;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement;
import com.zurrtum.create.content.schematics.requirement.ItemRequirement.ItemUseType;
import com.zurrtum.create.foundation.block.IBE;
import com.zurrtum.create.foundation.block.ProperWaterloggedBlock;
import org.jetbrains.annotations.NotNull;

import java.util.List;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.util.RandomSource;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.context.BlockPlaceContext;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelReader;
import net.minecraft.world.level.ScheduledTickAccess;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.FaceAttachedHorizontalDirectionalBlock;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition;
import net.minecraft.world.level.block.state.properties.AttachFace;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.level.material.FluidState;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.shapes.CollisionContext;
import net.minecraft.world.phys.shapes.VoxelShape;

public class PlacardBlock extends FaceAttachedHorizontalDirectionalBlock implements ProperWaterloggedBlock, IBE<PlacardBlockEntity>, SpecialBlockItemRequirement, IWrenchable {

    public static final BooleanProperty POWERED = BlockStateProperties.POWERED;

    public static final MapCodec<PlacardBlock> CODEC = simpleCodec(PlacardBlock::new);

    public PlacardBlock(Properties p_53182_) {
        super(p_53182_);
        registerDefaultState(defaultBlockState().setValue(WATERLOGGED, false).setValue(POWERED, false));
    }

    @Override
    protected void createBlockStateDefinition(StateDefinition.Builder<Block, BlockState> pBuilder) {
        super.createBlockStateDefinition(pBuilder.add(FACE, FACING, WATERLOGGED, POWERED));
    }

    @Override
    public boolean canSurvive(BlockState pState, LevelReader pLevel, BlockPos pPos) {
        return canAttachLenient(pLevel, pPos, getConnectedDirection(pState).getOpposite());
    }

    public static boolean canAttachLenient(LevelReader pReader, BlockPos pPos, Direction pDirection) {
        BlockPos blockpos = pPos.relative(pDirection);
        return !pReader.getBlockState(blockpos).getCollisionShape(pReader, blockpos).isEmpty();
    }

    @Override
    public BlockState getStateForPlacement(BlockPlaceContext pContext) {
        BlockState stateForPlacement = super.getStateForPlacement(pContext);
        if (stateForPlacement == null)
            return null;
        if (stateForPlacement.getValue(FACE) == AttachFace.FLOOR)
            stateForPlacement = stateForPlacement.setValue(FACING, stateForPlacement.getValue(FACING).getOpposite());
        return withWater(stateForPlacement, pContext);
    }

    @Override
    public boolean isSignalSource(BlockState pState) {
        return true;
    }

    @Override
    public int getSignal(BlockState pBlockState, BlockGetter pBlockAccess, BlockPos pPos, Direction pSide) {
        return pBlockState.getValue(POWERED) ? 15 : 0;
    }

    @Override
    public int getDirectSignal(BlockState pBlockState, BlockGetter pBlockAccess, BlockPos pPos, Direction pSide) {
        return pBlockState.getValue(POWERED) && getConnectedDirection(pBlockState) == pSide ? 15 : 0;
    }

    @Override
    public VoxelShape getShape(BlockState pState, BlockGetter pLevel, BlockPos pPos, CollisionContext pContext) {
        return AllShapes.PLACARD.get(getConnectedDirection(pState));
    }

    @Override
    public BlockState updateShape(
        BlockState pState,
        LevelReader pLevel,
        ScheduledTickAccess tickView,
        BlockPos pCurrentPos,
        Direction pFacing,
        BlockPos pFacingPos,
        BlockState pFacingState,
        RandomSource random
    ) {
        updateWater(pLevel, tickView, pState, pCurrentPos);
        return super.updateShape(pState, pLevel, tickView, pCurrentPos, pFacing, pFacingPos, pFacingState, random);
    }

    @Override
    public FluidState getFluidState(BlockState pState) {
        return fluidState(pState);
    }

    @Override
    protected InteractionResult useItemOn(
        ItemStack stack,
        BlockState state,
        Level level,
        BlockPos pos,
        Player player,
        InteractionHand hand,
        BlockHitResult hitResult
    ) {
        if (player.isShiftKeyDown())
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        if (level.isClientSide())
            return InteractionResult.SUCCESS;

        ItemStack inHand = player.getItemInHand(hand);
        return onBlockEntityUseItemOn(
            level, pos, pte -> {
                ItemStack inBlock = pte.getHeldItem();

                if (!player.mayBuild() || inHand.isEmpty() || !inBlock.isEmpty()) {
                    if (inBlock.isEmpty())
                        return InteractionResult.FAIL;
                    if (inHand.isEmpty())
                        return InteractionResult.FAIL;
                    if (state.getValue(POWERED))
                        return InteractionResult.FAIL;

                    boolean test = inBlock.getItem() instanceof FilterItem ? FilterItemStack.of(inBlock)
                        .test(level, inHand) : ItemStack.isSameItemSameComponents(inHand, inBlock);
                    if (!test) {
                        AllSoundEvents.DENY.play(level, null, pos, 1, 1);
                        return InteractionResult.SUCCESS;
                    }

                    AllSoundEvents.CONFIRM.play(level, null, pos, 1, 1);
                    level.setBlock(pos, state.setValue(POWERED, true), Block.UPDATE_ALL);
                    updateNeighbours(state, level, pos);
                    pte.poweredTicks = 19;
                    pte.notifyUpdate();
                    return InteractionResult.SUCCESS;
                }

                level.playSound(null, pos, SoundEvents.ITEM_FRAME_ADD_ITEM, SoundSource.BLOCKS, 1, 1);
                pte.setHeldItem(inHand.copyWithCount(1));

                if (!player.isCreative()) {
                    inHand.shrink(1);
                    if (inHand.isEmpty())
                        player.setItemInHand(hand, ItemStack.EMPTY);
                }

                return InteractionResult.SUCCESS;
            }
        );
    }

    public static Direction connectedDirection(BlockState state) {
        return getConnectedDirection(state);
    }

    @Override
    public void affectNeighborsAfterRemoval(BlockState pState, ServerLevel pLevel, BlockPos pPos, boolean pIsMoving) {
        if (!pIsMoving && pState.getValue(POWERED))
            updateNeighbours(pState, pLevel, pPos);
    }

    public static void updateNeighbours(BlockState pState, Level pLevel, BlockPos pPos) {
        pLevel.updateNeighborsAt(pPos, pState.getBlock(), null);
        pLevel.updateNeighborsAt(pPos.relative(getConnectedDirection(pState).getOpposite()), pState.getBlock(), null);
    }

    @Override
    public void attack(BlockState pState, Level pLevel, BlockPos pPos, Player pPlayer) {
        if (pLevel.isClientSide())
            return;
        withBlockEntityDo(
            pLevel, pPos, pte -> {
                ItemStack heldItem = pte.getHeldItem();
                if (heldItem.isEmpty())
                    return;
                pPlayer.getInventory().placeItemBackInInventory(heldItem);
                pLevel.playSound(null, pPos, SoundEvents.ITEM_FRAME_REMOVE_ITEM, SoundSource.BLOCKS, 1, 1);
                pte.setHeldItem(ItemStack.EMPTY);
            }
        );
    }

    @Override
    public ItemRequirement getRequiredItems(BlockState state, BlockEntity be) {
        ItemStack placardStack = AllItems.PLACARD.getDefaultInstance();
        if (be instanceof PlacardBlockEntity pbe) {
            ItemStack heldItem = pbe.getHeldItem();
            if (!heldItem.isEmpty()) {
                return new ItemRequirement(List.of(
                    new ItemRequirement.StackRequirement(placardStack, ItemUseType.CONSUME),
                    new ItemRequirement.StrictNbtStackRequirement(heldItem, ItemUseType.CONSUME)
                ));
            }
        }
        return new ItemRequirement(ItemUseType.CONSUME, placardStack);
    }

    @Override
    public Class<PlacardBlockEntity> getBlockEntityClass() {
        return PlacardBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends PlacardBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.PLACARD;
    }

    @Override
    protected @NotNull MapCodec<? extends FaceAttachedHorizontalDirectionalBlock> codec() {
        return CODEC;
    }

}
