package com.zurrtum.create.content.kinetics.transmission.sequencer;

import com.zurrtum.create.AllBlockEntityTypes;
import com.zurrtum.create.AllClientHandle;
import com.zurrtum.create.AllItems;
import com.zurrtum.create.api.contraption.transformable.TransformableBlock;
import com.zurrtum.create.content.contraptions.StructureTransform;
import com.zurrtum.create.content.kinetics.base.HorizontalAxisKineticBlock;
import com.zurrtum.create.content.kinetics.base.KineticBlock;
import com.zurrtum.create.content.kinetics.base.RotatedPillarKineticBlock;
import com.zurrtum.create.foundation.block.IBE;
import com.zurrtum.create.foundation.block.WeakPowerControlBlock;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.Direction.Axis;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.RandomSource;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResult;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.BlockItem;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.context.BlockPlaceContext;
import net.minecraft.world.item.context.UseOnContext;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelReader;
import net.minecraft.world.level.SignalGetter;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.entity.BlockEntityType;
import net.minecraft.world.level.block.state.BlockBehaviour;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.StateDefinition.Builder;
import net.minecraft.world.level.block.state.properties.BooleanProperty;
import net.minecraft.world.level.block.state.properties.IntegerProperty;
import net.minecraft.world.level.redstone.Orientation;
import net.minecraft.world.phys.BlockHitResult;
import org.jetbrains.annotations.Nullable;

public class SequencedGearshiftBlock extends HorizontalAxisKineticBlock implements IBE<SequencedGearshiftBlockEntity>, TransformableBlock, WeakPowerControlBlock {
    public static final BooleanProperty VERTICAL = BooleanProperty.create("vertical");
    public static final IntegerProperty STATE = IntegerProperty.create("state", 0, 5);

    public SequencedGearshiftBlock(BlockBehaviour.Properties properties) {
        super(properties);
    }

    @Override
    protected void createBlockStateDefinition(Builder<Block, BlockState> builder) {
        super.createBlockStateDefinition(builder.add(STATE, VERTICAL));
    }

    @Override
    public void neighborChanged(BlockState state, Level level, BlockPos pos, Block block, @Nullable Orientation wireOrientation, boolean isMoving) {
        if (level.isClientSide())
            return;
        if (!level.getBlockTicks().willTickThisTick(pos, this))
            level.scheduleTick(pos, this, 1);
    }

    @Override
    public void tick(BlockState state, ServerLevel worldIn, BlockPos pos, RandomSource r) {
        boolean previouslyPowered = state.getValue(STATE) != 0;
        boolean isPowered = worldIn.hasNeighborSignal(pos);
        withBlockEntityDo(worldIn, pos, sgte -> sgte.onRedstoneUpdate(isPowered, previouslyPowered));
    }

    @Override
    protected boolean areStatesKineticallyEquivalent(BlockState oldState, BlockState newState) {
        return false;
    }

    @Override
    public boolean hasShaftTowards(LevelReader world, BlockPos pos, BlockState state, Direction face) {
        if (state.getValue(VERTICAL))
            return face.getAxis().isVertical();
        return super.hasShaftTowards(world, pos, state, face);
    }

    @Override
    protected InteractionResult useItemOn(
        ItemStack stack,
        BlockState state,
        Level level,
        BlockPos pos,
        Player player,
        InteractionHand hand,
        BlockHitResult hitResult
    ) {
        if (stack.is(AllItems.WRENCH))
            return InteractionResult.TRY_WITH_EMPTY_HAND;
        if (stack.getItem() instanceof BlockItem blockItem) {
            if (blockItem.getBlock() instanceof KineticBlock && hasShaftTowards(level, pos, state, hitResult.getDirection()))
                return InteractionResult.TRY_WITH_EMPTY_HAND;
        }

        if (level.isClientSide()) {
            withBlockEntityDo(level, pos, be -> AllClientHandle.INSTANCE.openSequencedGearshiftScreen(be));
        }
        return InteractionResult.SUCCESS;
    }

    @Override
    public BlockState getStateForPlacement(BlockPlaceContext context) {
        Axis preferredAxis = RotatedPillarKineticBlock.getPreferredAxis(context);
        if (preferredAxis != null && (context.getPlayer() == null || !context.getPlayer().isShiftKeyDown()))
            return withAxis(preferredAxis, context);
        return withAxis(context.getNearestLookingDirection().getAxis(), context);
    }

    @Override
    public InteractionResult onWrenched(BlockState state, UseOnContext context) {
        BlockState newState = state;

        if (context.getClickedFace().getAxis() != Axis.Y)
            if (newState.getValue(HORIZONTAL_AXIS) != context.getClickedFace().getAxis())
                newState = newState.cycle(VERTICAL);

        return super.onWrenched(newState, context);
    }

    @Override
    public boolean shouldCheckWeakPower(BlockState state, SignalGetter level, BlockPos pos, Direction side) {
        return false;
    }

    private BlockState withAxis(Axis axis, BlockPlaceContext context) {
        BlockState state = defaultBlockState().setValue(VERTICAL, axis.isVertical());
        if (axis.isVertical())
            return state.setValue(HORIZONTAL_AXIS, context.getHorizontalDirection().getAxis());
        return state.setValue(HORIZONTAL_AXIS, axis);
    }

    @Override
    public Axis getRotationAxis(BlockState state) {
        if (state.getValue(VERTICAL))
            return Axis.Y;
        return super.getRotationAxis(state);
    }

    @Override
    public Class<SequencedGearshiftBlockEntity> getBlockEntityClass() {
        return SequencedGearshiftBlockEntity.class;
    }

    @Override
    public BlockEntityType<? extends SequencedGearshiftBlockEntity> getBlockEntityType() {
        return AllBlockEntityTypes.SEQUENCED_GEARSHIFT;
    }

    @Override
    public boolean hasAnalogOutputSignal(BlockState p_149740_1_) {
        return true;
    }

    @Override
    public int getAnalogOutputSignal(BlockState state, Level world, BlockPos pos, Direction direction) {
        return state.getValue(STATE);
    }

    @Override
    public BlockState transform(BlockState state, StructureTransform transform) {
        if (transform.mirror != null) {
            state = mirror(state, transform.mirror);
        }

        if (transform.rotationAxis == Axis.Y) {
            return rotate(state, transform.rotation);
        }

        if (transform.rotation.ordinal() % 2 == 1) {
            if (transform.rotationAxis != state.getValue(HORIZONTAL_AXIS)) {
                return state.cycle(VERTICAL);
            } else if (state.getValue(VERTICAL)) {
                return state.cycle(VERTICAL).cycle(HORIZONTAL_AXIS);
            }
        }
        return state;
    }

}