package com.zurrtum.create.content.fluids.transfer;

import com.zurrtum.create.AllFluidTags;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import com.zurrtum.create.foundation.fluid.FluidHelper;
import com.zurrtum.create.infrastructure.config.AllConfigs;
import com.zurrtum.create.infrastructure.fluids.FluidStack;
import com.zurrtum.create.infrastructure.packet.s2c.FluidSplashPacket;
import java.io.Serial;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_243;
import net.minecraft.class_3218;
import net.minecraft.class_3341;
import net.minecraft.class_3414;
import net.minecraft.class_3419;
import net.minecraft.class_3610;
import net.minecraft.class_3611;

public abstract class FluidManipulationBehaviour extends BlockEntityBehaviour<SmartBlockEntity> {

    public record BlockPosEntry(class_2338 pos, int distance) {
    }

    public static class ChunkNotLoadedException extends Exception {
        @Serial
        private static final long serialVersionUID = 1L;
    }

    class_3341 affectedArea;
    class_2338 rootPos;
    boolean infinite;
    protected boolean counterpartActed;

    // Search
    static final int searchedPerTick = 1024;
    static final int validationTimerMin = 160;
    List<BlockPosEntry> frontier;
    Set<class_2338> visited;

    int revalidateIn;

    public FluidManipulationBehaviour(SmartBlockEntity be) {
        super(be);
        setValidationTimer();
        infinite = false;
        visited = new HashSet<>();
        frontier = new ArrayList<>();
    }

    public boolean isInfinite() {
        return infinite;
    }

    public void counterpartActed() {
        counterpartActed = true;
    }

    protected int validationTimer() {
        int maxBlocks = maxBlocks();
        // Allow enough time for the server's infinite block threshold to be reached
        return maxBlocks < 0 ? validationTimerMin : Math.max(validationTimerMin, maxBlocks / searchedPerTick + 1);
    }

    protected int setValidationTimer() {
        return revalidateIn = validationTimer();
    }

    protected int setLongValidationTimer() {
        return revalidateIn = validationTimer() * 2;
    }

    protected int maxRange() {
        return AllConfigs.server().fluids.hosePulleyRange.get();
    }

    protected int maxBlocks() {
        return AllConfigs.server().fluids.hosePulleyBlockThreshold.get();
    }

    protected boolean fillInfinite() {
        return AllConfigs.server().fluids.fillInfinite.get();
    }

    public void reset() {
        if (affectedArea != null)
            scheduleUpdatesInAffectedArea();
        affectedArea = null;
        setValidationTimer();
        frontier.clear();
        visited.clear();
        infinite = false;
    }

    @Override
    public void destroy() {
        reset();
        super.destroy();
    }

    protected void scheduleUpdatesInAffectedArea() {
        class_1937 world = getWorld();
        class_2338.method_20437(
            new class_2338(affectedArea.method_35415() - 1, affectedArea.method_35416() - 1, affectedArea.method_35417() - 1),
            new class_2338(affectedArea.method_35418() + 1, affectedArea.method_35419() + 1, affectedArea.method_35420() + 1)
        ).forEach(pos -> {
            class_3610 nextFluidState = world.method_8316(pos);
            if (nextFluidState.method_15769())
                return;
            world.method_64312(pos, nextFluidState.method_15772(), world.method_8409().method_43048(5));
        });
    }

    protected int comparePositions(BlockPosEntry e1, BlockPosEntry e2) {
        class_243 centerOfRoot = VecHelper.getCenterOf(rootPos);
        class_2338 pos2 = e2.pos;
        class_2338 pos1 = e1.pos;
        if (pos1.method_10264() != pos2.method_10264())
            return Integer.compare(pos2.method_10264(), pos1.method_10264());
        int compareDistance = Integer.compare(e2.distance, e1.distance);
        if (compareDistance != 0)
            return compareDistance;
        return Double.compare(
            VecHelper.getCenterOf(pos2).method_1025(centerOfRoot),
            VecHelper.getCenterOf(pos1).method_1025(centerOfRoot)
        );
    }

    protected class_3611 search(
        class_3611 fluid,
        List<BlockPosEntry> frontier,
        Set<class_2338> visited,
        BiConsumer<class_2338, Integer> add,
        boolean searchDownward
    ) throws ChunkNotLoadedException {
        class_1937 world = getWorld();
        int maxBlocks = maxBlocks();
        int maxRange = maxRange();
        int maxRangeSq = maxRange * maxRange;
        int i;

        for (i = 0; i < searchedPerTick && !frontier.isEmpty() && (visited.size() <= maxBlocks || !canDrainInfinitely(fluid)); i++) {
            BlockPosEntry entry = frontier.remove(0);
            class_2338 currentPos = entry.pos;
            if (visited.contains(currentPos))
                continue;
            visited.add(currentPos);

            if (!world.method_8477(currentPos))
                throw new ChunkNotLoadedException();

            class_3610 fluidState = world.method_8316(currentPos);
            if (fluidState.method_15769())
                continue;

            class_3611 currentFluid = FluidHelper.convertToStill(fluidState.method_15772());
            if (fluid == null)
                fluid = currentFluid;
            if (!currentFluid.method_15780(fluid))
                continue;

            add.accept(currentPos, entry.distance);

            for (class_2350 side : Iterate.directions) {
                if (!searchDownward && side == class_2350.field_11033)
                    continue;

                class_2338 offsetPos = currentPos.method_10093(side);
                if (!world.method_8477(offsetPos))
                    throw new ChunkNotLoadedException();
                if (visited.contains(offsetPos))
                    continue;
                if (offsetPos.method_10262(rootPos) > maxRangeSq)
                    continue;

                class_3610 nextFluidState = world.method_8316(offsetPos);
                if (nextFluidState.method_15769())
                    continue;
                class_3611 nextFluid = nextFluidState.method_15772();
                if (nextFluid == FluidHelper.convertToFlowing(nextFluid) && side == class_2350.field_11036 && !VecHelper.onSameAxis(rootPos, offsetPos, class_2351.field_11052))
                    continue;

                frontier.add(new BlockPosEntry(offsetPos, entry.distance + 1));
            }
        }

        return fluid;
    }

    protected void playEffect(class_1937 world, class_2338 pos, class_3611 fluid, boolean fillSound) {
        if (fluid == null)
            return;

        class_2338 splooshPos = pos == null ? blockEntity.method_11016() : pos;
        FluidStack stack = new FluidStack(fluid, 1);

        class_3414 soundevent = fillSound ? FluidHelper.getFillSound(stack) : FluidHelper.getEmptySound(stack);
        world.method_8396(null, splooshPos, soundevent, class_3419.field_15245, 0.3F, 1.0F);
        if (world instanceof class_3218 serverLevel) {
            serverLevel.method_8503().method_3760().method_14605(
                null,
                splooshPos.method_10263(),
                splooshPos.method_10264(),
                splooshPos.method_10260(),
                10,
                serverLevel.method_27983(),
                new FluidSplashPacket(splooshPos, stack.getFluid())
            );
        }
    }

    protected boolean canDrainInfinitely(class_3611 fluid) {
        if (fluid == null)
            return false;
        return maxBlocks() != -1 && AllConfigs.server().fluids.bottomlessFluidMode.get().test(fluid);
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        if (infinite)
            view.method_71472("Infinite", true);
        if (rootPos != null)
            view.method_71468("LastPos", class_2338.field_25064, rootPos);
        if (affectedArea != null) {
            view.method_71468("AffectedAreaFrom", class_2338.field_25064, new class_2338(affectedArea.method_35415(), affectedArea.method_35416(), affectedArea.method_35417()));
            view.method_71468("AffectedAreaTo", class_2338.field_25064, new class_2338(affectedArea.method_35418(), affectedArea.method_35419(), affectedArea.method_35420()));
        }
        super.write(view, clientPacket);
    }

    @Override
    public void read(class_11368 view, boolean clientPacket) {
        infinite = view.method_71433("Infinite", false);
        rootPos = view.method_71426("LastPos", class_2338.field_25064).orElse(null);
        view.method_71426("AffectedAreaFrom", class_2338.field_25064).ifPresent(from -> view.method_71426("AffectedAreaTo", class_2338.field_25064).ifPresent(to -> {
            affectedArea = class_3341.method_34390(from, to);
        }));
        super.read(view, clientPacket);
    }

    @SuppressWarnings("deprecation")
    public enum BottomlessFluidMode implements Predicate<class_3611> {
        ALLOW_ALL(fluid -> true),
        DENY_ALL(fluid -> false),
        ALLOW_BY_TAG(fluid -> fluid.method_15791(AllFluidTags.BOTTOMLESS_ALLOW)),
        DENY_BY_TAG(fluid -> !fluid.method_15791(AllFluidTags.BOTTOMLESS_DENY));

        private final Predicate<class_3611> predicate;

        BottomlessFluidMode(Predicate<class_3611> predicate) {
            this.predicate = predicate;
        }

        @Override
        public boolean test(class_3611 fluid) {
            return predicate.test(fluid);
        }
    }

}
