package com.zurrtum.create.content.processing.burner;

import com.zurrtum.create.*;
import com.zurrtum.create.catnip.animation.LerpedFloat;
import com.zurrtum.create.catnip.data.Iterate;
import com.zurrtum.create.catnip.math.AngleHelper;
import com.zurrtum.create.catnip.math.VecHelper;
import com.zurrtum.create.content.fluids.tank.FluidTankBlock;
import com.zurrtum.create.content.logistics.stockTicker.StockTickerBlockEntity;
import com.zurrtum.create.content.processing.basin.BasinBlock;
import com.zurrtum.create.content.processing.burner.BlazeBurnerBlock.HeatLevel;
import com.zurrtum.create.foundation.blockEntity.SmartBlockEntity;
import com.zurrtum.create.foundation.blockEntity.behaviour.BlockEntityBehaviour;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1799;
import net.minecraft.class_1936;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_5819;

public class BlazeBurnerBlockEntity extends SmartBlockEntity {

    public static final int MAX_HEAT_CAPACITY = 10000;
    public static final int INSERTION_THRESHOLD = 500;

    public LerpedFloat headAnimation;
    public boolean stockKeeper;
    public boolean isCreative;
    public boolean goggles;
    public boolean hat;

    protected FuelType activeFuel;
    protected int remainingBurnTime;
    public LerpedFloat headAngle;


    public BlazeBurnerBlockEntity(class_2338 pos, class_2680 state) {
        super(AllBlockEntityTypes.HEATER, pos, state);
        activeFuel = FuelType.NONE;
        remainingBurnTime = 0;
        headAnimation = LerpedFloat.linear();
        headAngle = LerpedFloat.angular();
        isCreative = false;
        goggles = false;
        stockKeeper = false;

        headAngle.startWithValue((AngleHelper.horizontalAngle(state.method_61767(BlazeBurnerBlock.field_11177, class_2350.field_11035)) + 180) % 360);
    }

    public FuelType getActiveFuel() {
        return activeFuel;
    }

    public int getRemainingBurnTime() {
        return remainingBurnTime;
    }

    public boolean isCreative() {
        return isCreative;
    }

    @Override
    public void tick() {
        super.tick();

        if (field_11863.field_9236) {
            AllClientHandle.INSTANCE.tickBlazeBurnerAnimation(this);
            if (!isVirtual())
                spawnParticles(getHeatLevelFromBlock(), 1);
            return;
        }

        if (isCreative)
            return;

        if (remainingBurnTime > 0)
            remainingBurnTime--;

        if (activeFuel == FuelType.NORMAL)
            updateBlockState();
        if (remainingBurnTime > 0)
            return;

        if (activeFuel == FuelType.SPECIAL) {
            activeFuel = FuelType.NORMAL;
            remainingBurnTime = MAX_HEAT_CAPACITY / 2;
        } else
            activeFuel = FuelType.NONE;

        updateBlockState();
    }

    @Override
    public void lazyTick() {
        super.lazyTick();
        stockKeeper = getStockTicker(field_11863, field_11867) != null;
    }

    @Nullable
    public static StockTickerBlockEntity getStockTicker(class_1936 level, class_2338 pos) {
        for (class_2350 direction : Iterate.horizontalDirections) {
            if (level instanceof class_1937 l && !l.method_8477(pos))
                return null;
            class_2680 blockState = level.method_8320(pos.method_10093(direction));
            if (!blockState.method_27852(AllBlocks.STOCK_TICKER))
                continue;
            if (level.method_8321(pos.method_10093(direction)) instanceof StockTickerBlockEntity stbe)
                return stbe;
        }
        return null;
    }

    @Override
    public void addBehaviours(List<BlockEntityBehaviour<?>> behaviours) {
    }

    @Override
    public void write(class_11372 view, boolean clientPacket) {
        if (!isCreative) {
            view.method_71465("fuelLevel", activeFuel.ordinal());
            view.method_71465("burnTimeRemaining", remainingBurnTime);
        } else
            view.method_71472("isCreative", true);
        if (goggles)
            view.method_71472("Goggles", true);
        if (hat)
            view.method_71472("TrainHat", true);
        super.write(view, clientPacket);
    }

    @Override
    protected void read(class_11368 view, boolean clientPacket) {
        activeFuel = FuelType.values()[view.method_71424("fuelLevel", 0)];
        remainingBurnTime = view.method_71424("burnTimeRemaining", 0);
        isCreative = view.method_71433("isCreative", false);
        goggles = view.method_71433("Goggles", false);
        hat = view.method_71433("TrainHat", false);
        super.read(view, clientPacket);
    }

    public BlazeBurnerBlock.HeatLevel getHeatLevelFromBlock() {
        return BlazeBurnerBlock.getHeatLevelOf(method_11010());
    }

    public BlazeBurnerBlock.HeatLevel getHeatLevelForRender() {
        HeatLevel heatLevel = getHeatLevelFromBlock();
        if (!heatLevel.isAtLeast(HeatLevel.FADING) && stockKeeper)
            return HeatLevel.FADING;
        return heatLevel;
    }

    public void updateBlockState() {
        setBlockHeat(getHeatLevel());
    }

    protected void setBlockHeat(HeatLevel heat) {
        HeatLevel inBlockState = getHeatLevelFromBlock();
        if (inBlockState == heat)
            return;
        field_11863.method_8501(field_11867, method_11010().method_11657(BlazeBurnerBlock.HEAT_LEVEL, heat));
        notifyUpdate();
    }

    /**
     * @return true if the heater updated its burn time and an item should be
     * consumed
     */
    protected boolean tryUpdateFuel(class_1799 itemStack, boolean forceOverflow, boolean simulate) {
        if (isCreative)
            return false;

        FuelType newFuel = FuelType.NONE;
        int newBurnTime;

        if (itemStack.method_41409().method_40220(AllItemTags.BLAZE_BURNER_FUEL_SPECIAL)) {
            newBurnTime = 3200;
            newFuel = FuelType.SPECIAL;
        } else {
            newBurnTime = field_11863.method_61269().method_61755(itemStack);
            if (newBurnTime > 0) {
                newFuel = FuelType.NORMAL;
            } else if (itemStack.method_41409().method_40220(AllItemTags.BLAZE_BURNER_FUEL_REGULAR)) {
                newBurnTime = 1600; // Same as coal
                newFuel = FuelType.NORMAL;
            }
        }

        if (newFuel == FuelType.NONE)
            return false;
        if (newFuel.ordinal() < activeFuel.ordinal())
            return false;

        if (newFuel == activeFuel) {
            if (remainingBurnTime <= INSERTION_THRESHOLD) {
                newBurnTime += remainingBurnTime;
            } else if (forceOverflow && newFuel == FuelType.NORMAL) {
                if (remainingBurnTime < MAX_HEAT_CAPACITY) {
                    newBurnTime = Math.min(remainingBurnTime + newBurnTime, MAX_HEAT_CAPACITY);
                } else {
                    newBurnTime = remainingBurnTime;
                }
            } else {
                return false;
            }
        }

        if (simulate)
            return true;

        activeFuel = newFuel;
        remainingBurnTime = newBurnTime;

        if (field_11863.field_9236) {
            spawnParticleBurst(activeFuel == FuelType.SPECIAL);
            return true;
        }

        HeatLevel prev = getHeatLevelFromBlock();
        playSound();
        updateBlockState();

        if (prev != getHeatLevelFromBlock())
            field_11863.method_8396(
                null,
                field_11867,
                class_3417.field_14991,
                class_3419.field_15245,
                .125f + field_11863.field_9229.method_43057() * .125f,
                1.15f - field_11863.field_9229.method_43057() * .25f
            );

        return true;
    }

    protected void applyCreativeFuel() {
        activeFuel = FuelType.NONE;
        remainingBurnTime = 0;
        isCreative = true;

        HeatLevel next = getHeatLevelFromBlock().nextActiveLevel();

        if (field_11863.field_9236) {
            spawnParticleBurst(next.isAtLeast(HeatLevel.SEETHING));
            return;
        }

        playSound();
        if (next == HeatLevel.FADING)
            next = next.nextActiveLevel();
        setBlockHeat(next);
    }

    public boolean isCreativeFuel(class_1799 stack) {
        return stack.method_31574(AllItems.CREATIVE_BLAZE_CAKE);
    }

    public boolean isValidBlockAbove() {
        if (isVirtual())
            return false;
        class_2680 blockState = field_11863.method_8320(field_11867.method_10084());
        return BasinBlock.isBasin(field_11863, field_11867.method_10084()) || blockState.method_26204() instanceof FluidTankBlock;
    }

    protected void playSound() {
        field_11863.method_8396(
            null,
            field_11867,
            class_3417.field_14970,
            class_3419.field_15245,
            .125f + field_11863.field_9229.method_43057() * .125f,
            .75f - field_11863.field_9229.method_43057() * .25f
        );
    }

    protected HeatLevel getHeatLevel() {
        HeatLevel level = HeatLevel.SMOULDERING;
        switch (activeFuel) {
            case SPECIAL -> level = HeatLevel.SEETHING;
            case NORMAL -> {
                boolean lowPercent = (double) remainingBurnTime / MAX_HEAT_CAPACITY < 0.0125;
                level = lowPercent ? HeatLevel.FADING : HeatLevel.KINDLED;
            }
        }
        return level;
    }

    protected void spawnParticles(HeatLevel heatLevel, double burstMult) {
        if (field_11863 == null)
            return;
        if (heatLevel == BlazeBurnerBlock.HeatLevel.NONE)
            return;

        class_5819 r = field_11863.method_8409();

        class_243 c = VecHelper.getCenterOf(field_11867);
        class_243 v = c.method_1019(VecHelper.offsetRandomly(class_243.field_1353, r, .125f).method_18805(1, 0, 1));

        if (r.method_43048(4) != 0)
            return;

        boolean empty = field_11863.method_8320(field_11867.method_10084()).method_26220(field_11863, field_11867.method_10084()).method_1110();

        if (empty || r.method_43048(8) == 0)
            field_11863.method_8406(class_2398.field_11237, v.field_1352, v.field_1351, v.field_1350, 0, 0, 0);

        double yMotion = empty ? .0625f : r.method_43058() * .0125f;
        class_243 v2 = c.method_1019(VecHelper.offsetRandomly(class_243.field_1353, r, .5f).method_18805(1, .25f, 1).method_1029()
            .method_1021((empty ? .25f : .5) + r.method_43058() * .125f)).method_1031(0, .5, 0);

        if (heatLevel.isAtLeast(HeatLevel.SEETHING)) {
            field_11863.method_8406(class_2398.field_22246, v2.field_1352, v2.field_1351, v2.field_1350, 0, yMotion, 0);
        } else if (heatLevel.isAtLeast(HeatLevel.FADING)) {
            field_11863.method_8406(class_2398.field_11240, v2.field_1352, v2.field_1351, v2.field_1350, 0, yMotion, 0);
        }
    }

    public void spawnParticleBurst(boolean soulFlame) {
        class_243 c = VecHelper.getCenterOf(field_11867);
        class_5819 r = field_11863.field_9229;
        for (int i = 0; i < 20; i++) {
            class_243 offset = VecHelper.offsetRandomly(class_243.field_1353, r, .5f).method_18805(1, .25f, 1).method_1029();
            class_243 v = c.method_1019(offset.method_1021(.5 + r.method_43058() * .125f)).method_1031(0, .125, 0);
            class_243 m = offset.method_1021(1 / 32f);

            field_11863.method_8406(soulFlame ? class_2398.field_22246 : class_2398.field_11240, v.field_1352, v.field_1351, v.field_1350, m.field_1352, m.field_1351, m.field_1350);
        }
    }

    public enum FuelType {
        NONE,
        NORMAL,
        SPECIAL
    }

}
